import React, { Fragment, useState } from 'react';
import { I18n } from '@atlassian/wrm-react-i18n';
import TextField from '@atlaskit/textfield';
import { ErrorMessage, Field } from '@atlaskit/form';
import { Box } from '@atlaskit/primitives';
import { isValidHttpUrl } from './createIncomingLinkDialogUtils';
import { CreateIncomingLinkDialogState } from './CreateIncomingLinkDialog';

interface RedirectUriFieldProps {
    substate: CreateIncomingLinkDialogState;
}

export function RedirectUriField(props: RedirectUriFieldProps) {
    const { substate } = props;

    const { incomingApplicationLinkDetails, isPostLoading, enteredRedirectUriValue } = substate;
    const { readonlyRedirectUri, incomingRedirectUri } = incomingApplicationLinkDetails || {};

    const [fieldValue, setFieldValue] = useState<string>(incomingRedirectUri ?? '');
    const [isFieldFocused, setIsFieldFocused] = useState<boolean>(false);
    const [fieldHasError, setFieldHasError] = useState<boolean>(false);

    const validate = (value: string | undefined) => {
        setFieldValue(value ?? '');

        if (!isValidHttpUrl(value)) {
            return 'INVALID';
        }
    };

    const handleFocus = () => {
        setIsFieldFocused(true);
    };

    const handleKeyDown = () => {
        if (fieldValue === '' || !isValidHttpUrl(fieldValue)) {
            setFieldHasError(true);
        } else {
            setFieldHasError(false);
        }
    };

    const handleBlur = () => {
        if (fieldValue === '' || !isValidHttpUrl(fieldValue)) {
            setFieldHasError(true);
        } else {
            setFieldHasError(false);
        }
    };

    const defaultValue = isPostLoading ? enteredRedirectUriValue : incomingRedirectUri || '';

    return (
        <Field
            aria-required={true}
            name="redirectUri"
            label={I18n.getText('applinks-atlassian-oauth2-ui.redirect-uri')}
            defaultValue={defaultValue}
            isDisabled={readonlyRedirectUri}
            validate={validate}
        >
            {({ fieldProps }) => (
                <Fragment>
                    <TextField
                        {...fieldProps}
                        autoFocus={true}
                        onFocus={handleFocus}
                        onKeyDown={handleKeyDown}
                        onBlur={handleBlur}
                        testId="redirect-uri-incoming"
                    />
                    {fieldHasError && (
                        <ErrorMessage>
                            <Box
                                testId="redirect-uri-error"
                                aria-live="polite"
                                {...(isFieldFocused
                                    ? {}
                                    : { 'aria-relevant': 'all', 'aria-atomic': 'false' })}
                            >
                                {I18n.getText('applinks-atlassian-oauth2-ui.must-be-valid-uri')}
                            </Box>
                        </ErrorMessage>
                    )}
                </Fragment>
            )}
        </Field>
    );
}
