import React, { useState, Fragment } from 'react';
import { css } from '@compiled/react';
import { ErrorMessage, Field, FormSection } from '@atlaskit/form';

import {
    PERMISSIONS_DESCRIPTION,
    PERMISSIONS_ERROR_MESSAGE,
    PERMISSIONS_LABEL,
    PERMISSIONS_PLACEHOLDER,
} from './PermissionsSelect.i18n';
import Select from '@atlaskit/select';
import DropdownPermissionDescriptions from './DropdownPermissionDescriptions';
import { token } from '@atlaskit/tokens';

const styles = css({
    fontSize: '12px',
    color: token('color.text.subtle'),
});
const FieldDescription: React.FC<React.PropsWithChildren> = ({ children }) => (
    <p css={styles}>{children}</p>
);

interface SelectOption {
    value: string;
    label: string;
}

interface DropdownPermissionsProps {
    scope: string | null;
    scopeToSelectOption: (scope: string) => SelectOption;
    getPermissionRank: (permission: string) => number;
    selectOptions: SelectOption[];
    handleInheritedScopes: (scopes: string[]) => string[];
}

export function DropdownPermissions({
    scope,
    scopeToSelectOption,
    getPermissionRank,
    selectOptions,
    handleInheritedScopes,
}: DropdownPermissionsProps) {
    const defaultValue = scope ? scopeToSelectOption(scope) : null;
    const [selectedScope, setSelectedScope] = useState<string | null>(
        defaultValue ? defaultValue.value : null
    );

    const validateHasAtLeastOneSelection = (selected: SelectOption | null | undefined) => {
        return selected == null ? ' ' : undefined;
    };

    return (
        <FormSection>
            <Field
                name="scope"
                label={PERMISSIONS_LABEL}
                defaultValue={defaultValue}
                isRequired={true}
                validate={validateHasAtLeastOneSelection}
            >
                {({ fieldProps: { id, ...rest }, error }) => (
                    <Fragment>
                        <Select
                            placeholder={PERMISSIONS_PLACEHOLDER}
                            inputId={id}
                            {...rest}
                            isInvalid={!!error}
                            onChange={(val) => {
                                if (val !== null) {
                                    setSelectedScope(val.value);
                                    rest.onChange(val);
                                }
                            }}
                            menuPortalTarget={document.body}
                            styles={{ menuPortal: (base) => ({ ...base, zIndex: 9999 }) }}
                            options={selectOptions}
                        />
                        <FieldDescription>{PERMISSIONS_DESCRIPTION}</FieldDescription>
                        <DropdownPermissionDescriptions
                            scope={selectedScope}
                            getPermissionRank={getPermissionRank}
                            handleInheritedScopes={handleInheritedScopes}
                        />
                        {error && <ErrorMessage>{PERMISSIONS_ERROR_MESSAGE}</ErrorMessage>}
                    </Fragment>
                )}
            </Field>
        </FormSection>
    );
}
