import React from 'react';
import { css } from '@compiled/react';
import { CheckboxField, Fieldset } from '@atlaskit/form';
import { Checkbox } from '@atlaskit/checkbox';
import Tooltip from '@atlaskit/tooltip';

import {
    allBitbucketProjectPermissions,
    allBitbucketRepoPermissions,
    BitbucketPermission,
} from '../../permissions';
import {
    ACCOUNT_PERMISSION_LABEL,
    ADMIN_LABEL,
    ADMIN_PERMISSION_LABEL,
    PROJECTS_PERMISSION_LABEL,
    READ_LABEL,
    REPOSITORIES_PERMISSION_LABEL,
    SYSTEM_ADMIN_LABEL,
    TOOLTIP_MESSAGE,
    WRITE_LABEL,
} from './PermissionOptions.i18n';

const styles = css({
    marginTop: '19px',
    display: 'flex',
    justifyContent: 'space-between',
});
const Container: React.FC<React.PropsWithChildren> = ({ children }) => (
    <div css={styles}>{children}</div>
);

const labels = [READ_LABEL, WRITE_LABEL, ADMIN_LABEL];

interface PermissionOptionsProps {
    handleChange: (event: React.ChangeEvent<HTMLInputElement>) => void;
    checkedPermissions: Record<string, boolean>;
    disabledPermissions: Record<string, boolean>;
}

export default function PermissionOptions({
    handleChange,
    checkedPermissions,
    disabledPermissions,
}: PermissionOptionsProps) {
    return (
        <Container>
            <Fieldset legend={SYSTEM_ADMIN_LABEL}>
                <CheckboxField name={BitbucketPermission.SYSTEM_ADMIN} value="true">
                    {({ fieldProps }) => (
                        <Checkbox
                            {...fieldProps}
                            alt={BitbucketPermission.SYSTEM_ADMIN}
                            isChecked={checkedPermissions[BitbucketPermission.SYSTEM_ADMIN]}
                            isDisabled={disabledPermissions[BitbucketPermission.SYSTEM_ADMIN]}
                            onChange={(val) => {
                                handleChange(val);
                                fieldProps.onChange(val);
                            }}
                            value={BitbucketPermission.SYSTEM_ADMIN}
                            label={WRITE_LABEL}
                        />
                    )}
                </CheckboxField>
            </Fieldset>
            <Fieldset legend={ADMIN_PERMISSION_LABEL}>
                <CheckboxField name={BitbucketPermission.ADMIN_WRITE} value="true">
                    {({ fieldProps }) => (
                        <Tooltip
                            content={
                                disabledPermissions[BitbucketPermission.ADMIN_WRITE]
                                    ? TOOLTIP_MESSAGE
                                    : null
                            }
                        >
                            <Checkbox
                                {...fieldProps}
                                alt={BitbucketPermission.ADMIN_WRITE}
                                isChecked={checkedPermissions[BitbucketPermission.ADMIN_WRITE]}
                                isDisabled={disabledPermissions[BitbucketPermission.ADMIN_WRITE]}
                                onChange={(val) => {
                                    handleChange(val);
                                    fieldProps.onChange(val);
                                }}
                                value={BitbucketPermission.ADMIN_WRITE}
                                label={WRITE_LABEL}
                            />
                        </Tooltip>
                    )}
                </CheckboxField>
            </Fieldset>
            <Fieldset legend={PROJECTS_PERMISSION_LABEL}>
                {allBitbucketProjectPermissions.map((permission) => (
                    <CheckboxField key={permission} name={permission} value="true">
                        {({ fieldProps }) => (
                            <Tooltip
                                content={disabledPermissions[permission] ? TOOLTIP_MESSAGE : null}
                            >
                                <Checkbox
                                    {...fieldProps}
                                    alt={permission}
                                    isChecked={checkedPermissions[permission]}
                                    isDisabled={disabledPermissions[permission]}
                                    onChange={(val) => {
                                        handleChange(val);
                                        fieldProps.onChange(val);
                                    }}
                                    value={permission}
                                    label={ADMIN_LABEL}
                                />
                            </Tooltip>
                        )}
                    </CheckboxField>
                ))}
            </Fieldset>
            <Fieldset legend={REPOSITORIES_PERMISSION_LABEL}>
                {allBitbucketRepoPermissions.map((permission, index) => (
                    <CheckboxField key={permission} name={permission} value="true">
                        {({ fieldProps }) => (
                            <Tooltip
                                content={disabledPermissions[permission] ? TOOLTIP_MESSAGE : null}
                            >
                                <Checkbox
                                    {...fieldProps}
                                    alt={permission}
                                    isChecked={checkedPermissions[permission]}
                                    isDisabled={disabledPermissions[permission]}
                                    onChange={(val) => {
                                        handleChange(val);
                                        fieldProps.onChange(val);
                                    }}
                                    value={permission}
                                    label={labels[index]}
                                />
                            </Tooltip>
                        )}
                    </CheckboxField>
                ))}
            </Fieldset>
            <Fieldset legend={ACCOUNT_PERMISSION_LABEL}>
                <CheckboxField name={BitbucketPermission.ACCOUNT_WRITE} value="true">
                    {({ fieldProps }) => (
                        <Tooltip
                            content={
                                disabledPermissions[BitbucketPermission.ACCOUNT_WRITE]
                                    ? TOOLTIP_MESSAGE
                                    : null
                            }
                        >
                            <Checkbox
                                {...fieldProps}
                                alt={BitbucketPermission.ACCOUNT_WRITE}
                                isChecked={checkedPermissions[BitbucketPermission.ACCOUNT_WRITE]}
                                isDisabled={disabledPermissions[BitbucketPermission.ACCOUNT_WRITE]}
                                value={BitbucketPermission.ACCOUNT_WRITE}
                                onChange={(val) => {
                                    handleChange(val);
                                    fieldProps.onChange(val);
                                }}
                                label={WRITE_LABEL}
                            />
                        </Tooltip>
                    )}
                </CheckboxField>
            </Fieldset>
        </Container>
    );
}
