import React, { Fragment, useEffect, useState } from 'react';
import { AtlassianOauth2UiDialog } from '../AtlassianOauth2UiDialog';
import { I18n } from '@atlassian/wrm-react-i18n';

import SectionMessage from '@atlaskit/section-message';
import { getOutgoingLink, upsertOutgoingLink } from './OutgoingLinkRest';
import Spinner from '@atlaskit/spinner';
import { RedirectUriField } from './form/RedirectUriField';
import { ClientSecretField } from './form/ClientSecretField';
import { ClientIdField } from './form/ClientIdField';
import { OutgoingScopeField } from './form/OutgoingScopeField';
import { dispatchOutgoingApplinkCreatedEvent } from '../atlassianOauth2UiEvents';

interface CreateOutgoingLinkDialogProps {
    applinkId: string | null;
    onClose: () => void;
}

export function CreateOutgoingLinkDialog({ applinkId, onClose }: CreateOutgoingLinkDialogProps) {
    const [submitLoading, setSubmitLoading] = useState<boolean>(false);
    const [fetchLoading, setFetchLoading] = useState<boolean>(true);
    const [initialData, setInitialData] = useState<any>({});
    const [hasSaveFailed, setHasSaveFailed] = useState<boolean>(false);

    useEffect(() => {
        if (applinkId) {
            getOutgoingLink(applinkId)
                .then((data: any) => {
                    setInitialData(data);
                })
                .finally(() => {
                    setFetchLoading(false);
                });
        }
    }, [applinkId]);

    function prepareUpsertData(formState: any) {
        const data = {
            outgoingClientId: formState.clientId,
            outgoingClientSecret: formState.clientSecret,
            outgoingScopes: initialData.readonlyOutgoingScopes
                ? initialData.availableOutgoingScopes
                : formState.outgoingScopes.map((selected: any) => selected.value),
        };
        return data;
    }

    function handleSubmit(formState: any) {
        const upsertData = prepareUpsertData(formState);
        setSubmitLoading(true);

        upsertOutgoingLink(applinkId, upsertData)
            .then(() => {
                dispatchOutgoingApplinkCreatedEvent(applinkId);
                onClose();
            })
            .catch(() => setHasSaveFailed(true))
            .finally(() => setSubmitLoading(false));
    }

    function closeModal() {
        onClose();
    }

    function getOutgoingLinkDialogTitle() {
        if (!fetchLoading) {
            if (initialData.outgoingPresent) {
                return I18n.getText('applinks-atlassian-oauth2-ui.edit-outgoing-link');
            }
            return I18n.getText('applinks-atlassian-oauth2-ui.create-outgoing-link');
        }

        return '';
    }

    return (
        <AtlassianOauth2UiDialog
            id="AtlassianOauth2Ui_CreateOutgoingLinkDialog"
            onClose={closeModal}
            onSubmit={handleSubmit}
            titleText={getOutgoingLinkDialogTitle()}
            primaryButtonText={I18n.getText('applinks-atlassian-oauth2-ui.dialog.save')}
            primaryButtonIsLoading={submitLoading}
            closeButtonText={I18n.getText('applinks-atlassian-oauth2-ui.dialog.cancel')}
        >
            {fetchLoading ? (
                <p>
                    <Spinner />
                    {I18n.getText('applinks-atlassian-oauth2-ui.loading-link-data')}
                </p>
            ) : (
                <Fragment>
                    {hasSaveFailed && (
                        <SectionMessage
                            testId="default-outgoing-applink-error"
                            appearance="error"
                            title={I18n.getText(
                                'applinks-atlassian-oauth2-ui.create-outgoing-link.failure.title'
                            )}
                        >
                            <p>
                                {I18n.getText(
                                    'applinks-atlassian-oauth2-ui.create-outgoing-link.failure.description'
                                )}
                            </p>
                        </SectionMessage>
                    )}
                    <ClientIdField defaultValue={initialData.outgoingClientId} />
                    <ClientSecretField defaultValue={initialData.outgoingClientSecret} />
                    {!initialData.readonlyOutgoingScopes && (
                        <OutgoingScopeField
                            options={initialData.availableOutgoingScopes}
                            defaultValue={initialData.outgoingScopes}
                            isReadOnly={initialData.readonlyOutgoingScopes}
                        />
                    )}
                    <RedirectUriField value={initialData.outgoingRedirectUri} />
                </Fragment>
            )}
        </AtlassianOauth2UiDialog>
    );
}
