import { ErrorMessage, Field } from '@atlaskit/form';
import { I18n } from '@atlassian/wrm-react-i18n';
import TextField from '@atlaskit/textfield';
import React, { Fragment, useState } from 'react';
import { Box } from '@atlaskit/primitives';

interface ClientSecretFieldProps {
    defaultValue?: string;
}

export function ClientSecretField({ defaultValue }: ClientSecretFieldProps) {
    const [fieldValue, setFieldValue] = useState<string>(defaultValue || '');
    const [isFieldFocused, setIsFieldFocused] = useState<boolean>(false);
    const [fieldHasError, setFieldHasError] = useState<boolean>(false);

    const handleFocus = () => {
        setIsFieldFocused(false);
    };

    const handleBlur = () => {
        setIsFieldFocused(false);
        if (fieldValue === '') {
            setFieldHasError(true);
        } else {
            setFieldHasError(false);
        }
    };

    const validate = (value: string | undefined) => {
        setFieldValue(value ?? '');
        if (value === '') {
            return 'INVALID';
        }
    };

    return (
        <Field
            aria-required={true}
            name="clientSecret"
            label={I18n.getText('applinks-atlassian-oauth2-ui.client-secret')}
            isRequired={true}
            defaultValue={defaultValue || ''}
            validate={validate}
        >
            {({ fieldProps }) => (
                <Fragment>
                    <TextField
                        {...fieldProps}
                        type={'password'}
                        autoComplete={'off'}
                        autoFocus={true}
                        onFocus={handleFocus}
                        onBlur={handleBlur}
                        testId="clientSecret-outgoing"
                    />
                    {fieldHasError && (
                        <ErrorMessage>
                            <Box
                                testId="clientSecret-error"
                                aria-live="polite"
                                {...(isFieldFocused
                                    ? {}
                                    : { 'aria-relevant': 'all', 'aria-atomic': 'false' })}
                            >
                                {I18n.getText(
                                    'applinks-atlassian-oauth2-ui.error.settings.field.required'
                                )}
                            </Box>
                        </ErrorMessage>
                    )}
                </Fragment>
            )}
        </Field>
    );
}
