import React, { useState, useRef } from 'react';
import { I18n } from '@atlassian/wrm-react-i18n';
import { Field } from '@atlaskit/form';
import TextField from '@atlaskit/textfield';
import CopyIcon from '@atlaskit/icon/glyph/copy';
import { IconButton } from '@atlaskit/button/new';
import Tooltip from '@atlaskit/tooltip';

function copyTextFromInput(inputRef: React.RefObject<HTMLInputElement>) {
    if (!inputRef.current) return;
    inputRef.current.select();
    if (navigator && navigator.clipboard) {
        navigator.clipboard
            .writeText(inputRef.current.value)
            .then()
            .catch((error) => {
                console.error('Could not copy text: ', error);
            });
        return;
    }
    if ((window as any).clipboardData) {
        (window as any).clipboardData.setData('Text', inputRef.current.value);
    } else {
        document.execCommand('copy');
    }
}

interface CopyButtonProps {
    inputRef: React.RefObject<HTMLInputElement>;
}

function CopyButton(props: CopyButtonProps) {
    const { inputRef } = props;

    const [isCopied, setIsCopied] = useState(false);

    const handleClick = () => {
        copyTextFromInput(inputRef);
        setIsCopied(true);
    };

    const label = isCopied
        ? I18n.getText('applinks-atlassian-oauth2-ui.copied')
        : I18n.getText('applinks-atlassian-oauth2-ui.copy');

    return (
        <Tooltip
            content={label}
            key={isCopied ? 'copied' : 'default'}
            delay={isCopied ? 0 : undefined}
        >
            <IconButton
                icon={CopyIcon}
                onClick={handleClick}
                label={label}
                isTooltipDisabled={true}
                onMouseLeave={() => {
                    setIsCopied(false);
                }}
            />
        </Tooltip>
    );
}

interface CopyableFieldProps {
    value: string;
    name: string;
    label: string;
    type?: string;
}

export function CopyableField(props: CopyableFieldProps) {
    const { value, name, label, type } = props;

    const inputRef = useRef<HTMLInputElement>(null);

    return (
        <Field name={name} label={label} defaultValue={value}>
            {({ fieldProps }) => (
                <TextField
                    isReadOnly
                    autoComplete="off"
                    type={type || 'text'}
                    {...fieldProps}
                    elemAfterInput={<CopyButton inputRef={inputRef} />}
                    ref={inputRef}
                />
            )}
        </Field>
    );
}
