import React from 'react';
import { AtlassianOauth2UiDialog } from '../AtlassianOauth2UiDialog';
import { I18n } from '@atlassian/wrm-react-i18n';
import { postRevokeCredentials } from './RevokeCredentialsRest';
import { StyledParagraph } from '../common/StyledParagraph';
import { useFlags } from '@atlaskit/flag';
import SuccessIcon from '@atlaskit/icon/glyph/check-circle';
import ErrorIcon from '@atlaskit/icon/glyph/error';
import { token } from '@atlaskit/tokens';

export interface RevokeCredentialsDialogSubstate {
    isOpen: boolean;
    needToLoadDetails?: boolean;
    isRevoking?: boolean;
    hasRevokeError?: boolean;
    hasRevokeSuccess?: boolean;
    rotatedClientId?: string | null;
    clientEntityId?: string | null;
}

export interface RevokeCredentialsDialogProps {
    substate: RevokeCredentialsDialogSubstate;
    dispatch: React.Dispatch<any>;
    onClose: () => void;
}

export function RevokeCredentialsDialog(props: RevokeCredentialsDialogProps) {
    const { substate, dispatch } = props;
    const { showFlag } = useFlags();

    const { isRevoking, rotatedClientId, clientEntityId } = substate;

    function showSuccessFlag() {
        showFlag({
            id: 'revoke-credentials-success-flag',
            title: I18n.getText('applinks-atlassian-oauth2-ui.revoke.success'),
            icon: (
                <SuccessIcon
                    primaryColor={token('color.icon.success')}
                    label={I18n.getText('applinks-atlassian-oauth2-ui.revoke.success')}
                />
            ),
        });
    }

    function showErrorFlag() {
        showFlag({
            id: 'revoke-credentials-error-flag',
            title: I18n.getText('applinks-atlassian-oauth2-ui.revoke.error'),
            icon: (
                <ErrorIcon
                    primaryColor={token('color.icon.danger')}
                    label={I18n.getText('applinks-atlassian-oauth2-ui.revoke.error')}
                />
            ),
            actions: [
                {
                    content: I18n.getText('applinks-atlassian-oauth2-ui.try-again'),
                    onClick: () => {
                        handleSubmit();
                    },
                },
            ],
        });
    }

    async function handleSubmit() {
        dispatch({
            type: 'rcd:revokeStart',
        });

        try {
            await postRevokeCredentials(clientEntityId);
            dispatch({
                type: 'rcd:revokeSuccess',
            });
            showSuccessFlag();
        } catch (error) {
            dispatch({
                type: 'rcd:revokeError',
                error,
            });
            showErrorFlag();
        } finally {
            props.onClose();
        }
    }

    function closeModal() {
        props.onClose();
    }

    return (
        <AtlassianOauth2UiDialog
            id="AtlassianOauth2Ui_RevokeCredentialsDialog"
            onClose={closeModal}
            onSubmit={handleSubmit}
            titleText={I18n.getText('applinks-atlassian-oauth2-ui.revoke.dialog.title')}
            primaryButtonText={I18n.getText('applinks-atlassian-oauth2-ui.revoke.button')}
            primaryButtonIsLoading={isRevoking}
            primaryButtonAppearance="warning"
            closeButtonText={I18n.getText('applinks-atlassian-oauth2-ui.dialog.close')}
        >
            <StyledParagraph>
                {I18n.getText(
                    'applinks-atlassian-oauth2-ui.revoke.description.1',
                    rotatedClientId ?? ''
                )}
            </StyledParagraph>
            <StyledParagraph>
                {I18n.getText('applinks-atlassian-oauth2-ui.revoke.description.2')}
            </StyledParagraph>
        </AtlassianOauth2UiDialog>
    );
}
