import { I18n } from '@atlassian/wrm-react-i18n';
import React, { Component } from 'react';
import SectionMessage, { SectionMessageAction } from '@atlaskit/section-message';
import { ModalFooter, ModalBody, ModalTitle, ModalHeader } from '@atlaskit/modal-dialog';
import { ButtonGroup } from '@atlaskit/button';
import Button from '@atlaskit/button/new';
import { canDeleteEntityLinkResponse, deleteEntityLink, PermissionCode } from './EntitylinksRest';
import Form, { CheckboxField } from '@atlaskit/form';
import { Checkbox } from '@atlaskit/checkbox';
import { authorize } from './auth-button';
import Spinner from '@atlaskit/spinner';
import Modal from '@atlaskit/modal-dialog';

type EntityLink = {
    key: string;
    applicationId: string;
    name: string;
    typeId: string;
    isPrimary?: boolean;
};

interface DeleteEntityLinkDialogFormProps {
    entityLink: EntityLink;
    type: string;
    projectKey: string;
    currentApp: string;
    outgoingApplicationName: string;
    onClose: () => void;
    onDelete: (entityLink: EntityLink) => void;
}

interface DeleteEntityLinkDialogFormState {
    authStatus: string;
    twoWayAllowed?: boolean;
    authUrl?: string;
}

const AuthStatus = {
    UNKNOWN: 'unknown',
    SUCCESSFULL: 'successfull',
    REQUIRED: 'required',
};

export default class DeleteEntityLinkDialogForm extends Component<
    DeleteEntityLinkDialogFormProps,
    DeleteEntityLinkDialogFormState
> {
    constructor(props: DeleteEntityLinkDialogFormProps) {
        super(props);
        this.state = { authStatus: AuthStatus.UNKNOWN };
    }

    componentDidMount() {
        this.checkTwoWayDelete();
    }

    checkTwoWayDelete() {
        canDeleteEntityLinkResponse(this.props.entityLink, this.props.type, this.props.projectKey)
            .then((response) => response.json())
            .then((data) => {
                const permissionCode = data.code;
                if (
                    PermissionCode.CREDENTIALS_REQUIRED === permissionCode ||
                    PermissionCode.AUTHENTICATION_FAILED === permissionCode
                ) {
                    this.setState({
                        authUrl: data.url,
                        authStatus: AuthStatus.REQUIRED,
                    });
                } else {
                    this.setState({
                        twoWayAllowed: permissionCode === 'ALLOWED',
                        authStatus: AuthStatus.SUCCESSFULL,
                    });
                }
            });
    }

    getFooterButtons() {
        return () => (
            <ModalFooter>
                <span />
                <ButtonGroup>
                    <Button appearance="danger" type="submit" testId="delete-entitylink">
                        {I18n.getText('entitylinks.dialog.delete')}
                    </Button>
                    <Button appearance="subtle" onClick={this.props.onClose}>
                        {I18n.getText('entitylinks.dialog.cancel')}
                    </Button>
                </ButtonGroup>
            </ModalFooter>
        );
    }

    onDeleteSubmit(data: any) {
        const deleteReciprocal = data.twowayselect || false;
        this.deleteLink(deleteReciprocal);
    }

    deleteLink(twoWay: boolean) {
        const { entityLink, type, projectKey } = this.props;
        deleteEntityLink(entityLink, type, projectKey, twoWay).then((response) => {
            if (response.status === 200) {
                this.props.onDelete(entityLink);
            }
        });
    }

    getBody() {
        const { twoWayAllowed, authStatus } = this.state;
        switch (authStatus) {
            case AuthStatus.SUCCESSFULL:
                return (
                    <CheckboxField name="twowayselect" defaultIsChecked={twoWayAllowed}>
                        {({ fieldProps }) => (
                            <Checkbox
                                {...fieldProps}
                                isDisabled={!twoWayAllowed}
                                label={I18n.getText(
                                    'entitylinks.delete.dialog.checkbox.twoway.label',
                                    this.props.currentApp,
                                    this.props.outgoingApplicationName
                                )}
                            />
                        )}
                    </CheckboxField>
                );
            case AuthStatus.REQUIRED:
                return (
                    <div className="info-auth-container">
                        <SectionMessage
                            appearance="information"
                            actions={[
                                <SectionMessageAction
                                    key="authorize-link"
                                    testId="authorize-link"
                                    onClick={() =>
                                        authorize(this.state.authUrl, () =>
                                            this.checkTwoWayDelete()
                                        )
                                    }
                                >
                                    {I18n.getText(
                                        'entitylinks.dialog.authorize',
                                        this.props.currentApp
                                    )}
                                </SectionMessageAction>,
                            ]}
                        >
                            <p>
                                {I18n.getText(
                                    'entitylinks.delete.dialog.auth.message.summary',
                                    this.props.currentApp,
                                    this.props.outgoingApplicationName
                                )}
                            </p>
                            <p>
                                {I18n.getText(
                                    'entitylinks.delete.dialog.auth.message.main',
                                    this.props.currentApp,
                                    this.props.outgoingApplicationName
                                )}
                            </p>
                        </SectionMessage>
                    </div>
                );
            default:
                return (
                    <div className="info-auth-container">
                        <SectionMessage appearance="information">
                            <p>
                                {I18n.getText('entitylinks.delete.dialog.auth.message.waiting')}{' '}
                                <Spinner size="small" />
                            </p>
                        </SectionMessage>
                    </div>
                );
        }
    }

    render() {
        const { entityLink } = this.props;
        return (
            <Modal
                autoFocus={true}
                onClose={this.props.onClose}
                testId={'delete-entity-link-dialog-testId'}
            >
                {(({ children }) => (
                    <Form onSubmit={(data) => this.onDeleteSubmit(data)}>
                        {({ formProps }) => <form {...formProps}>{children}</form>}
                    </Form>
                ))({
                    children: (
                        <React.Fragment>
                            <ModalHeader>
                                <ModalTitle>
                                    {I18n.getText('entitylinks.delete.dialog.title')}
                                </ModalTitle>
                            </ModalHeader>

                            <ModalBody>
                                <p>
                                    {I18n.getText(
                                        'entitylinks.delete.dialog.entity.link.text',
                                        entityLink.name,
                                        this.props.outgoingApplicationName
                                    )}
                                </p>
                                {this.getBody()}
                            </ModalBody>
                            {this.getFooterButtons()()}
                        </React.Fragment>
                    ),
                })}
            </Modal>
        );
    }
}
