import React, { Component } from 'react';
import Lozenge from '@atlaskit/lozenge';
import DynamicTable from '@atlaskit/dynamic-table';
import { I18n } from '@atlassian/wrm-react-i18n';

import { getApplicationTypeName } from 'applinks/common/i18n';
import './entity-links';
import Tooltip from '@atlaskit/tooltip';
import EntityLinksMoreActions from './entitylinks-table-actions';

type ApplicationLink = {
    id: string;
    name: string;
    iconUri: string;
    type: string;
    displayUrl?: string;
};

type EntityLink = {
    key: string;
    applicationId: string;
    name: string;
    typeId: string;
    isPrimary?: boolean;
    iconUri?: string;
};

interface EntityLinkTableProps {
    currentApp: string;
    projectKey: string;
    type: string;
    entityLinks: EntityLink[];
    applicationLinks: ApplicationLink[];
    onDelete: (entityLink: EntityLink) => void;
    onMakePrimary: (entityLink: EntityLink) => void;
}

export default class EntityLinkTable extends Component<EntityLinkTableProps> {
    constructor(props: EntityLinkTableProps) {
        super(props);
    }

    static projectLinksApplicationName(
        entityLink: EntityLink,
        serverName: string,
        outgoingAppName: string
    ) {
        return (
            <div className="application-name-alignment">
                <img src={entityLink.iconUri} width={25} height={25} />
                <span className="application-name-margin">
                    {serverName}
                    <span className="application-name-secondary">{`(${outgoingAppName})`}</span>
                </span>
            </div>
        );
    }

    static projectLinksName(projectName: string, isProjectPrimary?: boolean) {
        return (
            <div className="name-container">
                <span className={'entity-link-name'}>{projectName}</span>
                {isProjectPrimary && (
                    <Tooltip content={I18n.getText('entitylinks.application.primary.tooltip')}>
                        <Lozenge>{I18n.getText('entitylinks.application.primary')}</Lozenge>
                    </Tooltip>
                )}
            </div>
        );
    }

    static createHead() {
        return {
            cells: [
                {
                    key: 'application',
                    content: I18n.getText('entitylinks.application.name.header'),
                    isSortable: true,
                },
                {
                    key: 'name',
                    content: I18n.getText('entitylinks.project.name.header'),
                    isSortable: true,
                },
                {
                    key: 'key',
                    content: I18n.getText('entitylinks.project.key.header'),
                    isSortable: true,
                },
                {
                    key: 'actions',
                },
            ],
        };
    }

    createRows(entityLinks: EntityLink[], type: string, applinks: ApplicationLink[]) {
        return entityLinks.map((entityLink, index) => {
            const applink = applinks.find(
                (applink) => applink.id === entityLink.applicationId
            ) as ApplicationLink;
            const outgoingApplicationName = getApplicationTypeName(applink.type);
            return {
                key: `row-${index}-${entityLink.applicationId}-${entityLink.key}`,
                cells: [
                    {
                        key: `${entityLink.applicationId}-${entityLink.key}`,
                        content: EntityLinkTable.projectLinksApplicationName(
                            entityLink,
                            applink.name,
                            outgoingApplicationName
                        ),
                    },
                    {
                        key: `${entityLink.applicationId}-${entityLink.key}`,
                        content: EntityLinkTable.projectLinksName(
                            entityLink.name,
                            entityLink.isPrimary
                        ),
                    },
                    {
                        key: `${entityLink.applicationId}-${entityLink.key}`,
                        content: entityLink.key,
                    },
                    {
                        key: `${entityLink.applicationId}-${entityLink.key}`,
                        content: (
                            <EntityLinksMoreActions
                                currentApp={this.props.currentApp}
                                entityLink={entityLink}
                                type={type}
                                onDelete={(entityLink) => this.props.onDelete(entityLink)}
                                onMakePrimary={(entityLink) => this.props.onMakePrimary(entityLink)}
                                outgoingApplicationName={outgoingApplicationName}
                                projectKey={this.props.projectKey}
                            />
                        ),
                    },
                ],
            };
        });
    }

    render() {
        const { type, entityLinks, applicationLinks } = this.props;

        return (
            <DynamicTable
                head={EntityLinkTable.createHead()}
                rows={this.createRows(entityLinks, type, applicationLinks)}
            />
        );
    }
}
