/**
 * Sets up the dialog with the options to copy a page with its children
 * including any relevant options.
 */
define('confluence/page-hierarchy/dialog/copy-hierarchy-dialog', [
    'ajs',
    'confluence/legacy',
    'confluence/page-hierarchy/action/copy-page-hierarchy-action',
    'confluence/page-hierarchy/service/dialog-service',
    'confluence/page-hierarchy/service/page-tree',
    'confluence/page-hierarchy/state/copy-state',
    'confluence/page-hierarchy/util/analytics-event'
], function (AJS,
             Confluence,
             copyPageHierarchyAction,
             DialogService,
             PageTree,
             state,
             analyticsEvent) {
    var NAME = 'copy-page-hierarchy-dialog';
    var SUBMIT_BUTTON_SELECTOR = '#copy-page-hierarchy-submit';
    var BACK_BUTTON_SELECTOR = '#copy-page-hierarchy-back';
    var CLOSE_BUTTON_SELECTOR = '#copy-page-hierarchy-close';
    var PREVIEW_SELECTOR = '#preview';
    var PREVIEW_MAIN_SELECTOR = '.preview_main';
    var PREVIEW_PARENT_PAGE_SELECTOR = '#parentPage';
    var PREVIEW_CHILDREN_SELECTOR = '#children';
    var COPY_ATTACHMENTS_SELECTOR = '#copy-attachments-2';
    var COPY_PERMISSIONS_SELECTOR = '#copy-permissions';
    var COPY_LABELS_SELECTOR = '#copy-labels';
    var TITLE_PREFIX_SELECTOR = '#title-prefix';
    var SEARCH_SELECTOR = '#search-string';
    var REPLACE_SELECTOR = '#replace-string';
    var TEMPLATE_NAME = 'copyHierarchyDialog';
    var TASK_ID_KEY = 'taskId';
    var MAX_ERROR_ARGS = 3;
    var DEFAULT_PAGE_COPY_LIMIT = 2000;
    var TITLE_CHARACTER_LIMIT = 255;
    var $ = AJS.$;

    var dialog = DialogService.get(NAME, {
        templateName: TEMPLATE_NAME,
        templateParameters: {
            currentPageTitle: AJS.Meta.get('page-title'),
            parentPageTitle: function () {
                return state.getState().destinationPageTitle;
            }
        },
        onShow: _show,
        onInit: _init
    });

    function _init() {
        _bindEvents();
        _setupPreview();
        _bindButtons();

        AJS.dialog2('#' + NAME).on('hide', function () {
            _removeError();
        });
    }

    function _show() {
        _writeValues();
        _updatePreviewDestinationPage();
        _enableActionButtons();
    }

    /**
     * Sets up the preview with the parent page and all the children.
     * @returns {undefined}
     * @private
     */
    function _setupPreview() {
        var $preview = dialog.$element.find(PREVIEW_SELECTOR);
        var $childrenElement = $preview.find(PREVIEW_CHILDREN_SELECTOR);
        var pageId = AJS.Meta.get('page-id');

        var pageTree = new PageTree({
            loadingIndicator: dialog.loadingIndicator,
            onExpand: _updateTitles
        });
        pageTree.load(pageId, $childrenElement);
    }

    /**
     * Updates the parent page title in the preview, based on what was selected in the 'copy-dialog'
     * @returns {undefined}
     * @private
     */
    function _updatePreviewDestinationPage() {
        var $preview = dialog.$element.find(PREVIEW_SELECTOR);
        var $pageLink = $preview.find(PREVIEW_PARENT_PAGE_SELECTOR);
        var destinationPageName = state.getState().destinationPageTitle;

        $pageLink.find('span').html(AJS.escapeEntities(destinationPageName));
    }

    /**
     * Updates the page titles in preview with the currently selected prefix
     * @param {jQuery} $container jQuery parent container to update the titles in
     * @returns {undefined}
     * @private
     */
    function _updateTitles($container) {
        $container.find('.title:visible').each(function (key, element) {
            var $element = $(element);
            var escapedTitle = AJS.escapeEntities($element.data('title'));
            $element.html(_replaceTitle(escapedTitle, true));
        });
    }

    /**
     * Replaces the original title with the added prefix and search and replace strings
     * @param {string} escapedOriginalTitle !important It must be escaped!
     * @param {boolean} format true if the formatting span should be applied, false if no formatting should be used
     * @returns {string} The replacement including the prefix and search/replace
     * @private
     */
    function _replaceTitle(escapedOriginalTitle, format) {
        var prefix = AJS.escapeEntities(state.getOptions().titleOptions.prefix);
        var search = AJS.escapeEntities(state.getOptions().titleOptions.search);
        var replace = AJS.escapeEntities(state.getOptions().titleOptions.replace);

        // Escape Regex in the search string
        var searchRegex = search.replace(/[-[\]{}()*+?.,\\^$|#\s]/g, "\\$&");

        var searchReplace =
            search
                ? escapedOriginalTitle.replace(new RegExp(searchRegex, 'gi'), (format ? _replaced(replace) : replace))
                : escapedOriginalTitle;

        prefix = (prefix ? (format ? _replaced(prefix) : prefix) : '');

        return prefix + searchReplace;
    }

    /**
     * Replaces titles by adding the currently selected prefix and search/replace
     * @param {Array} pageTitles Regular page titles.
     * @param {boolean} format {@see _replaceTitle}
     * @returns {Array} List of new page titles with search/replace and prefix applied
     * @private
     */
    function _replaceTitles(pageTitles, format) {
        return _unique(pageTitles.map(function (title) {
            return _replaceTitle(AJS.escapeEntities(title), format);
        }));
    }

    /**
     * Returns only unique elements from an array
     * @param {Array} array An Array
     * @returns {Array} The Array with no duplicates
     * @private
     */
    function _unique(array) {
        var map = {};
        return array.filter(function (item) {
            return !map[item] && (map[item] = 1);
        });
    }

    /**
     * Returns the text wrapped in span.replaced for styling
     * @param {string} escapedText !important it must be escaped
     * @returns {string} The HTML span.replaced + escapedText
     * @private
     */
    function _replaced(escapedText) {
        return '<span class="replaced">' + escapedText + '</span>';
    }

    /**
     * Bind events to the control buttons for this dialog
     * @returns {undefined}
     * @private
     */
    function _bindButtons() {
        var $copyPageHierarchySubmit = dialog.$element.find(SUBMIT_BUTTON_SELECTOR);
        var $copyPageHierarchyBack = dialog.$element.find(BACK_BUTTON_SELECTOR);
        var $copyPageHierarchyClose = dialog.$element.find(CLOSE_BUTTON_SELECTOR);

        $copyPageHierarchyBack.click(function () {
            AJS.trigger(state.COPY_HIERARCHY_BACK_EVENT);
        });

        $copyPageHierarchySubmit.click(function () {
            _submit().done(_success).fail(_fail);
        });

        $copyPageHierarchyClose.click(function (e) {
            e.preventDefault();
            dialog.hide();
        });
    }

    /**
     * Bind text and checkbox fields to the data model.
     * @param {boolean} writeValues If true, it will write the existing values from state to the fields.
     *                              If false, it will bind the events to the fields, instead.
     * @returns {undefined}
     * @private
     */
    function _bindFields(writeValues) {
        var $copyAttachments = dialog.$element.find(COPY_ATTACHMENTS_SELECTOR);
        var $copyPermissions = dialog.$element.find(COPY_PERMISSIONS_SELECTOR);
        var $copyLabels = dialog.$element.find(COPY_LABELS_SELECTOR);
        var $titlePrefix = dialog.$element.find(TITLE_PREFIX_SELECTOR);
        var $searchString = dialog.$element.find(SEARCH_SELECTOR);
        var $replaceString = dialog.$element.find(REPLACE_SELECTOR);
        var $preview = dialog.$element.find(PREVIEW_SELECTOR);
        var $previewMain = $preview.find(PREVIEW_MAIN_SELECTOR);
        var currentValues = state.getOptions();

        if (writeValues) {

            _maybeRestrictCheckbox(
                state.getPermissions().attachments,
                $copyAttachments,
                currentValues.copyAttachments,
                state.setCopyAttachments);
            _maybeRestrictCheckbox(
                state.getPermissions().restrictions,
                $copyPermissions,
                currentValues.copyPermissions,
                state.setCopyPermissions);

            $copyLabels.prop('checked', currentValues.copyLabels);
            $titlePrefix.val(currentValues.titleOptions.prefix);
            $searchString.val(currentValues.titleOptions.search);
            $replaceString.val(currentValues.titleOptions.replace);
        } else {
            // Booleans
            $copyAttachments.change(function () {
                state.setCopyAttachments($copyAttachments.prop('checked'));
            });
            $copyPermissions.change(function () {
                state.setCopyPermissions($copyPermissions.prop('checked'));
            });
            $copyLabels.change(function () {
                state.setCopyLabels($copyLabels.prop('checked'));
            });

            // Text
            $titlePrefix.keyup(function () {
                state.setTitlePrefix($titlePrefix.val());
                _updateTitles($previewMain);
            });
            $searchString.keyup(function () {
                var searchVal = $searchString.val();
                state.setSearchString(searchVal);
                if (searchVal && !$replaceString.data('changed')) {
                    state.setReplaceString(searchVal);
                    $replaceString.val(searchVal);
                }
                _updateTitles($previewMain);
            });
            $replaceString.keyup(function () {
                state.setReplaceString($replaceString.val());
                $replaceString.data('changed', true);
                _updateTitles($previewMain);
            });
        }
    }

    /**
     * Restrict the checkbox if the user doesn't have permission.
     * @param {boolean} permission     The value of the permission for this checkbox. If true the checkbox
     *                                 will be enabled, otherwise it will be disabled.
     * @param {jQuery} $element        jQuery checkbox element
     * @param {boolean} value          Current value in state
     * @param {function} setStateValue Function to update the current state value if it is disabled
     * @returns {undefined}
     * @private
     */
    function _maybeRestrictCheckbox(permission, $element, value, setStateValue) {
        if (permission) {
            $element.prop('checked', value);
            $element.prop('disabled', false);
        } else {
            setStateValue(false);
            $element.prop('checked', false);
            $element.prop('disabled', true);
        }
    }

    /**
     * Writes the initial values for fields
     * @returns {undefined}
     * @private
     */
    function _writeValues() {
        _bindFields(true);
    }

    /**
     * Binds events to the input fields to keep everything in sync. Run once.
     * @returns {undefined}
     * @private
     */
    function _bindEvents() {
        _bindFields(false);
    }

    /**
     * Submit the request to copy page hierarchy.
     * @returns {object} The promise object returned by the copyPageHierarchyAction.
     * @private
     */
    function _submit() {
        dialog.loadingIndicator.loading();
        _removeError();
        _disableActionButtons();
        state.clearErrors();
        return copyPageHierarchyAction(state.getOptions());
    }

    /**
     * Success callback runs when the task is successfully submitted.
     * @param {object} data  Data retrieved from the server. Contains the taskId.
     * @returns {undefined}
     * @private
     */
    function _success(data) {
        var taskId = data[TASK_ID_KEY];
        dialog.loadingIndicator.done();
        state.setTaskId(taskId);
        AJS.trigger(state.COPY_HIERARCHY_SUBMIT_EVENT);
    }

    /**
     * Fail callback runs when the request to submit the task fails.
     * @param {object} errorResponse Error response from the server.
     * @returns {undefined}
     * @private
     */
    function _fail(errorResponse) {
        analyticsEvent.publish(analyticsEvent.COPY_VALIDATION_FAILED_ANALYTICS_EVENT);
        var messages = [];
        dialog.loadingIndicator.done();
        _enableActionButtons();
        AJS.log(errorResponse);

        switch(errorResponse.status) {
            case 401:
                messages.push({body: AJS.I18n.getText("copy.page.hierarchy.dialog.error.unauthorized")});
                break;
            case 400:
            case 403:
                messages = messages.concat(_buildErrors(errorResponse.responseText));
                break;
            default:
                messages.push({body: AJS.I18n.getText("generic.hierarchy.dialog.error")});
                break;

        }

        if (messages.length === 0) {
            messages.push({body: AJS.I18n.getText("generic.hierarchy.dialog.error")});
        }
        for (var i in messages) {
            if (messages.hasOwnProperty(i)) {
                _setError(messages[i].title, messages[i].body);
            }
        }

        dialog.$element.find(PREVIEW_SELECTOR).scrollTop(0);
    }

    /**
     * @returns {undefined}
     * @private
     */
    function _disableActionButtons() {
        var $submitButton = dialog.$element.find(SUBMIT_BUTTON_SELECTOR);
        var $backButton = dialog.$element.find(BACK_BUTTON_SELECTOR);
        $submitButton.prop('disabled', true);
        $backButton.prop('disabled', true);
    }

    /**
     * @returns {undefined}
     * @private
     */
    function _enableActionButtons() {
        var $submitButton = dialog.$element.find(SUBMIT_BUTTON_SELECTOR);
        var $backButton = dialog.$element.find(BACK_BUTTON_SELECTOR);
        $submitButton.prop('disabled', false);
        $backButton.prop('disabled', false);
    }

    /**
     * Remove any errors lurking in the dialog
     * @returns {undefined}
     * @private
     */
    function _removeError() {
        dialog.$element.find("#copy-pages-error").remove();
    }

    /**
     * Get the errors coming back from the validation result and format them for display
     * @param {string} responseText Response text from the server.
     * @returns {Array} The array of errors
     * @private
     */
    function _buildErrors(responseText) {
        var messages = [];
        var response;
        try {
            response = JSON.parse(responseText);
        } catch (e) {
            AJS.log(e);
        }
        if (response && response.data && response.data.errors && response.data.errors.length) {
            for (var index = 0; index < response.data.errors.length; index++) {
                messages.push(_formatErrorFromRest(response.data.errors[index]));
            }
        }

        return messages;
    }

    var validationMessageMapping = {
        'copy.page.hierarchy.dialog.error.title.length': {
            title: AJS.I18n.getText('copy.page.hierarchy.dialog.error.title.length'),
            body: (pageTitles) => '<p>' + AJS.I18n.getText('copy.page.hierarchy.dialog.error.title.length.description', TITLE_CHARACTER_LIMIT) + '</p>'
                + _formatArgsAsList(pageTitles),
            analyticsEvent: analyticsEvent.COPY_VALIDATION_TITLE_LENGTH_ANALYTICS_EVENT,
        },
        'copy.page.hierarchy.dialog.page.title.conflict': {
            title: AJS.I18n.getText('copy.page.hierarchy.dialog.page.title.conflict'),
            body: (pageTitles) => '<p>' + AJS.I18n.getText('copy.page.hierarchy.dialog.page.title.conflict.description') + '</p>'
                + _formatArgsAsList(pageTitles),
            analyticsEvent: analyticsEvent.COPY_VALIDATION_TITLE_CONFLICT_ANALYTICS_EVENT,
        },
        'copy.page.hierarchy.dialog.over.page.limit': {
            body: (pageTitles, limit) => AJS.I18n.getText('copy.page.hierarchy.dialog.over.page.limit', limit || DEFAULT_PAGE_COPY_LIMIT),
            analyticsEvent: analyticsEvent.COPY_VALIDATION_PAGE_LIMIT_ANALYTICS_EVENT,
        },
        'copy.page.hierarchy.dialog.invalid.destination': {
            body: () => AJS.I18n.getText('copy.page.hierarchy.dialog.invalid.destination'),
            analyticsEvent: analyticsEvent.COPY_VALIDATION_INVALID_DESTINATION_ANALYTICS_EVENT,
        },
        'copy.page.hierarchy.dialog.invalid.origin': {
            body: () => AJS.I18n.getText('copy.page.hierarchy.dialog.invalid.origin'),
            analyticsEvent: analyticsEvent.COPY_VALIDATION_INVALID_ORIGIN_ANALYTICS_EVENT,
        },
        'copy.page.hierarchy.validation.noCreatePagePermission': {
            body: () => AJS.I18n.getText('copy.page.hierarchy.validation.noCreatePagePermission'),
            analyticsEvent: analyticsEvent.COPY_VALIDATION_CREATE_PAGE_PERMISSION_ANALYTICS_EVENT,
        },
    };

    /**
     * Figures out what to put in the title and body of the error if the operation fails to validate
     * @param {object} error An individual error from the list coming from the server
     * @returns {{title: string, body: string}} The error for displaying in the aui message element
     * @private
     */
    function _formatErrorFromRest(error) {
        var title;
        var body;
        var pageTitles = error.message.args;
        var hasPageTitles = pageTitles && pageTitles.length;
        var addMoreOption = false;
        var more = 0;
        var limit = error.message.args && error.message.args[0]

        if (hasPageTitles && pageTitles.length > MAX_ERROR_ARGS) {
            more = pageTitles.length - MAX_ERROR_ARGS;
            pageTitles = pageTitles.splice(0, MAX_ERROR_ARGS);
            addMoreOption = true;
        }
        if (hasPageTitles) {
            pageTitles = _replaceTitles(pageTitles, false);
        }
        if (addMoreOption) {
            pageTitles.push(AJS.I18n.getText('copy.page.hierarchy.dialog.errors.pages.more', more));
        }
        var mapping = validationMessageMapping[error.message.key];
        if (mapping) {
            title = mapping.title;
            body = mapping.body(pageTitles, limit);
            analyticsEvent.publish(mapping.analyticsEvent);
        } else {
            body = AJS.I18n.getText('generic.hierarchy.dialog.error');
        }

        return {title: title, body: body};
    }

    /**
     * Formats the arguments array as an HTML ul
     * @param {Array} args Array of arguments to format as a list.
     *                     !important Every element in args needs to be escaped with AJS.escapeEntities
     * @returns {string} The HTML String ul > (li + arg) *
     * @private
     */
    function _formatArgsAsList(args) {
        return (args && args.length || '') && '<ul><li>' + args.join('</li><li>') + '</li></ul>'
    }

    /**
     * Display the error.
     * @param {string} errorTitle   Title
     * @param {string} errorBody    Body
     * @returns {undefined}
     * @private
     */
    function _setError(errorTitle, errorBody) {
        var props = {
            id: "copy-pages-error",
            insert: "prepend",
            closeable: false
        };
        if (errorTitle) {
            props.title = errorTitle;
        }
        if (errorBody) {
            props.body = errorBody;
        }
        AJS.messages.error("#copy-page-hierarchy-dialog .preview", props);
    }

    return dialog;
});
