define('confluence/page-hierarchy/dialog/copy-progress-dialog', [
    'ajs',
    'confluence/legacy',
    'confluence/page-hierarchy/service/dialog-service',
    'confluence/page-hierarchy/service/progress-runner',
    'confluence/page-hierarchy/state/copy-state',
    'confluence/page-hierarchy/util/analytics-event',
    'confluence/page-hierarchy/util/is-error-type'
], function (AJS,
             Confluence,
             DialogService,
             ProgressRunner,
             state,
             analyticsEvent,
             isErrorType) {
    var NAME = 'copy-progress-dialog';
    var SELECTOR = '#page-hierarchy-progress-dialog';
    var VIEW_COPIED_PAGES_SELECTOR = '#view-copied-pages';
    var SEE_ERRORS_SELECTOR = '#see-errors';
    var CHOOSE_NEW_SELECTOR = '#choose-new';
    var TEMPLATE_NAME = 'progressDialog';
    var STATUS_COPYING_PAGES = 'STATUS_COPYING_PAGES';
    var STATUS_COPIED_PAGES = 'STATUS_COPIED_PAGES';
    var STATUS_RELINK_PAGES = 'STATUS_RELINK_PAGES';
    var ERROR_UNKNOWN = 'ERROR_UNKNOWN';
    var ERROR_RELINK = 'ERROR_RELINK';
    var ERROR_COPY_PAGE = 'ERROR_COPY_PAGE';
    var ERROR_LOCK_FAILED = 'ERROR_LOCK_FAILED';
    var WARN_RENAME_PAGE = 'WARN_RENAME_PAGE';

    var $ = AJS.$;

    var dialog = DialogService.get(NAME, {
        selector: SELECTOR,
        templateName: TEMPLATE_NAME,
        templateParameters: {
            type: 'copy'
        },
        onShow: _run,
        showDelay: 600,
        minimumShowDuration: 500
    });

    /**
     * Main method that runs when the dialog is shown.
     * @returns {undefined}
     * @private
     */
    function _run() {
        new ProgressRunner({
            state: state,
            dialog: dialog.$element,
            onStatus: _updateDisplayStatus,
            onProgress: _onProgress,
            onComplete: _triggerFlag,
            defaultMessages: _defaultMessages,
            ERROR_UNKNOWN: ERROR_UNKNOWN
        }).run();
    }

    /**
     * Triggers an AUI flag to be displayed based on the current state of the copy operation
     * @returns {undefined}
     * @private
     */
    // eslint-disable-next-line complexity
    function _triggerFlag() {
        var errors = state.getErrors() || [];
        var warnings = state.getWarnings() || [];
        var anyPagesCopied = state.getState().anyPagesCopied;
        var destinationPageUrl = state.getState().destinationPageUrl;
        var errorRelinkingPages = _isErrorRelinkingPages(errors);
        var lockFailedError = _isLockFailedError(errors);
        var unexpectedError = _isUnexpectedError(errors);
        var hasCopyPageErrors = _hasCopyPageErrors(errors);
        var warningRenamedPages = _isWarningRenamedPages(warnings);
        var successTitle = AJS.I18n.getText('copy.page.hierarchy.dialog.complete.title');
        var errorTitle = AJS.I18n.getText('copy.page.hierarchy.dialog.complete.title.error');
        var somePagesWerentCopiedTitle = AJS.I18n.getText('copy.page.hierarchy.dialog.complete.partialError');
        var type;
        var title;
        var body;

        if (errors.length) {
            if (lockFailedError) {
                type = 'error';
                title = errorTitle;
                body = AJS.I18n.getText('copy.page.hierarchy.dialog.error.space.lock.description');
                analyticsEvent.publish(analyticsEvent.COPY_FLAG_ERROR_LOCK);
            } else {
                if (errorRelinkingPages && !hasCopyPageErrors) {
                    type = 'warning';
                    title = AJS.I18n.getText('copy.page.hierarchy.dialog.complete.pageLinks.title');
                    body = AJS.I18n.getText('copy.page.hierarchy.dialog.complete.pageLinks');
                    analyticsEvent.publish(analyticsEvent.COPY_FLAG_WARNING_RELINKING_PAGES);
                }
                else if (anyPagesCopied) {
                    type = 'warning';
                    title = somePagesWerentCopiedTitle;
                    if (unexpectedError) {
                        body = AJS.I18n.getText('generic.hierarchy.dialog.error');
                        analyticsEvent.publish(analyticsEvent.COPY_FLAG_WARNING_SOME_PAGES_FAILED);
                    } else if (errorRelinkingPages) {
                        body = AJS.I18n.getText('copy.page.hierarchy.dialog.complete.pageLinks.additional');
                        analyticsEvent.publish(analyticsEvent.COPY_FLAG_WARNING_SOME_PAGES_FAILED_RELINKING_PAGES);
                    }
                } else {
                    type = 'error';
                    title = errorTitle;
                    body = AJS.I18n.getText('copy.page.hierarchy.dialog.complete.error');
                    analyticsEvent.publish(analyticsEvent.COPY_FLAG_ERROR_ALL_PAGES);
                }
            }
            // Done copying but maybe not totally finished yet -> Have to review errors
            AJS.trigger(state.COPY_HIERARCHY_DONE_EVENT);
        } else {
            type = 'success';
            title = successTitle;
            // We are ignoring other warnings. Some of them to do with permissions which will never be
            // displayed on the front end because the user won't be allowed to start the process without
            // the correct permissions anyway
            if (warnings.length && warningRenamedPages) {
                body = AJS.I18n.getText('copy.page.hierarchy.dialog.complete.pagesRenamed');
            }

            analyticsEvent.publish(analyticsEvent.COPY_FLAG_SUCCESS);

            // Totally done.
            AJS.trigger(state.FINISHED_EVENT);
        }

        _updateFlagDisplay(AJS.flag({
            type: type,
            close: 'manual',
            title: title,
            body: Confluence.Templates.PageHierarchy.copyFlag(
                {
                    body: body,
                    showErrors: hasCopyPageErrors,
                    anyPagesCopied: !errors || anyPagesCopied,
                    lockFailed: lockFailedError
                })
        }), destinationPageUrl);
    }

    /**
     * Updates the flag events
     * @param {object} flag DOM element for the flag
     * @param {string} destinationPageUrl Copy of the destination page url before the state gets reset
     * @returns {undefined}
     * @private
     */
    function _updateFlagDisplay(flag, destinationPageUrl) {
        var $flag = $(flag);
        // If we're finished the whole process.
        var finished = true;
        $flag.find(VIEW_COPIED_PAGES_SELECTOR).prop('href', destinationPageUrl).click(function () {
            analyticsEvent.publish(analyticsEvent.COPY_VIEW_COPIED_PAGES);
        });
        $flag.find(SEE_ERRORS_SELECTOR).click(function () {
            finished = false;
            flag.close();
            AJS.trigger(state.COPY_HIERARCHY_VIEW_ERROR_EVENT);
        });
        $flag.find(CHOOSE_NEW_SELECTOR).click(function () {
            finished = false;
            flag.close();
            AJS.trigger(state.COPY_HIERARCHY_CHOOSE_NEW_EVENT);
        });
        $flag.on('aui-flag-close', function () {
            if (finished) {
                AJS.trigger(state.FINISHED_EVENT);
            }
            analyticsEvent.publish(analyticsEvent.COPY_CLOSE_FLAG);
        });
    }

    /**
     * Determines if we have a LOCK error.
     * @param {Array} errors    Array of errors we have
     * @returns {boolean} true iff it encounters ERROR_LOCK_FAILED in the array
     * @private
     */
    function _isLockFailedError(errors) {
        return isErrorType(errors, ERROR_LOCK_FAILED);
    }

    /**
     * Determines if an unexpected error occurred based on the errors we have reported.
     * @param {Array} errors    Array of errors we have
     * @returns {boolean} true iff it encounters ERROR_UNKNOWN in the array
     * @private
     */
    function _isUnexpectedError(errors) {
        return isErrorType(errors, ERROR_UNKNOWN);
    }

    /**
     * Determines if we have some copy page errors based on the errors we have reported.
     * @param {Array} errors    Array of errors we have
     * @returns {boolean} true iff it encounters ERROR_COPY_PAGE in the array
     * @private
     */
    function _hasCopyPageErrors(errors) {
        return isErrorType(errors, ERROR_COPY_PAGE);
    }

    /**
     * Determines if there was an error updating the page links based on the errors we have reported.
     * @param {Array} errors    Array of errors we have
     * @returns {boolean} true iff it encounters ERROR_RELINK in the array
     * @private
     */
    function _isErrorRelinkingPages(errors) {
        return isErrorType(errors, ERROR_RELINK);
    }

    /**
     * Determines if there was an error with some of the pages and they had to be automatically renamed.
     * @param {Array} warnings    Array of warnings we have
     * @returns {boolean} true iff it encounters WARN_RENAME_PAGE in the array
     * @private
     */
    function _isWarningRenamedPages(warnings) {
        return isErrorType(warnings, WARN_RENAME_PAGE);
    }

    function _updateDisplayStatus(status) {
        var message = AJS.I18n.getText('copy.page.hierarchy.dialog.progress.copying');
        if (status) {
            switch (status.key) {
                case STATUS_COPIED_PAGES:
                    var numCopied = status.args[0] || 0;
                    var numTotal = status.args[1] || 0;
                    state.setAnyPagesCopied(state.getState().anyPagesCopied || numCopied > 0);
                    message = AJS.I18n.getText('copy.page.hierarchy.dialog.progress.copied', numCopied, numTotal);
                    break;
                case STATUS_RELINK_PAGES:
                    message = AJS.I18n.getText('copy.page.hierarchy.dialog.progress.relinking');
                    break;
                case STATUS_COPYING_PAGES:
                default:
                    message = AJS.I18n.getText('copy.page.hierarchy.dialog.progress.copying');
            }
        }

        return message;
    }

    /**
     * Default message to display if the long running task has not yet updated.
     * @returns {{status: object, errors: Array}} A default response from the server if some kind of error
     *                                              occurred
     * @private
     */
    function _defaultMessages() {
        return {
            status: {
                key: STATUS_COPYING_PAGES
            },
            errors: []
        }
    }

    function _onProgress(messages) {
        if (messages.destinationUrl) {
            state.setDestinationPageUrl(AJS.contextPath() + messages.destinationUrl);
        }
    }

    return dialog;
});
