define('confluence/page-hierarchy/dialog/delete-progress-dialog', [
    'ajs',
    'confluence/legacy',
    'confluence/page-hierarchy/service/delete-navigator',
    'confluence/page-hierarchy/service/dialog-service',
    'confluence/page-hierarchy/service/progress-runner',
    'confluence/page-hierarchy/state/delete-state',
    'confluence/page-hierarchy/util/analytics-event',
    'confluence/page-hierarchy/util/get-error-pages',
    'confluence/page-hierarchy/util/is-error-type'
], function (AJS,
             Confluence,
             deleteNavigator,
             DialogService,
             ProgressRunner,
             state,
             analyticsEvent,
             getErrorPages,
             isErrorType) {
    var NAME = 'delete-progress-dialog';
    var SELECTOR = '#page-hierarchy-progress-dialog';
    var SEE_ERRORS_SELECTOR = '#see-errors';
    var TEMPLATE_NAME = 'progressDialog';
    var STATUS_DELETING_PAGE = 'STATUS_DELETING_PAGE';
    var STATUS_DELETING_PAGES = 'STATUS_DELETING_PAGES';
    var STATUS_DELETED_PAGE = 'STATUS_DELETED_PAGE';
    var STATUS_DELETED_PAGES = 'STATUS_DELETED_PAGES';
    var STATUS_MOVE_PAGES = 'STATUS_MOVE_PAGES';
    var STATUS_MOVING_PAGES = 'STATUS_MOVING_PAGES';
    var ERROR_UNKNOWN = 'ERROR_UNKNOWN';
    var ERROR_DELETE_PAGE = 'ERROR_DELETE_PAGE';

    var $ = AJS.$;

    var dialog = DialogService.get(NAME, {
        selector: SELECTOR,
        templateName: TEMPLATE_NAME,
        templateParameters: {
            type: 'delete'
        },
        onShow: _run,
        showDelay: 600,
        minimumShowDuration: 500
    });

    /**
     * Main method that runs when the dialog is shown.
     * @returns {undefined}
     * @private
     */
    function _run() {
        new ProgressRunner({
            state: state,
            dialog: dialog.$element,
            onStatus: _getDisplayStatus,
            onComplete: _onComplete,
            defaultMessages: _defaultMessages,
            ERROR_UNKNOWN: ERROR_UNKNOWN
        }).run();
    }

    /**
     * Function that runs when the progress is complete. It decides if
     * we should navigate to the parent page or stay on the current page to display
     * the relevant flag.
     * @returns {undefined}
     * @private
     */
    function _onComplete() {
        var result = _determineResult();
        if (result.existingFlow) {
            return _triggerFlag(result);
        }
        var shouldNavigateToParent = result.type === 'success' || result.thisPageDeleted;
        if (shouldNavigateToParent) {
            deleteNavigator.navigateToParent(result.taskId);
        } else {
            _triggerFlag(result);
        }
    }

    /**
     * Determines the result of the delete process. It figures out if it was successful
     * or not and prepares information for the flag, as well as some other information
     * which is useful to determine what to do next.
     * @returns {object} Details of the result
     * @private
     */
    function _determineResult() {
        var errors = state.getErrors() || [];
        var numPagesDeleted = state.getState().numPagesDeleted;
        var taskId = state.getState().taskId;
        var existingFlow = state.getState().existingFlow;
        var unexpectedError = _isUnexpectedError(errors);
        var hasDeletePageErrors = _hasDeletePageErrors(errors);
        var thisPageDeleted = _thisPageDeleted(state);
        var errorTitle = AJS.I18n.getText('delete.page.hierarchy.error.title.all');
        var somePagesWereNotDeletedTitle = AJS.I18n.getText('delete.page.hierarchy.error.title.some');
        var type;
        var title;
        var body;

        if (errors.length) {
            if (numPagesDeleted > 0) {
                type = 'warning';
                if (thisPageDeleted) {
                    title = AJS.I18n.getText('delete.page.hierarchy.error.title.specific',
                        state.getState().originalPageTitle);
                } else {
                    title = somePagesWereNotDeletedTitle;
                }

                if (unexpectedError) {
                    body = AJS.I18n.getText('generic.hierarchy.dialog.error');
                }
                analyticsEvent.publish(analyticsEvent.DELETE_FLAG_WARNING_SOME_PAGES_FAILED);
            } else {
                type = 'error';
                title = errorTitle;
                body = AJS.I18n.getText('delete.page.hierarchy.error.description.all');
                analyticsEvent.publish(analyticsEvent.DELETE_FLAG_ERROR_ALL_PAGES);
            }
            // Done deleting but maybe not totally finished yet -> Have to review errors
            AJS.trigger(state.DELETE_HIERARCHY_DONE_EVENT);
        } else {
            type = 'success';
            title = numPagesDeleted > 1
                ? AJS.I18n.getText('delete.page.hierarchy.flag.success.title.multiple', numPagesDeleted)
                : AJS.I18n.getText('delete.page.hierarchy.flag.success.title.single');
            body = numPagesDeleted > 1
                ? AJS.I18n.getText('delete.page.hierarchy.flag.success.description.multiple')
                : AJS.I18n.getText('delete.page.hierarchy.flag.success.description.single');

            // Totally done.
            AJS.trigger(state.FINISHED_EVENT);
            analyticsEvent.publish(analyticsEvent.DELETE_FLAG_SUCCESS);
        }
        return {
            type: type,
            title: title,
            body: body,
            hasDeletePageErrors: hasDeletePageErrors,
            thisPageDeleted: thisPageDeleted,
            taskId: taskId,
            existingFlow: existingFlow
        }
    }

    /**
     * Triggers an AUI flag to be displayed based on the current state of the copy operation
     * @param {object} result The result of running _determineResult
     * @returns {undefined}
     * @private
     */
    function _triggerFlag(result) {
        _updateFlagDisplay(AJS.flag({
            type: result.type,
            close: 'manual',
            title: result.title,
            body: Confluence.Templates.PageHierarchy.deleteFlag(
                {
                    body: result.body,
                    showErrors: result.hasDeletePageErrors
                })
        }));
    }

    /**
     * Updates the flag events
     * @param {object} flag DOM element for the flag
     * @returns {undefined}
     * @private
     */
    function _updateFlagDisplay(flag) {
        var $flag = $(flag);
        // If we're finished the whole process.
        var finished = true;
        $flag.find(SEE_ERRORS_SELECTOR).click(function () {
            finished = false;
            flag.close();
            AJS.trigger(state.DELETE_HIERARCHY_VIEW_ERROR_EVENT);
        });
        $flag.on('aui-flag-close', function () {
            if (finished) {
                AJS.trigger(state.FINISHED_EVENT);
            }
            analyticsEvent.publish(analyticsEvent.DELETE_CLOSE_FLAG);
        });
    }

    /**
     * Determines if an unexpected error occurred based on the errors we have reported.
     * @param {Array} errors    Array of errors we have
     * @returns {boolean} true iff it encounters ERROR_UNKNOWN in the array
     * @private
     */
    function _isUnexpectedError(errors) {
        return isErrorType(errors, ERROR_UNKNOWN);
    }

    /**
     * Determines if we have some delete page errors based on the errors we have reported.
     * @param {Array} errors    Array of errors we have
     * @returns {boolean} true iff it encounters ERROR_DELETE_PAGE in the array
     * @private
     */
    function _hasDeletePageErrors(errors) {
        return isErrorType(errors, ERROR_DELETE_PAGE);
    }

    /**
     * Checks to see if the current page has been deleted. If it has then
     * we need to land on the parent page regardless of the rest of the results,
     * but if it has not then we stay on the page.
     *
     * It scans through the pages that had errors to see if one of them is the current page.
     * @param {object} state  The current delete state
     * @returns {boolean} true if this page has been deleted, false otherwise.
     * @private
     */
    function _thisPageDeleted(state) {
        if (!state.getErrors().length) {
            return true;
        }
        if (_isUnexpectedError(state.getErrors())) {
            return false;
        }
        if (state.getState().numPagesDeleted === 0) {
            return false;
        }
        var pageTitle = AJS.Meta.get('page-title');
        var pages = getErrorPages(state, ERROR_DELETE_PAGE);

        for (var i = 0; i < pages.length; i++) {
            var page = pages[i];
            if (page.title === pageTitle) {
                return false;
            }
        }

        return true;
    }

    /**
     * Returns the status to display in the progress dialog
     * @param {object} status JSON object returning from the Long Running Task status string.
     * @returns {string} The status to display in the dialog
     * @private
     */
    function _getDisplayStatus(status) {
        var numDeleted;
        var numTotal;
        if (status) {
            switch (status.key) {
                case STATUS_DELETED_PAGE:
                case STATUS_DELETED_PAGES:
                    numDeleted = status.args[0] || 0;
                    state.setNumPagesDeleted(numDeleted);
                    return AJS.I18n.getText('delete.page.hierarchy.progress.finished');
                case STATUS_MOVE_PAGES:
                case STATUS_MOVING_PAGES:
                    return AJS.I18n.getText('delete.page.hierarchy.progress.moving');
                case STATUS_DELETING_PAGE:
                case STATUS_DELETING_PAGES:
                    numDeleted = status.args[0] || 0;
                    numTotal = status.args[1] || 0;
                    state.setNumPagesDeleted(numDeleted);
                    return AJS.I18n.getText('delete.page.hierarchy.progress.description', numDeleted, numTotal);
                default:
                    return AJS.I18n.getText('delete.page.hierarchy.progress.generic.description');
            }
        }
    }

    /**
     * Default message to display if the long running task has not yet updated.
     * @returns {{status: object, errors: Array}} A default response from the server if some kind of error
     *                                              occurred
     * @private
     */
    function _defaultMessages() {
        return {
            status: {
                key: ''
            },
            errors: []
        }
    }

    return dialog;
});