define('confluence/page-hierarchy/flow/delete-flow', [
    'ajs',
    'confluence/page-hierarchy/state/delete-state',
    'confluence/page-hierarchy/dialog/delete-dialog',
    'confluence/page-hierarchy/dialog/delete-hierarchy-dialog',
    'confluence/page-hierarchy/dialog/delete-progress-dialog',
    'confluence/page-hierarchy/dialog/delete-error-dialog',
    'confluence/page-hierarchy/service/data-service',
    'confluence/page-hierarchy/util/analytics-event'
], function (AJS,
             state,
             deleteDialog,
             deleteHierarchyDialog,
             progressDialog,
             errorDialog,
             DataService,
             analyticsEvent) {

    // Delete Dialog
    AJS.bind(state.DELETE_HIERARCHY_EVENT, function () {
        deleteDialog.hide();
        deleteHierarchyDialog.show();
        analyticsEvent.publish(analyticsEvent.DELETE_HIERARCHY_ANALYTICS_EVENT);
    });

    // Delete Hierarchy Dialog
    AJS.bind(state.DELETE_HIERARCHY_SUBMIT_EVENT, function () {
        deleteDialog.hide();
        deleteHierarchyDialog.hide();
        progressDialog.show();
        analyticsEvent.publish(analyticsEvent.DELETE_HIERARCHY_SUBMIT_ANALYTICS_EVENT);
    });

    AJS.bind(state.DELETE_HIERARCHY_BACK_EVENT, function () {
        deleteHierarchyDialog.hide();
        deleteDialog.show();
        analyticsEvent.publish(analyticsEvent.DELETE_HIERARCHY_BACK_ANALYTICS_EVENT);
    });

    AJS.bind(state.DELETE_HIERARCHY_DONE_EVENT, function () {
        progressDialog.hide();
        analyticsEvent.publish(analyticsEvent.DELETE_HIERARCHY_DONE_ANALYTICS_EVENT);
    });

    AJS.bind(state.DELETE_HIERARCHY_VIEW_ERROR_EVENT, function () {
        errorDialog.show();
        analyticsEvent.publish(analyticsEvent.DELETE_HIERARCHY_VIEW_ERROR_ANALYTICS_EVENT)
    });

    // Finished
    AJS.bind(state.FINISHED_EVENT, function () {
        deleteDialog.remove();
        deleteHierarchyDialog.remove();
        progressDialog.remove();
        errorDialog.remove();

        window.location.hash = '';
        state.reset();
        analyticsEvent.publish(analyticsEvent.DELETE_HIERARCHY_FINISHED_ANALYTICS_EVENT);
    });

    AJS.bind(state.DELETE_HIERARCHY_CANCEL_EVENT, function () {
        deleteDialog.remove();
        deleteHierarchyDialog.remove();
        state.reset();
        analyticsEvent.publish(analyticsEvent.DELETE_HIERARCHY_CANCEL_ANALYTICS_EVENT);
    });

    /**
     * Starts an existing flow if one is in progress.
     * @returns {boolean} true if the flow is in progress, false otherwise.
     * @private
     */
    function _startExistingFlow() {
        var hash = window.location.hash;
        if (hash.indexOf('delete-complete') === 1) {
            var split = hash.split('&');
            var taskId = split[1];
            var originalPageTitle = decodeURIComponent(split[2]);
            state.setOriginalPageTitle(originalPageTitle);
            var doFlow = function () {
                state.setTaskId(taskId);
                state.setExistingFlow(true);
                progressDialog.show();
                analyticsEvent.publish(analyticsEvent.DELETE_HIERARCHY_START_EXISTING_ANALYTICS_EVENT);
            };
            if (taskId) {
                doFlow();
            } else {
                // if there is no taskId specified, we might be in a test, so just
                // wait for the test to trigger the flow
                AJS.bind('copy-page-hierarchy:delete-hierarchy-existing-flow-parent-page', doFlow);
            }

            return true;
        }
        return false;
    }

    /**
     * Starts a brand new flow.
     * @returns {undefined}
     * @private
     */
    function _startNewFlow() {
        DataService.reset();
        deleteDialog.show();
        analyticsEvent.publish(analyticsEvent.DELETE_HIERARCHY_START_ANALYTICS_EVENT);
    }

    function startFlow() {
        _startExistingFlow() || _startNewFlow();
    }

    return {
        startFlow: startFlow
    }
});