/**
 * Async resources loader for copy-page-hierarchy resources.
 */
require([
    'ajs',
    'wrm',
    'confluence/legacy'
], function (AJS, WRM, Confluence) {
    var COPY_CONTEXT = 'confluence-copy-page-hierarchy';
    var DELETE_CONTEXT = 'confluence-delete-page-hierarchy';
    var PAGE_HIERARCHY_NAMESPACE = 'confluencePageHierarchy';

    AJS.toInit(function ($) {
        var $toolsCopyLink = $('#action-copy-page-link');
        var $operationsCopyLink = $('#copyPageLink');
        var $toolsDeleteLink = $('#action-remove-content-link');

        _loadResourcesAsync($toolsCopyLink, COPY_CONTEXT);
        _loadResourcesAsync($operationsCopyLink, COPY_CONTEXT);
        _loadResourcesAsync($toolsDeleteLink, DELETE_CONTEXT);
        _checkInProgressDeleteFlow($toolsDeleteLink);
    });

    /**
     * Loads the require resources for either copy or delete
     * @param {jQuery} $button The button that loads the resources
     * @param {string} context The WRM resource context to load
     * @returns {undefined}
     * @private
     */
    function _loadResourcesAsync($button, context) {
        if ($button.length) {
            $button.off('click.' + PAGE_HIERARCHY_NAMESPACE);
            $button.one('click.' + PAGE_HIERARCHY_NAMESPACE, function (e) {
                e.preventDefault();
                _loadResources($button, context);
            });
        }
    }

    /**
     * Function handler to run when the delete or copy button is clicked
     * @param {jQuery} $button The button that loads the resources
     * @param {string} context The WRM resource context to load
     * @returns {Promise|{then:Function}} a then-able object which will be called once resources load
     * @private
     */
    function _loadResources($button, context) {
        var loadingIndicator = Confluence.PageLoadingIndicator($('body'));
        loadingIndicator.show();
        $button.off('click.' + PAGE_HIERARCHY_NAMESPACE);

        AJS.debug('Loading ' + context + ' resources...');
        return WRM.require(['wrc!' + context], function () {
            loadingIndicator.hide();
            $button.addClass(PAGE_HIERARCHY_NAMESPACE);
            AJS.debug('Loaded ' + context + ' resources.');
        });
    }

    /**
     * Checks to see if there is a flow in progress based on url parameters
     * and loads the resources if the flow is in progress.
     * @param {jQuery} $toolsDeleteLink The Delete button, if present
     * @returns {undefined}
     * @private
     */
    function _checkInProgressDeleteFlow($toolsDeleteLink) {
        var hash = window.location.hash;
        if (hash.indexOf('delete-complete') === 1) {
            _loadResources($toolsDeleteLink, DELETE_CONTEXT).then(function() {
                // Webdriver test code expects the `.confluencePageHierarchy` class to indicate
                // all code has lazy-loaded.
                $("#content").addClass(PAGE_HIERARCHY_NAMESPACE);
            });
        }
    }
});
