define('confluence/page-hierarchy/util/validation-error', [
    'ajs',
    'confluence/page-hierarchy/util/analytics-event'
], function (AJS,
             analyticsEvent) {

    var PREVIEW_SELECTOR = '#preview';

    /**
     * Remove any errors lurking in the dialog
     * @param {object} dialog the delete page dialog
     * @returns {undefined}
     * @private
     */
    function removeError(dialog) {
        dialog.$element.find("#validation-error").remove();
    }

    /**
     * Fail callback runs when the request to submit the task fails.
     * @param {object} errorResponse Error response from the server.
     * @param {string} analyticsErrorEvent the type of analytics error
     * @param {object} dialog the delete page dialog
     * @param {object} errorMessages static array of error messages
     * @returns {undefined}
     */
    function fail(errorResponse, analyticsErrorEvent, dialog, errorMessages) {
        analyticsEvent.publish(analyticsErrorEvent);
        var messages = [];
        dialog.loadingIndicator.done();

        switch(errorResponse.status) {
            case 401:
                messages.push({body: errorMessages["unauthorized"]});
                break;
            case 400:
            case 403:
                messages = messages.concat(_buildErrors(errorResponse.responseText, errorMessages));
                break;
            case 404:
                messages.push({body: errorMessages["notfound"]});
                break;
            case 405:
                var responseJson = JSON.parse(errorResponse.responseText);
                if (responseJson.reason === "READ_ONLY") {
                    messages.push({body: errorMessages["readonly"]});
                } else {
                    messages.push({body: errorMessages["description"]});
                }
                break;
            default:
                messages.push({body: errorMessages["description"]});
                break;
        }

        if (messages.length === 0) {
            messages.push({body: errorMessages["description"]});
        }
        for (var i in messages) {
            if (messages.hasOwnProperty(i)) {
                _setError(messages[i].title, messages[i].body, dialog.selector || dialog.$element.selector);
            }
        }

        dialog.$element.find(PREVIEW_SELECTOR).scrollTop(0);
    }

    /**
     * Get the errors coming back from the validation result and format them for display
     * @param {string} responseText Response text from the server.
     * @param {object} errorMessages the error messages from the server
     * @returns {Array} The array of errors
     * @private
     */
    function _buildErrors(responseText, errorMessages) {
        var messages = [];
        var response;
        try {
            response = JSON.parse(responseText);
        } catch (e) {
            AJS.log(e);
        }
        if (response && response.data && response.data.errors && response.data.errors.length) {
            for (var index = 0; index < response.data.errors.length; index++) {
                messages.push(_formatErrorFromRest(response.data.errors[index], errorMessages));
            }
        }

        return messages;
    }

    /**
     * Figures out what to put in the title and body of the error if the operation fails to validate
     * @param {object} error An individual error from the list coming from the server
     * @param {object} errorMessages the error messages
     * @returns {{title: string, body: string}} The error for displaying in the aui message element
     * @private
     */
    function _formatErrorFromRest(error, errorMessages) {
        var title = errorMessages["title"];
        var body;

        switch (error.message.key) {
            case 'delete.page.hierarchy.dialog.invalid.target':
                body = AJS.I18n.getText('delete.page.hierarchy.dialog.invalid.target');
                analyticsEvent.publish(analyticsEvent.DELETE_VALIDATION_INVALID_TARGET_ANALYTICS_EVENT);
                break;
            case 'delete.page.hierarchy.validation.noDeletePagePermission':
                body = AJS.I18n.getText('delete.page.hierarchy.dialog.noDeletePagePermission');
                analyticsEvent.publish(analyticsEvent.DELETE_VALIDATION_NO_DELETE_PAGE_PERMISSION_ANALYTICS_EVENT);
                break;
            case 'delete.page.hierarchy.dialog.invalid.deleteHierarchy':
                body = AJS.I18n.getText('delete.page.hierarchy.dialog.deleteHierarchy');
                analyticsEvent.publish(analyticsEvent.DELETE_VALIDATION_INVALID_DELETE_HIERARCHY_ANALYTICS_EVENT);
                break;
            case 'delete.page.hierarchy.validation.maxPagesExceeded':
                body = AJS.I18n.getText('delete.page.hierarchy.validation.maxPagesExceeded');
                analyticsEvent.publish(analyticsEvent.DELETE_VALIDATION_MAX_PAGES_EXCEEDED_ANALYTICS_EVENT);
                break;
            default:
                body = AJS.I18n.getText('generic.hierarchy.dialog.error');
        }

        return {title: title, body: body};
    }

    /**
     * Display the error.
     * @param {string} errorTitle   Title
     * @param {string} errorBody    Body
     * @param {string} dialogCssSelector The css selector to element inside delete dialog
     * @returns {undefined}
     * @private
     */
    function _setError(errorTitle, errorBody, dialogCssSelector) {
        var props = {
            id: "validation-error",
            insert: "prepend",
            closeable: false
        };
        if (errorTitle) {
            props.title = errorTitle;
        }
        if (errorBody) {
            props.body = errorBody;
        }
        AJS.messages.error(dialogCssSelector + " .validation-error-parent", props);
    }

    return {
        fail: fail,
        removeError: removeError
    };
});
