define('confluence-collaborative-editor-plugin/synchrony-entity', [
    'jquery',
    'ajs',
    'confluence-collaborative-editor-plugin/synchrony-auth',
    'confluence-collaborative-editor-plugin/synchrony-util',
    'confluence-collaborative-editor-plugin/unsupported-extension',
    'confluence-collaborative-editor-plugin/util/is-valid-uri',
], function ($, AJS, Auth, Util, UnsupportedExtension, isValidUri) {
    'use strict';

    function rxTest(rx) {
        return function (value) {
            // deletes are always allowed
            return null === value || rx.test(value);
        };
    }

    /**
     * !! WARNING !!
     *
     * Modifying this whitelist can introduce XSS attack vectors!
     *
     * Do not modify the whitelist unless you are 100% sure you know
     * what you are doing. Never do this
     *
     *   'some-attribute': true,
     *
     * Or this
     *
     *   'some-style': true
     *
     * Use restrictive regexes to whitelist the individual characters
     * that are allowed in the attribute or style instead:
     *
     *   var numbers = function (value) { return null == value || (/^[0-9]$/).test(value); }
     *
     *   'some-attribute': numbers,
     *   'some-style': numbers
     *
     * TODO: change all the true values in this whitelist to a restrictive regex.
     */
    function makeWhitelist(Synchrony) {
        var cssQuiteSafeValueTest = rxTest(/^[\w\.\-\+\%]+$/i);
        var emoticonNameSafeValueTest = rxTest(/^[\w-]+$/);
        var embeddedImgSrcDataURITest = rxTest(/^data:image\/png;base64,([\w+\/%\-]+)=*$/);
        var designTokenColorTest = rxTest(
            /^(var\(--ds-[\w-]+, ?#[a-fA-F\d]{6}\))|(?:#[a-f\d]+|rgb\(\s*[\d\.\-\+]+%?\s*,\s*[\d\.\-\+]+%?\s*,\s*[\d\.\-\+]+%?\s*\)|\w+)$/i
        );
        var synchronyIsValidUrl = Synchrony.whitelists.tinymce.attributes.src;
        return $.extend({}, Synchrony.whitelists.tinymce, {
            styles: $.extend({}, Synchrony.whitelists.tinymce.styles, {
                // Syncing the padding-top on the body element will
                // cause an infinite sync loop (WD-170). Other
                // padding-* styles are blacklisted for
                // completeness.
                'padding-top': false,
                'padding-right': false,
                'padding-bottom': false,
                'padding-left': false,
                padding: false,
                display: cssQuiteSafeValueTest,
                'list-style-type': cssQuiteSafeValueTest,
                color: designTokenColorTest,
            }),
            attributes: $.extend({}, Synchrony.whitelists.tinymce.attributes, {
                // ** Image effects support
                'confluence-query-params': true,
                'data-element-title': true,
                // --end image effects support
                // ** inline comments support
                'data-ref': true,
                // -- end inline comments support
                //Extra HTML attributes CONF uses
                accesskey: true,
                datetime: true,
                'data-anchor': true,
                'data-encoded-xml': true,
                'data-highlight-class': true,
                'data-highlight-colour': true,
                'data-space-key': true,
                'data-username': true,
                // CONFSERVER-52441
                'data-emoticon-name': emoticonNameSafeValueTest,
                'data-emoji-id': true,
                'data-hipchat-emoticon': true,
                // ** space-list macro
                'data-entity-id': true,
                'data-entity-type': true,
                'data-favourites-bound': true,
                // -- end space-list macro
                // ** macro placeholder support
                'data-macro-id': true,
                'data-macro-name': true,
                'data-macro-schema-version': true,
                'data-macro-body-type': true,
                'data-macro-parameters': true,
                'data-macro-default-parameter': true,
                // -- end macro placeholder support
                // ** page layout support
                'data-atlassian-layout': true,
                'data-placeholder-type': true,
                'data-layout': true,
                'data-title': true,
                'data-type': true,
                // -- end page layout support
                // ** inline task support
                'data-inline-task-id': true,
                'data-uuid': true,
                'data-inline-tasks-content-id': true,
                // -- end inline task support
                // ** WD-323
                'data-base-url': true,
                'data-linked-resource-id': true,
                'data-linked-resource-type': true,
                'data-linked-resource-version': true,
                'data-linked-resource-default-alias': true,
                'data-linked-resource-container-version': true,
                'data-linked-resource-content-type': true,
                // -- end WD-323
                'data-unresolved-comment-count': true,
                'data-location': true,
                'data-image-height': true,
                'data-image-width': true,
                'data-attachment-copy': true,
                // -- end WD-323

                // CONFSERVER-100831: attributes expected on copied attachments to enable previewing
                'data-linked-resource-container-id': true,
                'data-mime-type': true,
                'data-has-thumbnail': true,
                'data-file-src': true,
                'data-nice-type': true,

                // CONFSRVDEV-20858: data attr used by tinymce advanced table plugin
                'data-content-title': true,
                'data-snooker-locked-cols': true,
                'data-snooker-col-series': true,
                'data-mce-resize': true,

                //CONFSERVER-51998
                'data-filename': true,
                //fix for CONFSRVDEV-6580
                username: true,
                src: function (attr) {
                    return embeddedImgSrcDataURITest(attr) || synchronyIsValidUrl(attr);
                },
                href: isValidUri,
                role: true,
                tabindex: true,
                'aria-haspopup': true,
                'aria-label': true,
            }),
            classes: function (className) {
                switch (className) {
                    case 'mceSelected':
                    case 'active-resizable':
                    case 'valid':
                    case 'active':
                        return false;
                        break;

                    default:
                        return true;
                }
            },
            elements: $.extend({}, Synchrony.whitelists.tinymce.elements, {
                // ** date lozenge support
                time: true,
                // -- end date lozenge support
                mark: true,
                label: true,
                form: true,
            }),
            elementsByClass: $.extend({}, Synchrony.whitelists.tinymce.elementsByClass, {
                'mce-pastebin': false,
            }),
        });
    }

    function makeBindArgs(Synchrony, editor, htmlDoc) {
        var whitelist = makeWhitelist(Synchrony);
        return {
            profile: 'tinymce',
            selectionCorrections: false,
            telepointer: {
                refreshOnResize: true,
                label: {
                    hover: true,
                    movement: 1000,
                    text: function (session) {
                        try {
                            var name = session.fullname || AJS.I18n.getText('anonymous.name');
                            return name.charAt(0).toUpperCase();
                        } catch (e) {
                            AJS.log(e);
                        }
                        return '&#9786;';
                    },
                },
            },
            tinymce: {
                monkeyPatchUndoManager: true,
                instance: editor,
            },
            whitelist: function (validationProps) {
                var domNode = validationProps.domNode;
                var isBody = domNode === htmlDoc;
                var isElement = validationProps.type === 'node';
                var isDataTitleAttr = validationProps.type === 'attribute' && validationProps.name === 'data-title';

                //For the body, we only want to whitelist the element itself and the data-title attribute
                //The rest should stay as it is (i.e. not sycnhronised or removed)
                if (isBody) {
                    return isElement || isDataTitleAttr;
                }

                // only allow sync contenteditable attribute if it is on element with class numberingColumn
                // numberingColumn is used by table auto numbering feature
                // can't add contenteditable to whitelist because other elements also have it(<div/>, <time/>)
                var isAutoNumberingColumn =
                    domNode &&
                    domNode.classList.contains('numberingColumn') &&
                    (domNode.nodeName === 'TD' || domNode.nodeName === 'TH');
                var isHipchatEmoji = !!domNode && domNode.hasAttribute('data-hipchat-emoticon');

                if (
                    validationProps.type === 'attribute' &&
                    validationProps.name === 'contenteditable' &&
                    (isAutoNumberingColumn || isHipchatEmoji)
                ) {
                    return true;
                }

                // CONFSRVDEV-25030
                // mce-visual-caret from TinyMCE is constantly patching Synchrony on a specific edge case
                // This also causes flaky test from infinite "Saving Changes" status indicator, we need to prevent it
                if (validationProps.name) {
                    if (validationProps.name.includes('mce-visual-caret') && validationProps.type === 'class') {
                        if (domNode) {
                            const { classList, attributes, nodeName } = domNode || {};
                            const isMceVisualCaret =
                                classList.contains('mce-visual-caret') &&
                                attributes['data-mce-bogus'].value === 'all' &&
                                nodeName === 'DIV';
                            if (isMceVisualCaret) return false;
                        }
                    }
                }

                return Synchrony.isWhitelisted(whitelist, validationProps);
            },
            domReadHook: function () {
                AJS.trigger('cursor-target-refresh');
                UnsupportedExtension.check();
            },
        };
    }

    function bind(Synchrony, editor, htmlDoc, history, content) {
        return Synchrony.entity({
            url: Util.getServiceUrl(),
            entityId: Util.getEntityId(),
            jwt: Auth.getTokenPromise,
            initRev: content.confRev,
            history: history,
            presence: true,
            useFallback: Util.getXhrFallbackFlag(),
        }).bind(htmlDoc, makeBindArgs(Synchrony, editor, htmlDoc));
    }

    return {
        bind: bind,
        makeWhitelist: makeWhitelist,
    };
});
