define('jpos-editor', [
    'ajs',
    'aui/dialog2',
    'confluence/templates',
    'confluence/macro-js-overrides',
], function (
    AJS,
    Dialog2,
    Templates,
    MacroJsOverrides
) {
    'use strict';

    return {
        _validateUrl: function(url) {
            var urlPattern = new RegExp(/^https?:\/\/.+\/secure\/PortfolioRoadmapConfluence\.jspa\?r=[a-zA-Z0-9]{0,30}$/);
            return url.match(urlPattern);
        },

        _validateHeight: function(height) {
            return height.length > 0 && parseInt(height, 10) >= 300;
        },

        _renderPreview: function() {
            var self = this;
            var previewContainerHeight = self.previewContainer.offsetHeight;
            var previewUrl = AJS.contextPath() + "/rest/tinymce/1/macro/preview";
            var previewPayload = {
                "contentId": AJS.Meta.get("page-id"),
                "macro": {
                    "name": self.macro.name,
                    "params": {
                        url: self.urlInput.value,
                        planHeight: previewContainerHeight
                    }
                }
            };

            self._disablePreviewBtn.call(self, 'Loading...');
            self._clearPreview.call(self);

            AJS.$.ajax({
                url: previewUrl,
                type: "POST",
                contentType: "application/json",
                data: JSON.stringify(previewPayload)
            }).done(function (data) {

                self.previewContainer.innerHTML = Templates.JPOS.jposMacroIframe({
                    height: self.heightInput.value
                });

                var iframe = document.querySelector('.jpos__iframe');
                var iframeDocument = iframe.contentWindow.document;

                if (iframe) {
                    self.iframeLoadTimeout = setTimeout(self._iframeLoadTimeout.bind(self), 20000);

                    iframe.addEventListener('load', function () {
                        clearTimeout(self.iframeLoadTimeout);
                        self._enablePreviewBtn.call(self);

                        iframeDocument.querySelector('#main').setAttribute('style', 'padding: 0px; overflow: hidden');
                        iframeDocument.querySelector('body').style.overflow = 'hidden';
                    });
                }

                // we don't want to trigger the preview page onload event in our macro browser preview
                data = data.replace('window.onload', 'var preventPreviewOnloadFnTrigger');
                // our macro has a 2px border, but we don't want to see this in our macro browser preview
                data = data.replace('2px solid #efefef', 'none');

                iframeDocument.open();
                iframeDocument.write(data);
                iframeDocument.close();

            });
        },

        _iframeLoadTimeout: function() {
            var self = this;
            self.previewContainer.innerHTML = Templates.JPOS.jposIframeTimeout();
            self._enablePreviewBtn.call(self);
        },

        _clearPreview: function() {
            var self = this;
            self.previewContainer.innerHTML = '';
            clearTimeout(self.iframeLoadTimeout);
        },

        _enablePreviewBtn: function() {
            var self = this;
            self.previewBtn.innerHTML = AJS.I18n.getText('Preview');
            self.previewBtn.removeAttribute('disabled');
        },

        _disablePreviewBtn: function(text) {
            var self = this;
            self.previewBtn.innerHTML = AJS.I18n.getText(text);
            self.previewBtn.setAttribute('disabled', true);
        },

        _addListeners: function() {
            var self = this;
            var heightBtns = document.querySelectorAll('.jpos__size-button');
            var closeBtn = document.querySelector('.jpos__close-dialog');
            var urlError = document.querySelector('.jpos__field-error--url');
            var heightError = document.querySelector('.jpos__field-error--height');


            function updateHeightVal(e) {
                self.heightInput.value = e.target.dataset.value;
                heightError.style.display = "none";

                if(self._validateUrl(self.urlInput.value)) {
                    self.insertBtn.removeAttribute('disabled');
                }
            }

            function debounce(func, delay) {
                var inDebounce;
                return function() {
                    var context = this;
                    var args = arguments;
                    clearTimeout(inDebounce);
                    inDebounce = setTimeout(function() {func.apply(context, args), delay});
                }
            }

            // this function removes white spaces in the URL input field (if any)
            function removeWhiteSpacesFromURL() {
                self.urlInput.value = self.urlInput.value.replace(/(%20|\s)/g, "");
            }

            for(var i = 0; i < heightBtns.length; i++) {
                heightBtns[i].addEventListener('click', updateHeightVal);
            }

            // fired when clicking on the Preview button
            self.previewBtn.addEventListener('click', function () {
                removeWhiteSpacesFromURL();
                self._renderPreview();
            });

            // fired when clicking on the Cancel button
            closeBtn.addEventListener('click', function () {
                self.dialog.hide();
            });

            // fired when clicking on the Insert button
            self.insertBtn.addEventListener('click', function () {
                removeWhiteSpacesFromURL();

                // save the values to the macro
                tinymce.confluence.macrobrowser.macroBrowserComplete({
                    name: self.macro.name,
                    params: {
                        url: self.urlInput.value,
                        planHeight: self.heightInput.value,
                    }
                });
                self.dialog.hide();
            });

            // fired when a key is pressed in the URL input field
            self.urlInput.addEventListener('keydown', function(e) {
                // if the user pressed the space bar, the key is ' '
                if (e.key === ' ') {
                    // preventing the whitespace to be rendered in the input field
                    e.preventDefault();
                }
            });

            // fired when the user provides input e.g. pastes URL in the URL input field
            self.urlInput.addEventListener('input', debounce(function (e) {
                removeWhiteSpacesFromURL();

                self._clearPreview();

                if(!self._validateUrl(this.value)) {
                    urlError.style.display = "block";
                    self._disablePreviewBtn.call(self, 'Preview');
                    self.insertBtn.setAttribute('disabled', true);
                    return false;
                }

                urlError.style.display = "none";
                self._enablePreviewBtn.call(self);

                if (self._validateHeight(self.heightInput.value)) {
                    self.insertBtn.removeAttribute('disabled');
                }
            }, 500));

            self.heightInput.addEventListener('input', debounce(function (e) {
                if(!self._validateHeight(e.target.value)) {
                    heightError.style.display = "block";
                    self.insertBtn.setAttribute('disabled', true);
                    return false;
                }
                heightError.style.display = "none";

                if(self._validateUrl(self.urlInput.value)) {
                    self.insertBtn.removeAttribute('disabled');
                }
            }, 500));
        },

        init: function () {
            var self = this;

            var overrideOpener = function(macro) {
                var templateContainer = document.createElement('div');
                templateContainer.innerHTML = Templates.JPOS.jposMacroDialog();
                document.body.appendChild(templateContainer.firstChild);

                self.macro = macro;
                self.dialog = Dialog2(".jpos__macro-dialog");
                self.dialog.show();

                self.urlInput = document.querySelector('.jpos__url-input');
                self.heightInput = document.querySelector('.jpos__height-input');
                self.insertBtn = document.querySelector('.jpos__insert-macro');
                self.previewBtn = document.querySelector('.jpos__preview-btn');
                self.previewContainer = document.querySelector('.jpos__preview-container');

                self._addListeners();

                // if we have params, generate a preview, otherwise focus the url input and disable the action buttons
                const defaultHeight = '700';
                if(macro.params) {
                    self.urlInput.value = macro.params.url;
                    self.heightInput.value = macro.params.planHeight || defaultHeight;
                    self.previewBtn.focus();
                } else {

                    self.heightInput.value = defaultHeight;
                    self._disablePreviewBtn.call(self, 'Preview');
                    self.insertBtn.setAttribute('disabled', true);
                    self.urlInput.focus();
                }
            };

            MacroJsOverrides.put("portfolio-for-jira-plan", {
                opener: overrideOpener
            });
        }
    };
});

require('confluence/module-exporter')
    .safeRequire('jpos-editor', function (jposMacro) {
        require('ajs').toInit(function () {
            jposMacro.init();
        });
    });