(function () {
    'use strict';

    const contextPath = window.WRM.contextPath;

    const levelMap = {
        debug: 1,
        info: 2,
        log: 3,
        warn: 4,
        error: 5,
    };
    let loggingLevelName = 'log';
    const isLoggable = (level) => {
        const usedLevel = levelMap[loggingLevelName] || levelMap.log;
        return level >= usedLevel;
    };
    const generateLogFn = (levelName, headers) => (...args) => {
        if (isLoggable(levelMap[levelName])) {
            console[levelName](...headers, ...args);
        }
    };
    const createLogger = ({ context = "default" }) => {
        const headers = [`[WRM]`, `[${context}]`];
        return Object.keys(levelMap).reduce((acc, levelName) => {
            return Object.assign(Object.assign({}, acc), { [levelName]: generateLogFn(levelName, headers) });
        }, {});
    };
    const setLoggingLevel = (levelName) => {
        loggingLevelName = levelName;
    };

    const logger$2 = createLogger({ context: "tracker" });
    const serverSideRenderedAttribute = "data-initially-rendered";
    const wrmKeyAttribute = "data-wrm-key";
    const wrmBatchTypeAttribute = "data-wrm-batch-type";
    const batchTypePrefixMap = {
        resource: "wr!",
        context: "wrc!",
    };
    const keyPartIsAValidInclusion = (key) => key.length > 0 &&
        key[0] !== "-"; // Exclusions start with "-"
    class RequestedResourcesTracker {
        constructor(_container) {
            this._container = _container;
            this._initialized = false;
            this._resourceKeys = new Set();
        }
        /**
         * @private
         */
        _getLoaded() {
            return Array.from(this._resourceKeys);
        }
        /**
         * PLUGWEB-399: Inspect the DOM to see if any <script> or <link> tags have already been added, in which case
         * they should not be loaded again.
         *
         * PLUGWEB-402: Only <script> or <link> tags in the DOM with the "data-initially-rendered" attribute are
         * inspected to see which resources have already been loaded. HOWEVER, these attributes are only added if the
         * <script> or <link> tags were rendered on the server-side. These must all have the "data-wrm-key" and
         * "data-wrm-batch-type" attributes for the clientside to know what needs to be excluded.
         *
         * Resources loaded on the clientside do not have this attribute and are instead tracked when the batch download
         * URLs are returned from "/rest/wrm/2.0/resources" and are processed.
         *
         * PLUGWEB-670: An existing assumption is that the script tags on the page DO NOT change neither before nor after
         * this is called. Jira "SPA" transitions _currently_ respect this.
         */
        getResources() {
            if (this._initialized) {
                return this._getLoaded();
            }
            this._container.querySelectorAll(`[${serverSideRenderedAttribute}]`).forEach((el) => {
                var _a, _b;
                if (el.nodeName === "SCRIPT" || el.nodeName === "LINK") {
                    this.addResource((_a = el.getAttribute(wrmKeyAttribute)) !== null && _a !== void 0 ? _a : '', ((_b = el.getAttribute(wrmBatchTypeAttribute)) !== null && _b !== void 0 ? _b : ''));
                }
            });
            this._initialized = true;
            return this._getLoaded();
        }
        /**
         * @param {string} wrmKey This is a bad name for what it really is, which is batch key, it can be a single resource, single
         * context, or a batch of them, potentially with exclusions of what was previously loaded.
         * @param {BatchType} batchType
         */
        addResource(wrmKey, batchType) {
            const lowercasedBatchType = batchType.toLowerCase();
            if (!Object.keys(batchTypePrefixMap).includes(lowercasedBatchType)) {
                const unknownBatchTypeErrorMsg = `Unknown batch type '${batchType}' discovered when adding a resource '${wrmKey}'`;
                logger$2.error(unknownBatchTypeErrorMsg);
                throw new Error(unknownBatchTypeErrorMsg);
            }
            const prefix = batchTypePrefixMap[lowercasedBatchType];
            wrmKey
                .split(",") // split the batch
                // TODO: Fix PLUGWEB-691 clientside inclusions cannot overcome serverside exclusions
                .filter(keyPartIsAValidInclusion)
                .forEach((key) => {
                this._resourceKeys.add(`${prefix}${key}`);
            });
        }
    }

    const logger$1 = createLogger({ context: "queue" });
    const makeQueue = (phase) => {
        const items = [];
        const enqueue = (resourceKeys, onEnqueued) => new Promise((resolve, reject) => {
            logger$1.debug("Resources entering the queue... ", { resourceKeys, phase });
            items.push({
                phase,
                resourceKeys,
                resolve,
                reject,
            });
            onEnqueued();
        });
        const isNotEmpty = () => items.length > 0;
        const drain = () => {
            const queuedElements = [...items];
            items.length = 0;
            return queuedElements;
        };
        return {
            enqueue,
            isNotEmpty,
            drain,
        };
    };
    const settleItemPromises = (variant) => ({ result, queuedElements }) => Promise.all(queuedElements.map((queuedElement) => queuedElement[variant](result)));
    const makeQueueManager = (process) => {
        const queuesByPhase = {
            require: makeQueue("require"),
            interaction: makeQueue("interaction"),
        };
        const drainableQueues = [];
        const drainAllQueues = () => Promise.all(drainableQueues.flatMap((queue) => queue.drain()));
        const areQueuesNonEmpty = () => drainableQueues.some((queue) => queue.isNotEmpty());
        const semaphore = ((draining) => () => {
            if (areQueuesNonEmpty()) {
                logger$1.debug("There are resources waiting in queue: ", queuesByPhase);
            }
            else {
                logger$1.debug("Nothing waiting in queue. Draining won't be scheduled this time.");
                return;
            }
            // prevent draining until any in-process drain is done
            if (draining) {
                logger$1.debug("Draining already scheduled");
                return;
            }
            logger$1.debug("Scheduling next draining");
            draining = Promise.resolve()
                .then(drainAllQueues)
                .then(process)
                .then(settleItemPromises("resolve"), settleItemPromises("reject"))
                .finally(() => {
                logger$1.debug("Resources from drained queues processed");
                draining = undefined;
                semaphore();
            });
        })();
        switch (document.readyState) {
            default:
                // Prefer not loading over double loading -- easier to troubleshoot
                logger$1.error("There's a missing readyState, fix it ASAP! No guarantees the page will ever work nor load");
            case "loading":
                document.addEventListener("DOMContentLoaded", () => {
                    drainableQueues.push(queuesByPhase.interaction);
                    semaphore();
                }, { once: true });
                document.addEventListener("readystatechange", () => {
                    if (document.readyState === 'interactive') {
                        drainableQueues.push(queuesByPhase.require);
                        semaphore();
                    }
                }, { once: true });
                break;
            case "interactive":
                document.addEventListener("DOMContentLoaded", () => {
                    drainableQueues.push(queuesByPhase.interaction);
                    semaphore();
                }, { once: true });
                drainableQueues.push(queuesByPhase.require);
                semaphore();
                break;
            case "complete":
                drainableQueues.push(queuesByPhase.require);
                drainableQueues.push(queuesByPhase.interaction);
                semaphore();
                break;
        }
        // Normally should use the logger, this runs before the logging level can be set to debug or below
        // AND it's for debug level (i.e. normally not seen)
        console.debug(`[WRM] Client-side initialised during '${document.readyState}' state`);
        return {
            enqueue: (resourceKeys, phase) => queuesByPhase[phase].enqueue(resourceKeys, semaphore),
        };
    };

    const resourceTypes = {
        JS: "JS",
        CSS: "CSS",
    };

    const injectIntoDocument = (element) => {
        document.head.appendChild(element);
        return new Promise((resolve, reject) => {
            element.addEventListener("load", resolve);
            element.addEventListener("error", reject);
        });
    };
    const loadStyle = (href) => {
        const linkNode = document.createElement("link");
        linkNode.href = href;
        linkNode.rel = "stylesheet";
        return injectIntoDocument(linkNode);
    };
    const loadScript = (src) => {
        const scriptNode = document.createElement("script");
        scriptNode.src = src;
        scriptNode.async = false; // violates ordering contract of the WRM
        scriptNode.defer = true; // minimum phase is DEFER -- for performance and to prevent racing server included DEFERs
        return injectIntoDocument(scriptNode);
    };
    const load = (key) => {
        const [id, url] = key.split("!");
        if (id === "js") {
            return loadScript(url);
        }
        if (id === "css") {
            return loadStyle(url);
        }
        return Promise.reject(new Error("Unsupported resource identifier"));
    };
    const loadResources = (keys, onSuccess, onFail) => {
        return Promise.all(keys.map(load)).then(onSuccess, onFail);
    };

    /**
     * @typedef {Object} Resource
     * @property {string} url - the location of the resource on the server.
     * @property {string} key - the complete module key for the resource's parent web-resource.
     * @property {('JAVASCRIPT'|'CSS')} resourceType - the kind of asset at the given url.
     * @property {('context'|'resource')} batchType - whether this url represents a context batch or a standalone resource.
     * @property {string} [media] - (for CSS resources) the type of media the resource should render to (e.g., print).
     */
    /**
     * @enum
     */
    const failcases = {
        DISCOVERY: 1,
        LOADER: 2,
        NETWORK: 4,
        EXCEPTION: 8,
    };
    const logger = createLogger({ context: "handler" });
    const noop = () => { };
    // To be sure that it will not override the real curl.
    if (!("curl" in WRM)) {
        WRM.curl = () => {
            logger.error("WRM.curl is deprecated. Use WRM.require instead. See https://bitbucket.org/atlassian/atlassian-plugins-webresource/src/master/UPGRADE_500.md");
        };
    }
    /**
     * Loads CSS and sets media queries as appropriate.
     * @param resource
     * @private
     */
    function loadCss(resource) {
        logger.warn("asynchronously loading a CSS resource containing a media query", resource.url);
        var tag = document.createElement("link");
        tag.setAttribute("rel", "stylesheet");
        tag.setAttribute("type", "text/css");
        tag.setAttribute("href", resource.url);
        tag.setAttribute("media", resource.media);
        document.head.appendChild(tag);
    }
    /**
     * Checks if a script element whose src is the given url exists on the page
     * @param url url
     * @return {boolean} True if the script is on the page, otherwise false
     * @private
     */
    function isJsAlreadyLoaded(url) {
        // TODO PLUGWEB-685 we should be monitoring this
        return Boolean(document.querySelector(`script[src='${url}']`));
    }
    /**
     * Checks if a link element whose href is the given url exists on the page
     * @param url url
     * @return {boolean} True if the link is on the page, otherwise false
     * @private
     */
    function isCSSAlreadyLoaded(url) {
        // TODO PLUGWEB-685 we should be monitoring this
        return Boolean(document.querySelector(`link[href='${url}']`));
    }
    function getResourceKeys(queuedElements, phase) {
        return queuedElements.filter((el) => el.phase === phase).flatMap((el) => el.resourceKeys);
    }
    const deferrify = (promise) => {
        // legacy for jQuery.Deferred API shape
        const deferred = promise;
        let state = "pending";
        deferred.state = () => state;
        deferred.done = (cb) => {
            promise.then(() => {
                state = "resolved";
                return cb();
            }, noop);
            return deferred;
        };
        deferred.fail = (cb) => {
            promise.catch((vals) => {
                state = "rejected";
                return cb.apply(undefined, vals);
            });
            return deferred;
        };
        deferred.always = (cb) => {
            promise.finally(cb);
            return deferred;
        };
        deferred.promise = () => deferred;
        deferred.progress = () => deferred;
        return deferred;
    };
    const buildCacheKey = (resources) => {
        return resources.join(",");
    };
    const withLocale = (url) => {
        const locale = window.WRM.__localeOverride;
        if (locale === undefined) {
            return url;
        }
        if (typeof locale !== "string" || locale.trim() === "") {
            logger.warn("Locale override failed. The window.WRM.__localeOverride should either be undefined or a non-empty, non-blank string.");
            return url;
        }
        try {
            const urlObject = new URL(url);
            urlObject.searchParams.set("locale", locale);
            logger.warn("Resource locale is overriden by window.WRM.__localeOverride: ", locale);
            return urlObject.toString();
        }
        catch (e) {
            logger.warn("Locale override failed. Invalid resource URL. Falling back to the default locale.");
            return url;
        }
    };
    class RequireHandler {
        constructor() {
            this._requireCache = {
                // handle empty resource requests gracefully
                "": Promise.resolve(),
            };
            // Contains a promise for the currently executing to the WRM REST API, otherwise false
            this._queueManager = makeQueueManager(this._getScriptsForResources.bind(this));
            // Helps keep track of already-loaded resources
            this._tracker = new RequestedResourcesTracker(document);
        }
        /**
         * Requires resources on the page.
         * @param {LoaderKey[]} requiredResources list of resources (eg webresources or webresource contexts)
         * @param {Function} [cb] [Deprecated] callback that is executed when the returned Promise resolves.
         * @param {{ phase: 'interaction' }} options
         * @return {Promise} a Promise that is resolved when all JS / CSS resources have been included on the page, or
         * rejects if any errors occur during the loading process.
         */
        _genericRequire(requiredResources, cb, options) {
            // This takes care of requiredResources being `undefined`, `null`, empty strings or something else
            // which is not an Array of strings
            const resources = []
                .concat(requiredResources)
                .filter((res) => typeof res === "string" && res.trim().length > 0);
            // Requests are cached in this._requireCache so that if a client makes multiple requests for the same
            // set of resources, we can piggyback off the promise for the first request.
            // In other words, if a client calls require([a, b]), then does some work, then calls require([a, b])
            // again, the second call should resolve immediately (or after the first call has resolved).
            const cacheKey = buildCacheKey(resources);
            // Just in case some resource being called `hasOwnProperty`
            if (!this._requireCache.hasOwnProperty(cacheKey)) {
                this._requireCache[cacheKey] = new Promise((resolve, reject) => {
                    this._resolveAsync(resources, options.phase).then(resolve, reject);
                });
            }
            const promise = this._requireCache[cacheKey];
            const deferred = deferrify(promise);
            // legacy callback-based style instead of promise-based.
            if (typeof cb === "function") {
                deferred.done(cb);
            }
            return deferred;
        }
        /**
         * Requires resources on the page.
         * @param {LoaderKey[]} requiredResources list of resources (eg webresources or webresource contexts)
         * @param {Function} [callback] [Deprecated] callback that is executed when the returned Promise resolves.
         * @return {Promise} a Promise that is resolved when all JS / CSS resources have been included on the page, or
         * rejects if any errors occur during the loading process.
         */
        require(requiredResources, callback) {
            return this._genericRequire(requiredResources, callback, {
                phase: "require",
            });
        }
        /**
         * Requires resources lazily on the page.
         * @param {LoaderKey[]} requiredResources list of resources (eg webresources or webresource contexts)
         * @param {Function} [callback] [Deprecated] callback that is executed when the returned Promise resolves.
         * @return {Promise} a Promise that is resolved when all JS / CSS resources have been included on the page, or
         * rejects if any errors occur during the loading process.
         */
        requireLazily(requiredResources, callback) {
            return this._genericRequire(requiredResources, callback, {
                phase: "interaction",
            });
        }
        /**
         * Given a list of resources, translates those resources to actual CSS / JS files and includes them on the page
         * @param {LoaderKey[]} resourceKeys - the list of requests made to `WRM.require`
         * @param {'interaction'} phase
         * @return a Promise that is resolved only after all resources have been included on the page
         * @private
         */
        _resolveAsync(resourceKeys, phase) {
            const onRequestFail = ([failcase, args]) => {
                if (failcase !== failcases.LOADER) {
                    const cacheKey = buildCacheKey(resourceKeys);
                    delete this._requireCache[cacheKey];
                }
                return Promise.reject(args);
            };
            return this._queueManager.enqueue(resourceKeys, phase).catch(onRequestFail);
        }
        /**
         * Processes a response from the WRM REST endpoint.
         * This needs to have a 1:1 call relationship between _getScriptsForResources' use of fetch
         * @param {Object} resourceResponse
         * @param {Resource[]} resourceResponse.resources
         * @param {Object} [resourceResponse.unparsedData]
         * @param {Object} [resourceResponse.unparsedErrors]
         * @private
         */
        _processResourceResponse({ unparsedData, unparsedErrors, resources, }) {
            // TODO: Missing tests cases when resources variable is empty and unparsedData or unparsedError contain sth.
            if (unparsedData) {
                WRM._unparsedData || (WRM._unparsedData = {});
                Object.assign(WRM._unparsedData, unparsedData);
                WRM._dataArrived();
            }
            if (unparsedErrors) {
                WRM._unparsedErrors || (WRM._unparsedErrors = {});
                Object.assign(WRM._unparsedErrors, unparsedErrors);
                WRM._dataArrived();
            }
            if (!resources.length) {
                logger.debug("There is nothing to be requested by resource-loader");
                return Promise.resolve();
            }
            const resourcesToLoad = [];
            const cssMediaResourcesToLoad = [];
            for (let resource of resources) {
                const url = withLocale(resource.url);
                this._tracker.addResource(resource.key, resource.batchType);
                if (resource.resourceType === resourceTypes.JS) {
                    if (!isJsAlreadyLoaded(url)) {
                        resourcesToLoad.push(`js!${url}`);
                    }
                }
                else if (resource.resourceType === resourceTypes.CSS) {
                    if (!isCSSAlreadyLoaded(url)) {
                        if (resource.media && "all" !== resource.media) {
                            // HACK: this can't be loaded by curl.js. The solution is to the DOM immediately
                            // using a <link> tag. This means that the callback may be called before the CSS
                            // has been loaded, resulting in a flash of unstyled content.
                            cssMediaResourcesToLoad.push(resource);
                        }
                        else {
                            resourcesToLoad.push(`css!${url}`);
                        }
                    }
                }
                else {
                    logger.log("Unknown resource type required", url);
                }
            }
            logger.log("Downloading resources", resourcesToLoad);
            return new Promise(function (resolve, reject) {
                logger.debug("resource-loader requesting resources: ", resourcesToLoad);
                loadResources(resourcesToLoad, (...args) => {
                    // Add all css media resources. This is done after curl resources to ensure ordering is consistent
                    // with the way resources are delivered on the server.
                    cssMediaResourcesToLoad.forEach(loadCss);
                    logger.debug("resource-loader resolves", args);
                    resolve(args);
                }, (...args) => {
                    logger.debug("resource-loader fails", args);
                    reject(args);
                });
            });
        }
        /**
         * Makes a fetch request to retrieve the actual JS and CSS files required to satisfy the requested resources.
         * @param {Array<LoaderKey[]>} requests - the list of requests made to `WRM.require`
         * @return a Promise that either resolves when all resources from the requested web-resources and contexts
         * have been loaded, or rejects if any errors occur during the loading process.
         * @private
         */
        _getScriptsForResources(queuedElements) {
            if (queuedElements.length === 0) {
                logger.debug("Nothing to request from WRM API");
                return Promise.resolve({ result: undefined, queuedElements });
            }
            // Determine what to request from the server
            const payload = {
                require: getResourceKeys(queuedElements, "require"),
                interaction: getResourceKeys(queuedElements, "interaction"),
                // Determine what resources have already been loaded previously,
                // so we can avoid loading them multiple times.
                exclude: this._tracker.getResources(),
            };
            logger.debug("Payload for WRM API call prepared: ", payload);
            // Wrap the server request; resolve once any post-processing and side-effects
            // have occurred (e.g., adding scripts and links to the page).
            return new Promise((resolve, reject) => {
                let err;
                const failBy = (reason) => (vals) => {
                    err = err || [reason, vals];
                    return Promise.reject();
                };
                logger.debug("Calling WRM API...");
                fetch(contextPath() + "/rest/wrm/2.0/resources", {
                    // the request data may get too long to represent in GET parameters.
                    method: "POST",
                    // each request to the REST endpoint should be considered unique.
                    cache: "no-cache",
                    // ensure we send user credentials.
                    // especially relevant for older browsers where `omit` was the default.
                    credentials: "same-origin",
                    // we should only ever send requests to the same server the original HTML came from.
                    mode: "same-origin",
                    headers: {
                        "Content-Type": "application/json",
                    },
                    body: JSON.stringify(payload),
                })
                    .then((resp) => (resp.ok ? resp.json() : Promise.reject(resp)), failBy(failcases.NETWORK))
                    .then(({ require, interaction }) => {
                    logger.debug("WRM API responded: ", {
                        require,
                        interaction,
                    });
                    // This is the place where the phases are forced to load in sequence.
                    // We don't start loading lazy scripts until all non-lazy are executed.
                    const requireQueuePromise = () => this._processResourceResponse(require);
                    const interactionQueuePromise = () => this._processResourceResponse(interaction);
                    return requireQueuePromise().then(interactionQueuePromise);
                }, failBy(failcases.DISCOVERY))
                    .then((result) => resolve({ result, queuedElements }), failBy(failcases.LOADER))
                    .catch((data) => {
                    const error = data ? [failcases.EXCEPTION, data] : err;
                    logger.debug("WRM API call failed", error);
                    reject({ result: error, queuedElements });
                });
            });
        }
    }

    const requireHandler = new RequireHandler();
    const makeRequire = (functionName) => (resources, callback) => {
        return requireHandler[functionName](resources, callback);
    };
    const wrmRequire = makeRequire('require');
    const wrmRequireLazily = makeRequire('requireLazily');
    // We expose those in a way that preserves context of RequireHandler instance
    WRM.require = wrmRequire;
    WRM.requireLazily = wrmRequireLazily;
    WRM.setLoggingLevel = setLoggingLevel;
    // add also AMD module
    if (typeof define === 'function') {
        define('wrm/require', () => wrmRequire);
        define('wrm/require-lazily', () => wrmRequireLazily);
        define('wrm/set-logging-level', () => setLoggingLevel);
        define('wrm', () => WRM);
    }

})();
