/**
 * @module confluence-like/like-summary-factory
 */
define('confluence-like/like-summary-factory', [
    'ajs',
    'confluence/templates',
    'jquery'
], function(
    AJS,
    Templates,
    $
) {
    'use strict';

    /**
     * Maximum number of likes from a person's network to promote (read: show the full names of upfront) in the "like" summary
     */
    var MAX_PROMOTED = 3;

    var TRANSLATIONS = {
        'likes.summary.you': AJS.I18n.getText('likes.summary.you'),
        'likes.summary.1.promoted': AJS.I18n.getText('likes.summary.1.promoted'),
        'likes.summary.2.promoted': AJS.I18n.getText('likes.summary.2.promoted'),
        'likes.summary.3.promoted': AJS.I18n.getText('likes.summary.3.promoted'),
        'likes.summary.1.non-promoted': AJS.I18n.getText('likes.summary.1.non-promoted'),
        'likes.summary.x.non-promoted': AJS.I18n.getText('likes.summary.x.non-promoted'),
        'likes.summary.you.1.promoted': AJS.I18n.getText('likes.summary.you.1.promoted'),
        'likes.summary.you.2.promoted': AJS.I18n.getText('likes.summary.you.2.promoted'),
        'likes.summary.you.3.promoted': AJS.I18n.getText('likes.summary.you.3.promoted'),
        'likes.summary.you.1.non-promoted': AJS.I18n.getText('likes.summary.you.1.non-promoted'),
        'likes.summary.you.x.non-promoted': AJS.I18n.getText('likes.summary.you.x.non-promoted'),
        'likes.summary.1.promoted.1.non-promoted': AJS.I18n.getText('likes.summary.1.promoted.1.non-promoted'),
        'likes.summary.1.promoted.x.non-promoted': AJS.I18n.getText('likes.summary.1.promoted.x.non-promoted'),
        'likes.summary.2.promoted.1.non-promoted': AJS.I18n.getText('likes.summary.2.promoted.1.non-promoted'),
        'likes.summary.2.promoted.x.non-promoted': AJS.I18n.getText('likes.summary.2.promoted.x.non-promoted'),
        'likes.summary.3.promoted.1.non-promoted': AJS.I18n.getText('likes.summary.3.promoted.1.non-promoted'),
        'likes.summary.3.promoted.x.non-promoted': AJS.I18n.getText('likes.summary.3.promoted.x.non-promoted'),
        'likes.summary.you.1.promoted.1.non-promoted': AJS.I18n.getText('likes.summary.you.1.promoted.1.non-promoted'),
        'likes.summary.you.1.promoted.x.non-promoted': AJS.I18n.getText('likes.summary.you.1.promoted.x.non-promoted'),
        'likes.summary.you.2.promoted.1.non-promoted': AJS.I18n.getText('likes.summary.you.2.promoted.1.non-promoted'),
        'likes.summary.you.2.promoted.x.non-promoted': AJS.I18n.getText('likes.summary.you.2.promoted.x.non-promoted'),
        'likes.summary.you.3.promoted.1.non-promoted': AJS.I18n.getText('likes.summary.you.3.promoted.1.non-promoted'),
        'likes.summary.you.3.promoted.x.non-promoted': AJS.I18n.getText('likes.summary.you.3.promoted.x.non-promoted')
    };

    var LikeSummaryFactory = {
        /**
         * Returns an i18n message that can be translated into a like summary
         * @param likes a collection of likes
         * @param contentId the content id
         * @param remoteUser the remote user (logged in user)
         */
        getLikeSummary: function(likes, contentId, remoteUser) {
            if (!likes || likes.length === 0) {
                return { key: '', text: '' };
            }

            if (!contentId) {
                throw new Error('contentId is required.');
            }

            var me;
            var promoted = []; // "promoted" is defined as the user's full name is shown in the summary rather than hidden in the dialog triggered by "X others"
            var nonPromoted = [];

            $.each(likes, function(i, like) {
                if (like.user && like.user.name == remoteUser) {
                    me = like;
                } else if (promoted.length < MAX_PROMOTED && like.user.followedByRemoteUser) {
                    promoted.push(like);
                } else {
                    nonPromoted.push(like);
                }
            });

            var i18nKey = ['likes.summary'];
            var i18nArgs = [];

            if (me != null) {
                i18nKey.push('.you');
            }

            if (promoted.length > 0) {
                i18nKey.push('.');
                i18nKey.push(promoted.length);
                i18nKey.push('.promoted');

                $.each(promoted, function(i, like) {
                    i18nArgs.push(Templates.Likes.userLink(like));
                });
            }

            if (nonPromoted.length === 1) {
                i18nKey.push('.1.non-promoted');
                i18nArgs.push(Templates.Likes.userLink(nonPromoted[0]));
            } else if (nonPromoted.length > 1) {
                i18nKey.push('.x.non-promoted');
                i18nArgs.push(nonPromoted.length);
                i18nArgs.push('class="likes" data-content-id="' + contentId + '"');
            }

            var i18nKeyString = i18nKey.join('');

            return {
                key: i18nKeyString,
                args: i18nArgs.length === 0 ? undefined : i18nArgs,
                text: (i18nKeyString in TRANSLATIONS) ? TRANSLATIONS[i18nKeyString] : ''
            };
        }
    };

    return LikeSummaryFactory;
});

require('confluence/module-exporter').exportModuleAsGlobal('confluence-like/like-summary-factory', 'Confluence.Likes.LikeSummaryFactory');
