/* eslint-disable strict,@atlassian/confluence-server/use-strict-amd */
/**
 * @module confluence-like/like
 */
define('confluence-like/like', [
    'ajs',
    'confluence/legacy',
    'jquery',
    'confluence/meta',
    'confluence/message-controller'
], function(
    AJS,
    Confluence,
    $,
    Meta,
    MessageController
) {
    var LIKE = 0;
    var UNLIKE = 1;
    var likesDialog;
    /**
     * A hash of contentId to likes data (used to facilitate "instant like").
     *
     * The "data" value must contain: "content_type" string and a "likes" array. e.g.
     * {
     *     likes: [ { name: "fred" } ],
     *     "content_type": "page"
     * }
     */
    var likesCache = {};

    function getRestUrl(contentId) {
        return AJS.contextPath() + '/rest/likes/1.0/content/' + contentId + '/likes';
    }

    /**
     * Like and follow actions should only be available to licensed authenticated users.
     * @returns {boolean|*}
     */
    var remoteUserCanLike = remoteUserHasLicensedAccess() && !isReadOnlyModeEnabled();
    var showFollowActionsForRemoteUser = remoteUserHasLicensedAccess;

    function isReadOnlyModeEnabled() {
        return Meta.get('access-mode') === 'READ_ONLY';
    }

    function manageFocusAndUnbind() {
        $('a.likes').focus();
        // eslint-disable-next-line @atlassian/confluence-server/deprecated-identifier
        AJS.unbind('remove.dialog', manageFocusAndUnbind);
        // eslint-disable-next-line @atlassian/confluence-server/deprecated-identifier
        AJS.unbind('hide.dialog', manageFocusAndUnbind);
    }

    function remoteUserHasLicensedAccess() {
        var remoteUser = Meta.get('remote-user') || '';
        return remoteUser.length > 0 && Meta.get('remote-user-has-licensed-access') !== false;
    }

    var likesCacheDeferred = $.Deferred();

    var Likes = {};

    Likes.getLikesCache = function() {
        return likesCacheDeferred.promise();
    };

    Likes.showLikeUsers = function(e) {
        e && e.preventDefault();
        var $this = $(this);
        var contentId = $this.data('content-id');
        $this.blur();

        if (likesDialog) {
            likesDialog.remove();
            likesDialog = undefined;
        }

        likesDialog = new AJS.Dialog(400, 365, 'likes-dialog');
        likesDialog.addHeader(AJS.I18n.getText('likes.peoplewholikethis'));
        likesDialog.addPanel('Panel 1', '<div class=\'spinner-container\'></div>');
        likesDialog.addCancel(AJS.I18n.getText('close.name'), function(dialog) {
            dialog.remove();
            likesDialog = undefined;
        });
        likesDialog.getCurrentPanel().setPadding(0);

        likesDialog.show();

        // set focus to modal
        likesDialog.popup.element.attr('tabindex', -1);
        likesDialog.popup.element.focus();

        // put focus back to trigger element when modal is closed
        // eslint-disable-next-line @atlassian/confluence-server/deprecated-identifier
        AJS.bind('remove.dialog', manageFocusAndUnbind);
        // eslint-disable-next-line @atlassian/confluence-server/deprecated-identifier
        AJS.bind('hide.dialog', manageFocusAndUnbind);

        $.ajax({
            type: 'GET',
            url: getRestUrl(contentId),
            data: {
                expand: 'user',
                max: 50
            },
            dataType: 'json'
        }).done(function(data) {
            if (!likesDialog.popup.element.is(':visible')) {
                return;
            }

            data.showFollowActions = showFollowActionsForRemoteUser();
            likesDialog.getCurrentPanel().html(Confluence.Templates.Likes.likesDialog(data));

            $('#likes-dialog').find('.likes-dialog-follow-button').click(function() {
                var $button = $(this);
                $.ajax({
                    type: 'PUT',
                    // eslint-disable-next-line @atlassian/confluence-server/deprecated-identifier
                    url: AJS.contextPath() + '/rest/likes/1.0/user/' + Meta.get('remote-user') + '/following?username=' + $button.data('username'),
                    contentType: 'application/json',
                    dataType: 'json'
                }).done(function() {
                    $button.replaceWith(AJS.I18n.getText('likes.following'));
                });
            });
        });
    };

    /**
     * @param data a hash with a "likes" and "content_type" as required properties.
     * @param contentId
     * @param $summaryContainer
     */
    function updateLikeSummary(data, contentId, $summaryContainer) {
        var remoteUser = Meta.get('remote-user') || '';

        var summary = Confluence.Likes.LikeSummaryFactory.getLikeSummary(data.likes, contentId, remoteUser);

        if (!summary.key) {
            if (remoteUserCanLike && (data.content_type === 'page' || data.content_type === 'blogpost')) {
                summary.text = AJS.I18n.getText('likes.bethefirst');
            }
        }

        // Can't pass summary.args to AJS.format because it is an array. Need to call AJS.format.apply()
        // which takes an array of arguments, so need to construct the string and args into an array
        var formatArgs = [];
        formatArgs.push(summary.text);
        formatArgs = formatArgs.concat(summary.args);
        var summaryText = AJS.format.apply(AJS, formatArgs);

        if (summaryText && summaryText.length > 0) {
            if (remoteUserCanLike) {
                $summaryContainer.addClass('like-summary-margin-left');
            }
            $summaryContainer.html(summaryText);
        } else {
            $summaryContainer.removeClass('like-summary-margin-left');
            $summaryContainer.empty();
        }

        if (!summaryText) {
            return;
        }

        $summaryContainer.find('.likes').click(Likes.showLikeUsers);
    }

    function createHandlerFor(type, contentId, contentType) {
        var remoteUser = Meta.get('remote-user') || '';

        if (type === undefined) {
            throw new Error('type is required');
        }
        if (contentId === undefined) {
            throw new Error('contentId is required');
        }
        if (contentType === undefined) {
            throw new Error('contentType is required');
        }

        return function() {
            if (typeof this !== 'object' || !this.nodeType || this.nodeType !== 1 || this.nodeName !== 'BUTTON') {
                throw new Error('this handler should be bound to a DOM button element');
            }

            var $likeButton = $(this);
            var callee = arguments.callee;
            var $summaryContainer = $likeButton.next('.like-summary');
            $.ajax({
                type: type === LIKE ? 'POST' : 'DELETE',
                url: getRestUrl(contentId),
                contentType: 'application/json',
                data: {
                    'atlassian-token': Meta.get('atlassian-token')
                },
                dataType: 'json',
                timeout: 10000
            }).fail(function(jqXhr) {
                var likeErrorText;
                if (jqXhr.status === 405) {
                    MessageController.showError(MessageController.parseError(jqXhr), MessageController.Location.FLAG);
                } else if (type === LIKE) {
                    likeErrorText = AJS.I18n.getText('likes.like.failed');
                } else {
                    likeErrorText = AJS.I18n.getText('likes.unlike.failed');
                }
                var $likeError = $summaryContainer.siblings('.like-error');
                if ($likeError.length === 0 && likeErrorText !== undefined) {
                    $summaryContainer.after('<span class="like-error" title="' + likeErrorText + '"></span>');
                } else {
                    $likeError.attr('title', likeErrorText);
                }
                $likeButton
                    .off('click')
                    .on('click', null, type === LIKE ? createHandlerFor(LIKE, contentId, contentType) : createHandlerFor(UNLIKE, contentId, contentType))
                    .find('.like-button-text')
                    .html(type === LIKE ? AJS.I18n.getText('likes.like') : AJS.I18n.getText('likes.unlike'));
            }).success(function() {
                $summaryContainer.attr('data-liked', type === LIKE);
                $summaryContainer.parent().find('.like-error').remove();
                updateLikeSummary(likesCache[contentId], contentId, $summaryContainer);
            });
            updateLikeButton($likeButton, type, callee, remoteUser, contentId, contentType, $summaryContainer);
            return false;
        };
    }

    /**
     * @param $likeButton the like button jQuery object
     * @param type the type (LIKE or UNLIKE)
     * @param callee the handler method
     * @param remoteUser the remote user
     * @param contentId the content ID
     * @param contentType the content type
     */
    function updateLikeButton($likeButton, type, callee, remoteUser, contentId, contentType) {
        // instantly update like link
        $likeButton
            .off('click', null, callee)
            .on('click', null, type === LIKE ? createHandlerFor(UNLIKE, contentId, contentType) : createHandlerFor(LIKE, contentId, contentType))
            .attr('aria-pressed', type === LIKE)
            .find('.like-button-text')
            .html(type === LIKE ? AJS.I18n.getText('likes.unlike') : AJS.I18n.getText('likes.like'));
        likesCache[contentId] = likesCache[contentId] || {
            content_type: contentType,
            likes: []
        };

        if (type === LIKE) { // simulate an immediate "like"
            likesCache[contentId].likes.push({
                user: { name: remoteUser }
            });
        } else { // simulate an immediate "unlike"
            likesCache[contentId].likes = $.grep(likesCache[contentId].likes, function(like) {
                return like.user.name !== remoteUser;
            });
        }

        if (type === LIKE) {
            // eslint-disable-next-line @atlassian/confluence-server/deprecated-identifier
            AJS.trigger('analytics', {
                name: 'confluence.' + contentType + '.like.create',
                data: {
                    pageID: Meta.get('page-id')
                }
            });
        }
    }

    Likes.appendAction = function($comments) {
        /* CONFDEV-13029: Get in between comment-action-remove and comment-date, will just append in case selector doesn't match. */
        var commentActionsPrimaryItems = $comments.find('.comment-actions-primary');
        var commentLikeItemText = $(Confluence.Templates.Likes.commentLikeSection({ showLikeButton: remoteUserCanLike }));
        commentActionsPrimaryItems.append(commentLikeItemText);
    };

    Likes.reload = function(data) {
        $.each(data, function(contentId, innerData) {
            var $comment = $('#comment-' + contentId);

            var $likeSummary = $comment.find('.like-summary');
            updateLikeSummary(innerData, contentId, $likeSummary);

            likesCache[contentId] = innerData;
        });

        var $pageComments = $('#page-comments');

        !remoteUserCanLike && $pageComments.find('.like-summary:empty').each(function() {
            $(this).closest('.comment-action-like').hide();
        });

        $pageComments.find('.comment').each(function() {
            Likes.updateComment($(this), data);
        });

        likesCacheDeferred.resolve(likesCache);
    };

    Likes.updateComment = function($comment, data) {
        var id = $comment.attr('id');
        if (!id) {
            return true; // comment should have an ID attribute, if not skip
        }

        var commentId = (/^comment-(\d+)$/.exec(id) || [])[1];
        if (!commentId) {
            throw new Error('Expecting ID attribute of comment to be in format "comment-XXX", found: ' + id);
        }

        var existingLike = data[commentId] && Meta.get('remote-user') && $.grep(data[commentId].likes, function(like) {
            return like.user.name === Meta.get('remote-user');
        }).length > 0;

        $comment.find('.like-button')
            .click(existingLike ? createHandlerFor(UNLIKE, commentId, 'comment') : createHandlerFor(LIKE, commentId, 'comment'))
            .find('.like-button-text')
            .html(existingLike ? AJS.I18n.getText('likes.unlike') : AJS.I18n.getText('likes.like'));
    };

    Likes.init = function() {
        var $contentLikeSection = $(Confluence.Templates.Likes.likeSection({
            showLikeButton: remoteUserCanLike
        }));

        if (Meta.get('page-id')) {
            $.ajax({
                type: 'GET',
                url: getRestUrl(Meta.get('page-id')),
                data: {
                    commentLikes: true
                },
                dataType: 'json'
            }).done(function(data) {
                if (Meta.get('remote-user')) {
                    var existingLike = $.grep(data.likes, function(like) {
                        return like.user.name === Meta.get('remote-user');
                    }).length > 0;

                    var contentType = data.content_type;
                    $contentLikeSection.find('.like-button')
                        .click(existingLike ? createHandlerFor(UNLIKE, Meta.get('page-id'), contentType) : createHandlerFor(LIKE, Meta.get('page-id'), contentType))
                        .attr('aria-pressed', existingLike)
                        .find('.like-button-text')
                        .html(existingLike ? AJS.I18n.getText('likes.unlike') : AJS.I18n.getText('likes.like'));
                }

                var $likeSummary = $contentLikeSection.find('.like-summary');

                updateLikeSummary(data, data.content_id, $likeSummary);

                if ($likeSummary.html() === '' && !remoteUserCanLike) {
                    $contentLikeSection.hide();
                }

                // This container is also used by the labels subsystem, so it must always be added,
                // even if the likes part of it isn't present.
                $('<div id=\'likes-and-labels-container\'></div>')
                    .insertBefore('#labels-section')
                    .append($contentLikeSection)
                    .append($('#labels-section'));

                likesCache[data.content_id] = data;

                Likes.reload(data.commentLikes);
            });
        }

        Likes.appendAction($('#page-comments'));
    };

    return Likes;
});

require('confluence/module-exporter').safeRequire('confluence-like/like', function(Likes) {
    'use strict';

    var Confluence = require('confluence/legacy');
    var AJS = require('ajs');

    Confluence.Likes.getLikesCache = Likes.getLikesCache;

    var isPageGadget = AJS.PageGadget || (window.parent.AJS && window.parent.AJS.PageGadget);
    if (!isPageGadget) {
        AJS.toInit(Likes.init);
        Confluence.Likes.showLikeUsers = Likes.showLikeUsers;
        Confluence.Likes.appendAction = Likes.appendAction;
        Confluence.Likes.reload = Likes.reload;
        Confluence.Likes.updateComment = Likes.updateComment;
    }
});
