define('acm/app/views/data-supplier-view', [
    'acm/app/ajs',
    'backbone',
    'csvgenerator'
],
function (
    AJS,
    Backbone,
    Csvgenerator
) {
    'use strict';

    function getData(rows, options) {
        return Object
            .keys(rows)
            .map(function(key) {
                return {
                    key: key,
                    values: rows[key]
                };
            })
            .sort(function(rowA, rowB) {
                if (options.sortAutomatically) {
                    return rowA.values[0].localeCompare(rowB.values[0]);
                }
                return 0;
            });
    }

    var DataSupplierView = Backbone.View.extend(
        {
            el: '#cluster-monitoring-data-container',
            headerEl: '.header-container',
            descriptionEl: ".description-container",
            actionsEl: '.actions-container',
            tableEl: '.data-supplier-table-container',
            tableSearchEl: '.data-supplier-table-search-container',
            noDataInfoEl: '.no-data-info-container',

            containerView: Cluster.Monitoring.Templates.supplierContainerView,
            headerView: Cluster.Monitoring.Templates.headerContainerView,
            descriptionView: Cluster.Monitoring.Templates.descriptionContainerView,
            actionsView: Cluster.Monitoring.Templates.actionsView,
            supplierDataTableView: Cluster.Monitoring.Templates.supplierDataTable,
            supplierDataTableSearchView: Cluster.Monitoring.Templates.supplierDataTableSearch,
            noDataInfoView: Cluster.Monitoring.Templates.supplierNoDataInfoContainerView,

            events: {
                'click .export-link': 'exportCsv',
                'click .refresh-link': 'refreshView',
                'click .clustering-view-link': 'closeView',
                'input .data-supplier-table-search': 'filterTable'
            },

            initialize: function () {
                this.listenTo(this.model, 'error', this.renderNoData);
                this.listenTo(this.model, 'sync', this.renderSupplierData);
            },

            //clearing all dom updates, can't call "this.remove" because it will remove container el :-(
            closeView: function () {
                this.$el.empty();

                this.undelegateEvents();
                this.stopListening();
            },

            refreshView: function (e) {
                e.preventDefault();
                this.model.fetch();
            },

            exportCsv: function (e) {
                var jsonModel = this.model.toJSON();

                var csv = Csvgenerator.generate(jsonModel.data.columns, jsonModel.data.rows);

                var uri = 'data:text/csv;charset=utf-8,' + encodeURI(csv);

                this.$el.find('.export-link').attr('href', uri);
                this.$el.find('.export-link').attr('download', 'export.csv');
            },

            renderViewContainer: function() {
                AJS.trigger('analyticsEvent', {
                    name: 'monitoringDataSupplierView.rendered',
                    data: {
                        pluginKey: this.model.pluginKey,
                        moduleKey: this.model.moduleKey
                    }
                });

                this.$el.empty();

                var containerView = this.containerView({
                    pluginKey: this.model.pluginKey,
                    moduleKey: this.model.moduleKey
                });

                this.$el.html(containerView);
            },

            filterTable: function(e) {
                var query = e.currentTarget.value;
                this.$('.data-supplier-table tbody tr').each(function(index, row) {
                    var $row = $(row);
                    $row.addClass('hidden');
                    if ($row.text().toLowerCase().includes(query.toLowerCase())) {
                        $row.removeClass('hidden');
                    }
                });
            },

            renderActions: function (displayClusteringViewLink, displayRefreshLink, displayExportLink) {
                this.$el.find(this.actionsEl).html(this.actionsView({
                        displayClusteringViewLink: displayClusteringViewLink,
                        displayRefreshLink: displayRefreshLink,
                        displayExportLink: displayExportLink
                    }
                ));
            },

            renderSupplierData: function () {

                this.renderViewContainer();

                var jsonModel = this.model.toJSON();

                this.$el.find(this.headerEl).html(this.headerView({title: jsonModel.supplier.i18nNameLabel}));

                if (jsonModel.data.description) {
                    var descriptionView = this.descriptionView({
                        description: jsonModel.data.description
                    });

                    this.$el.find(this.descriptionEl).html(descriptionView);
                }

                if (_.isEmpty(jsonModel.data.columns)) {
                    this.renderNoData();
                } else {
                    this.renderTableView(jsonModel);
                }
            },

            renderNoData: function () {
                this.renderActions(true, true, false);

                var noDataInfoView = this.noDataInfoView({
                    header: AJS.I18n.getText("com.atlassian.cluster.monitoring.supplier.nodata.header"),
                    text: AJS.I18n.getText("com.atlassian.cluster.monitoring.supplier.nodata.text")
                });

                this.$el.find(this.noDataInfoEl).html(noDataInfoView);
            },

            renderTableView: function(jsonModel) {
                this.renderActions(true, true, true);
                this.renderDataTableSearch();
                this.renderDataTable(jsonModel);
            },

            renderDataTableSearch: function() {
                this.$el
                    .find(this.tableSearchEl)
                    .html(this.supplierDataTableSearchView());
            },

            renderDataTable: function(jsonModel){
                var supplierView = this.supplierDataTableView({
                    pluginKey: this.model.pluginKey,
                    moduleKey: this.model.moduleKey,
                    nodeId: this.model.nodeId,
                    label: jsonModel.supplier.i18nNameLabel,
                    columns: jsonModel.data.columns,
                    rows: getData(jsonModel.data.rows, {
                        sortAutomatically: jsonModel.data.sortAutomatically,
                    })
                });

                var $tableContainer = this.$el
                    .find(this.tableEl)
                    .html(supplierView);

                AJS.tablessortable.setTableSortable($tableContainer.find('.data-supplier-table'));
            }
        }
    );

    return DataSupplierView;
});
