$(function($) {

    /**
     * Create a resourceDescriptor with the supplied completeKey, version and
     * optional context that the resource was served for.
     *
     * @param completeKey the unique key for the resource (e.g. plugin key and resource key)
     * @param version the version of its plugin this resource came from
     */
    function ResourceDescriptor(completeKey, version) {
        if (!completeKey) {
            throw new Error("The resourceDescriptor requires a completeKey");
        }

        if (!version) {
            throw new Error("The resourceDescriptor requires a plugin version");
        }

        this.completeKey = completeKey;
        this.version = version;
    }

    /**
     * A comparator for resourceDescriptor which only takes the completeKey into account.
     *
     * The version of resourceDescriptors is not currently significant.
     */
    function resourceDescriptorComparator(a,b) {
        return a.completeKey.localeCompare(b.completeKey);
    }

    /**
     * Given a jQuery wrapped collection of Elements, create the ResourceDescriptors they represent.
     * @param $elements the elements contains resource descriptions
     * @return an array of ResourceDescriptors for all the elements supplied.
     */
    function extractResourceDescriptors($elements) {
        var resourceDescriptors = new Array();
        if (!$elements) {
            return resourceDescriptors;
        }

        $.each($elements, function(index, item) {
            if (item.nodeType == 1) {
                var resourceAttr = $(item).data("atlassian-webresource-contents");
                if (resourceAttr) {
                    var resources = resourceAttr.split(",");
                    $.each(resources, function(index, item) {
                        // extract the version info using regex
                        var version = item.match(/\[(.+)\]$/);
                        var versionStr = "0"; // default to no version
                        if (version && version.length == 2) {
                            item = item.substring(0, item.length - version[0].length);
                            versionStr = version[1];
                        }

                        resourceDescriptors.push(new ResourceDescriptor(item, versionStr));
                    });
                }
            }
        });

        return resourceDescriptors;
    };

    /**
     * Given a jQuery wrapped collection of Elements, extract the contexts they contain web resources for.
     * @return an Array of Strings which are the contexts represented by the supplied Elements.
     */
    function extractContexts($elements) {
        var contexts = new Array();
        if (!$elements) {
            return contexts;
        }

        $.each($elements, function(index,item) {
            if (item.nodeType == 1) {
                var contextAttr = $(item).data("atlassian-webresource-contexts");
                if (contextAttr) {
                    contexts = contexts.concat(contextAttr.split(","));
                }
            }
        });

        return contexts;
    };

    /**
     * The callback used to evaluate any scripts dynamically loaded. Based on the globalEval
     * function from jQuery but with the IE stuff stripped.
     */
    function evalScriptCallback(data, status, xhr) {
        if (data && /\S/.test(data)) {
            try {
                (function(data) {
                    window["eval"].call(window, data);
                })(data);

            } catch (err) {
                if (console && console.log && console.error) {
                    console.log("Error while evaluating dynamically loaded script.");
                    console.error(err);
                }
            }
        }
    };

    /**
     * @param numScripts the number of scripts to be loaded
     * @param completeCallback the callback to be called once all scripts have loaded as provided by the client.
     */
    var ScriptLoadTracker = function(numScripts, completeCallback) {
        this.counter = numScripts;
        this.callback = completeCallback;

        this.complete = function() {
            this.counter--;
            if (this.counter == 0) {
                this.callback();
            }
        };
    };

    function createScriptLoadCallback(numScripts, completeCallback) {
        var scriptLoadTracker = new ScriptLoadTracker(numScripts, completeCallback);
        return function(xhr, status) {
            scriptLoadTracker.complete();
        }
    };

    /**
     * Create a load callback that will update the contentLoader with any retrieved
     * dependencies on successful load before calling the supplied onSuccess function.
     *
     * @param onSuccess a function taking the parameters (data, status, xhr) which has been provided by the client
     * (see the Java class com.atlassian.confluence.plugins.mobile.rest.model.ContentDto)
     * @return a jQuery success callback function
     */
    function createContentLoadCallback(contentLoader, onSuccess) {
        return function(data, status, xhr) {
            if (data.webResourceDependencies) {
                var deps = data.webResourceDependencies;

                if (deps.cssResourceTags) {
                    var $cssTags = $(deps.cssResourceTags);
                    var contexts = extractContexts($cssTags);
                    var resourceDescriptors = extractResourceDescriptors($cssTags);

                    contentLoader.addContexts(contexts);
                    contentLoader.addResources(resourceDescriptors);

                    $("head").append($cssTags);
                }

                var scriptLoadComplete = function() {
                    onSuccess(data,status,xhr);
                };

                if (deps.jsResourceTags) {
                    var $jsTags = $(deps.jsResourceTags).filter("script");
                    var contexts = extractContexts($jsTags);
                    var resourceDescriptors = extractResourceDescriptors($jsTags);

                    contentLoader.addContexts(contexts);
                    contentLoader.addResources(resourceDescriptors);


                    // Use jQuery to AJAX fetch and then globalEval the scripts
                    var ajaxSettings = {
                        dataType: "script",
                        success: evalScriptCallback,
                        complete: createScriptLoadCallback($jsTags.length, scriptLoadComplete),
                        timeout: 30000
                    };

                    $jsTags.each(function(index, scriptEl) {
                        ajaxSettings.url = $(scriptEl).attr("src");
                        $.ajax(ajaxSettings);
                    });
                } else {
                    // no scripts to load so call onSuccess
                    scriptLoadComplete();
                }
            }
        };
    };

    /**
     * Construct a new ContentLoader initialised from the supplied elements.
     *
     * @param $elements the elements to initialise the dependencies from
     * @param contentResourceUrl the URL of the ContentResource to load content from
     */
    function ContentLoader($elements, contentResourceUrl) {
        if (!contentResourceUrl) {
            throw new Error("A URL for the content REST resource is required.");
        }

        this._resourceTree = new ConfluenceMobile.Utils.BinarySearchTree(resourceDescriptorComparator);
        this._contexts = new Array(); // perhaps should be a tree but for now we expect a small number of contexts
        this._contentResourceUrl = contentResourceUrl;

        this.addResources(extractResourceDescriptors($elements));
        this.addContexts(extractContexts($elements));
    };

    ContentLoader.prototype = {

        constructor: ContentLoader,

        /**
         * Add multiple resourceDescriptors to the maintained collection.
         */
        addResources: function(resourceDescriptors) {
            var pageLoader = this;
            _.each(resourceDescriptors, function(element,index,list) {
               pageLoader._resourceTree.add(element);
            });
        },

        /**
         * Add multiple contexts preventing duplicates.
         *
         * @param contexts an array of contexts to add
         */
        addContexts: function(contexts) {
            var pageLoader = this;
            _.each(contexts, function(element,index,list) {
                pageLoader.addContext(element);
            });
        },

        /**
         * @param completeKey the unique key for the resource (e.g. plugin key and resource key)
         * @param version the version of its plugin this resource came from
         * requested directly in which case there will be no context.
         */
        addResource: function(completeKey, version) {
            var resourceDescriptor = new ResourceDescriptor(completeKey, version);
            this._resourceTree.add(resourceDescriptor);
        },

        addContext: function(context) {
            if (_.indexOf(this._contexts, context) == -1) {
                this._contexts.push(context);
            }
        },

        /**
         * @param completeKey the key of the resource to check for. Any version of this resource is considered a match
         * @return true if the ContentLoader has already loaded the identified resource
         */
        hasResource: function(completeKey) {
            return this._resourceTree.contains(new ResourceDescriptor(completeKey, "0")); // version is irrelevant at the moment
        },

        /**
         * @param context the context to be checked
         * @return true if the ContentLoader has already loaded resources for this context
         */
        hasContext: function(context) {
            return (_.indexOf(this._contexts, context)) != -1;
        },

        /**
         * @return the number of resources known
         */
        resourceCount: function() {
            return this._resourceTree.size();
        },

        /**
         * @return the number of contexts known
         */
        contextCount: function() {
            return this._contexts.length;
        },

        /**
         * @return an array of the resource keys (without version information) known to the loader.
         */
        getResourceKeys: function() {
            var keys = new Array();

            this._resourceTree.traverse(function(node) {
                keys.push(node.value.completeKey);
            });

            return keys;
        },

        /**
         * @return an array of contexts known to the loader
         */
        getContexts: function() {
            var contexts = new Array();
            return contexts.concat(this._contexts);
        },

        /**
         * Request an identified piece of content from the ContentResource REST resource, tracking
         * the required dependencies as part of the process.
         *
         * @param id the id of the content to be loaded.
         * @param onSuccess the function to call on a successful load. Should take the parameters (data, status, xhr)
         * @param onError the function to call if an error occurs during the load. Should take the parameters (xhr, type)
         * which is the response.
         */
        load: function(id, onSuccess, onError) {
            return $.ajax({
                type: "GET",
                url: this._contentResourceUrl + "/" + id,
                data: {
                    knownContexts: this.getContexts().toString(),
                    knownResources: this.getResourceKeys().toString()
                },
                dataType: "json",
                success: createContentLoadCallback(this, onSuccess),
                error: onError
            });
        }
    };

    // expose the ContentLoader constructor to enable unit testing
    ConfluenceMobile.Utils._ContentLoader = ContentLoader;

    /**
     * A component that tracks the context and resource dependencies within the mobile app.
     * This should be used for all content loads to allow it to keep up to date with the
     * dependencies introduced into the single page app as it is navigated around.
     *
     * In the current implementation resource versions are ignored for simplicity.
     */
    ConfluenceMobile.ContentLoader = new ContentLoader($("link[rel=\"stylesheet\"],script"), ConfluenceMobile.AppData.get('confluence-context-path') + '/rest/mobile/1.0/content');
});
