ConfluenceMobile.PhotoUploader = function(options) {
    this.browse_button = options.browse_button;
    this.commentContainer = options.commentContainer;
    this.uploadedFiles = [];
};

ConfluenceMobile.PhotoUploader.prototype = {
    _generateUniqueId: function () {
        return Math.random().toString(36).substr(2, 16); // Generate a random unique ID
    },
    _buildUrl: function(file) {
        const pageId = $("#inner-container").find(".content").data("content-id");
        let url = ConfluenceMobile.AppData.get("confluence-context-path") + "/plugins/drag-and-drop/upload.action";
        const params = {pageId: pageId};
        const extension = file.name.substr(file.name.lastIndexOf(".") + 1);
        params.filename = file.name;
        params.mimeType = AJS.UploadUtils.mimeTypes[extension.toLowerCase()] || "application/x-upload-data"; // if we dont have the mime type just send a default
        params.size = file.size;

        let query = '';
        _.each(params, function(value, name) {
            query += (query ? '&' : '') + encodeURIComponent(name) + '=' + encodeURIComponent(value);
        });

        if (query) {
            url += (url.indexOf('?') > 0 ? '&' : '?') + query;
        }

        return url;
    },
    /**
     * Uploads a file with progress tracking.
     * @param filteredFile object that contains the actual file and corresponding metadata. Created in UploadUtils class.
     * @param upload_url the URL to which the file should be uploaded.
     * @param onUploadProgress callback function that will be called with the filteredFile object when upload progress is made.
     * @param onFileUploaded callback function that will be called with the filteredFile object when the file is successfully uploaded.
     */
    _uploadFileWithProgress: function (filteredFile, upload_url, onUploadProgress, onFileUploaded) {
        if (!filteredFile || !filteredFile.file) {
            console.log("Invalid filteredFile object passed to uploadFileWithProgress.");
            return;
        }

        if (upload_url) {
            return new Promise(function (resolve, reject) {
                filteredFile.xhrConnection = new XMLHttpRequest();
                filteredFile.xhrConnection.open("POST", upload_url, true);
                // Set up progress event listener
                filteredFile.xhrConnection.upload.onprogress = function (event) {
                    if (event.lengthComputable) {
                        let bytesUploaded = event.loaded;
                        try {
                            filteredFile.loaded = bytesUploaded;
                            filteredFile.percent = (filteredFile.loaded / filteredFile.size) * 100;
                            onUploadProgress && onUploadProgress(filteredFile);
                        } catch (error) {
                            filteredFile.errorMessage = error.errorMessage;
                            filteredFile.hasError = true;
                            reject(filteredFile);
                        }
                    }
                };

                filteredFile.xhrConnection.onload = function () {
                    if (filteredFile.xhrConnection.status && filteredFile.xhrConnection.status >= 200 && filteredFile.xhrConnection.status < 300) {
                        try {
                            onFileUploaded(filteredFile);
                            resolve(filteredFile.xhrConnection.response);
                        } catch (error) {
                            filteredFile.errorMessage = error.errorMessage;
                            filteredFile.hasError = true;
                            reject(filteredFile);
                        }
                    } else {
                        filteredFile.errorMessage = filteredFile.xhrConnection.response.responseText;
                        filteredFile.hasError = true;
                        reject(filteredFile);
                    }
                };
                //
                // // Set up error event listener
                filteredFile.xhrConnection.onerror = function () {
                    filteredFile.errorMessage = filteredFile.xhrConnection.response.responseText;
                    filteredFile.hasError = true;
                    reject(filteredFile);
                };

                // Prepare and send the file
                filteredFile.xhrConnection.setRequestHeader('Content-Type', 'application/octet-stream');
                filteredFile.xhrConnection.setRequestHeader('X-Atlassian-Token', 'no-check'); // Add this header
                filteredFile.xhrConnection.send(filteredFile.file);
            });
        }
    },
    /**
     * Callback function that will be called with the filteredFile object when upload progress is made.
      * @param file
     * @private
     */
    _onUploadProgress:  function (file){
        ConfluenceMobile.photoUploaderEventAggregator.trigger("update-dialog-progressbar", file);

    },
    /**
     * Callback function that will be called with the filteredFile object when the file is successfully uploaded.
     * @param file
     * @private
     */
    _onFileUploaded:  function (file) {
        var $commentEditor = $(this.commentContainer).find(".comment-editor");
        ConfluenceMobile.insertInputText($commentEditor, "!" + file.name + "!");
        $commentEditor.focus();
        ConfluenceMobile.Analytics.trackEvent('mobile-photo-upload', 'file-uploaded');
    },
    /**
     * Uploads an array of files.
     * @param files
     * @returns {Promise<void>}
     * @private
     */
    _upload: async function  (files) {
        for (let fileIndex = 0; fileIndex < files.length; fileIndex++) {
            try {
                if (!files[fileIndex].hasError) {
                    let upload_url = this._buildUrl(files[fileIndex]);
                    await this._uploadFileWithProgress(files[fileIndex], upload_url, this._onUploadProgress.bind(this), this._onFileUploaded.bind(this));
                }
            } catch (errorFilteredFileObj) {
                console.log(errorFilteredFileObj)
            }
        }
    },
    /**
     * Aborts the upload of a file if it is in progress.
     * @param file
     * @private
     */
    _abortUpload: function(file) {
        if (file && file.xhrConnection) {
            file.xhrConnection.abort(); // Abort the upload
            file.aborted = true; // Mark the file as aborted
        }
    },
    /**
     * Handles the dialog close event, aborting any ongoing uploads.
     */
    handleDialogClose: function() {
        if (this.uploadedFiles && this.uploadedFiles.length > 0) {
            this.uploadedFiles.forEach(file => {
                this._abortUpload(file);
            });
        }
    },
    /**
     * Initializes the PhotoUploader by binding the change event to the browse button.
     */
    init: function () {
        const self = this;
        this.browse_button.addEventListener('change', async function(event) {
            event.stopPropagation();
            let files = event.target.files; // Get the selected files

            files = Array.from(files).map(file => ({
                id: file.id || self._generateUniqueId(), // Generate or use an existing unique ID
                loaded: 0,
                name: file.name,
                percent: 0,
                size: file.size,
                file: file // Include the original file object
            }));
            self.uploadedFiles = files;
            for (let fileIndex = 0; fileIndex < files.length; fileIndex++) {
                let extension = files[fileIndex].name.split('.').pop();
                files[fileIndex].name = files[fileIndex].id + "." + extension;
            }

            new ConfluenceMobile.PhotoUploadDialog({
                blanket: new ConfluenceMobile.BlanketView(),
                handleDialogClose: self.handleDialogClose.bind(self)
            });

            ConfluenceMobile.photoUploaderEventAggregator.trigger("show-dialog-progressbar", files);

            await self._upload(files);

            ConfluenceMobile.photoUploaderEventAggregator.trigger("close-photoupload-dialog");
        });
    }
}

ConfluenceMobile.PhotoUploader.prototype.destroy = function() {
    if (this.browse_button) {
        this.browse_button.removeEventListener('change', this.changeHandler);
    }
    this.browse_button = null; // Clear reference to allow garbage collection
};
