/**
 * The view for an individual comment.
 *
 * This view raises events using the ConfluenceMobile.contentEventAggregator during rendering to allow for plugin
 * code to add to the display. The following events are raised -
 *
 * 'render:comment-complete' - when the rendering of the comment is complete and the view is present in the DOM. The event listener
 * will get the following parameters -
 *  the CommentView itself,
 *  the jQuery wrapped container for the comment,
 *  the Comment model object.

 */
ConfluenceMobile.CommentView = Backbone.View.extend({

    $uploadButton: null,
    $saveButton: null,
    $contentEditor: null,
    mentionMarkup: [],

    /**
     * A flag that when true indicates we should render an edit view of the comment.
     */
    editing: false,

    uploader: null,

    tagName: "li",
    className: "comment",

    events: {
        "click .add-comment-button": "send",
        "click .fuzzy-time a": "navigateComment",
        "keyup .comment-editor": "typing",
        "click .reply-button": "replyToComment",
        "click .add-mention-button": "showMentionPrompt"
    },

    initialize: function () {
        this.model.bind('change', function () {
            this.toggleEdit(false);
            this.render();
            this.renderActivityIndicator();
        }, this);
        this.model.bind('sync', this.sync, this);
        this.model.bind('error', this.handleError, this);
        this.model.bind('ic:disable-error', this.handleInlineCommentError, this);
    },

    sync: function () {
        this.$(".activity-indicator").hide();
    },

    render: function () {
        var id = this.model.get("id");

        if (this.editing) {
            if (ConfluenceMobile.AppData.getBoolean("require-captcha")) {
                // CONFDEV-9042 - prevent commenting if captcha is required
                this.$el.html(Confluence.Templates.Mobile.Comments.commentingDisabledByCaptcha());
            } else {
                this.$el.html(Confluence.Templates.Mobile.Comments.newCommentView({
                    comment:this.model.attributes,
                    placeholder: this.options.newCommentPlaceholder || AJS.I18n.getText("confluence.mobile.comment.placeholder.message")
                }));
                this.$saveButton = $(".add-comment-button", this.$el);
                this.$contentEditor = $(".comment-editor", this.$el);
                var that = this;

                // Zepto auto bind dynamically injected element.
                // After Zepto got replaced by jQuery, to maintain the same behavior,
                // these events have to be bound manually
                this.$contentEditor.on('keyup', function(e) { that.typing(e); });
                this.$saveButton.on('click', function(e) {that.send(e); });

                ConfluenceMobile.toggleState(this.$saveButton, false);

                // new button each time, so need to re-create/bind
                this.$uploadButton = $(".add-image-button", this.$el);
                this.uploader = new ConfluenceMobile.PhotoUploader({
                    browse_button: this.$uploadButton[0],
                    commentContainer: this.$el
                });
                this.uploader.init();
                this.edited(); // enable the send button if there is content already
            }
        } else {
            id && (this.el.id = "comment-" + id);
            this.$el.html(Confluence.Templates.Mobile.Comments.commentView({ comment:this.model.attributes }));
            this.$saveButton = null;
            this.$contentEditor = null;

            ConfluenceMobile.contentEventAggregator.trigger("render:comment-complete", this, this.$el, this.model);
        }

        return this;
    },

    /**
     * Render an activity indicator. This only applies to a comment that is not being
     * edited. If the CommentView has 'editing' set then nothing will be rendered
     * by this method.
     */
    renderActivityIndicator: function () {
        if (this.editing)
            return;

        this.$(".activity-indicator").html(Confluence.Templates.Mobile.Comments.savingIndicator());
    },

    /**
     * @param edit if true then set the view to render the comment as editable
     */
    toggleEdit: function (edit) {
        this.editing = edit;
    },

    /**
     * Send the comment content.
     */
    send: function (event) {
        // Click events are still happening when the element is "disabled" on iPhone.
        if (event.target.disabled) {
            return;
        }

        if (this.model.id != 0) {
            return; // we only support sending new comments for now.
        }

        var html = ConfluenceMobile.textToHtml(this.$contentEditor.val(), this.mentionMarkup);
        var imgHtml = ConfluenceMobile.includeImages(html);
        this.model.set("html", imgHtml);
        this.saveComment(imgHtml);
        this.uploader.destroy();

        ConfluenceMobile.Analytics.trackEvent("comment", "submit");
        return false; // cancel the form submit
    },

    /**
     * Save the supplied comment content.
     *
     * @param content the content to save for the comment. If not supplied then assume that the comment
     * model already has its content set and simply needs to be saved.
     *
     */
    saveComment: function (content) {
        this.model.save();
    },

    retryCommentSave: function () {
        var retry = confirm(AJS.I18n.getText("confluence.mobile.comment.retry.message"));
        if (retry) {
            this.saveComment();
            this.renderActivityIndicator();
        }
    },

    handleError: function (model, resp, options) {
        var $indicator = this.$(".activity-indicator").html(Confluence.Templates.Mobile.Comments.saveFailIndicator());
        $indicator.click(_.bind(this.retryCommentSave, this));
    },

    handleInlineCommentError: function() {
        this.$el.html(Confluence.Templates.Mobile.Comments.inlineCommentUnavailable());
    },

    /**
     * Set the saving placeholder back to edit. This is typically used after a
     * cancel has occurred from a save operation (e.g. perhaps cancelled a captcha).
     */
    setEditHandler: function(model, view) {
        var content = model.get("html");
        content = ConfluenceMobile.htmlToText(content, this.mentionMarkup);
        model.set("html", content, {"silent":true}); // going to explicitly render so silence the change event
        view.toggleEdit(true);
        view.render();
    },

    typing: function (e) {
        if (this.edited()) {
            this.triggerMention(e);
        }
    },

    /**
     * Called when a comment has been edited. Note that we only allow the 'new comment' to be edited.
     */
    edited: function () {
        if (!this.editing) {
            return false;
        }
        ConfluenceMobile.toggleState(this.$saveButton, this.$contentEditor.val().trim());
        return true;
    },

    /**
     * @param busy if true then show the UI for the new comment as busy (and disabled). Otherwise enable it.
     */
    toggleBusy: function (busy) {
        toggleState(this.$saveButton, !busy);
        toggleState(this.$contentEditor, !busy);
        $(".mobile-comment-saving-indicator", this.$el).toggle(busy);
    },

    navigateComment: function (event) {
        event.preventDefault();
        event.stopPropagation();
        //ConfluenceMobile.router.navigate(event.target.hash, { replace: true, trigger: false });
    },

    replyToComment: function (event) {
        event.preventDefault();
        event.stopPropagation();

        var model = ConfluenceMobile.Comment.nestedCommentFromParent(this.model);
        var view = new ConfluenceMobile.CommentView({
            model: model,
            commentListView: this.options.commentListView,
            newCommentPlaceholder: AJS.I18n.getText("confluence.mobile.comment.placeholder.reply.message")
        });
        view.toggleEdit(true);

        // tell parent view that we open a new comment editor (that closes other open
        // nested comment editors)
        this.options.commentListView.trigger("reply-to-comment", view);

        this.$el.children(".comment-list").prepend(view.render().el);
        this.$el.find('textarea').focus();
    },

    triggerMention: function(e) {
        var BACKSPACE = 8, ENTER = 13;
        var caretPos = this.$contentEditor[0].selectionStart;
        var comment = this.$contentEditor.val();
        var lastTwoCharsTyped = comment.substr(Math.max(0, caretPos - 2), caretPos);
        if (e.which !== BACKSPACE && e.which !== ENTER && lastTwoCharsTyped.match(/\s@$|^@$/)) {
            this.showMentionPrompt(e);
        }
    },

    showMentionPrompt: function(e) {
        e.preventDefault();
        e.stopPropagation();
        if (e.type === "keyup") {
            this.$contentEditor.blur(); // to dismiss the keyboard on iOS because we can't focus field without a click
        }
        new ConfluenceMobile.MentionView({
            insertCallback: _.bind(this.insertUserMention, this),
            cancelCallback: _.bind(this.cancelUserMention, this),
            blanket: new ConfluenceMobile.BlanketView()
        });
    },

    insertUserMention: function(user) {
        if (!user.get('title')) {
            this.cancelUserMention();
            return;
        }
        user.getMentionMarkup(this.mentionMarkup);

        var userString = user.getMentionString();
        ConfluenceMobile.insertInputText(this.$contentEditor, userString);
        this.$contentEditor.focus();

        ConfluenceMobile.Analytics.trackEvent("mention", "insert");
    },

    cancelUserMention: function() {
        this.$contentEditor.focus();
    }
});
