/**
 * @module confluence/convert-page-to-blogpost-dialog
 */
define('confluence/convert-page-to-blogpost-dialog', [
    'ajs',
    'jquery',
    'confluence/meta',
    'confluence/templates',
    'confluence/api/logger',
    'confluence/form-state-control',
    'wrm/context-path',
    'confluence-ui-components/js/space-picker',
    'aui/dialog2',
    'confluence/convert-page-to-blogpost-dialog-analytics',
], function (AJS, $, Meta, Templates, logger, FormStateControl, contextPath, spacePicker, dialog2, analytics) {
    'use strict';

    function viewConvertToBlogHandler(_dialog, newSpaceKey, errorHandler) {
        const convertToBlogDialog = document.querySelector('#convert-page-to-blogpost-dialog');
        convertToBlogDialog.querySelector('.convert-errors').innerHTML = '';
        convertToBlogDialog.querySelector('#convert-error-container').style.display = 'none';
        const convertButton = convertToBlogDialog.querySelector('#convert-button');
        const cancelButton = convertToBlogDialog.querySelector('#cancel-button');
        convertButton.busy();
        FormStateControl.disableElement([convertButton, cancelButton]);

        function error(message) {
            errorHandler(message);
            FormStateControl.enableElement([convertButton, cancelButton]);
            convertButton.idle();
        }

        if (!newSpaceKey) {
            error(AJS.I18n.getText('move.blog.space.picker.label'));
            return;
        }

        const params = new URLSearchParams();
        params.append('pageId', Meta.get('page-id'));
        params.append('spaceKey', newSpaceKey);

        fetch(contextPath() + '/pages/converttoblogpost.action', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/x-www-form-urlencoded',
                Accept: 'application/json',
                'X-Atlassian-Token': 'no-check',
            },
            body: params.toString(),
            credentials: 'same-origin',
        })
            .then((response) => {
                if (!response.ok) {
                    analytics.triggerConvertPageToBlogModalConvertFailureEvent(Meta.get('page-id'), response.status);
                    return response.json().then((errorData) => {
                        error(
                            AJS.I18n.getText('convert.page.to.blogpost.dialog.failed') + ': ' + errorData.errorMessage
                        );
                    });
                }
                return response.json().then((data) => ({ data, status: response.status }));
            })
            .then((result) => {
                if (!result) {
                    return;
                }
                const { data, status } = result;
                // response is 200 with validation errors
                if (data.licenseCheckFailed) {
                    analytics.triggerConvertPageToBlogModalConvertFailureEvent(Meta.get('page-id'), status);
                    dialog2(convertToBlogDialog).$el.removeClass("aui-dialog2-small").addClass("aui-dialog2-medium");
                    convertButton.idle();
                    FormStateControl.enableElement([cancelButton]);
                    if (Templates.licenseCheckFailed[data.licenseCheckFailed]) {
                        convertToBlogDialog.querySelector(".aui-dialog2-content").innerHTML = Templates.licenseCheckFailed[data.licenseCheckFailed]();
                        // set focus to the dialog itself, which is the default behaviour of aui-dialog2 initialisation, to make sure the new content is navigable.
                        convertToBlogDialog.focus();
                    } else {
                        error(AJS.I18n.getText('convert.page.to.blogpost.dialog.failed'));
                    }
                    return;
                }
                const errors = []
                    .concat(data.validationErrors || [])
                    .concat(data.actionErrors || [])
                    .concat(data.errorMessage || []);
                if (errors.length > 0) {
                    analytics.triggerConvertPageToBlogModalConvertFailureEvent(Meta.get('page-id'), status);
                    error(errors[0]);
                    return;
                }

                // response is 200 with success
                analytics.triggerConvertPageToBlogModalConvertSuccessEvent(Meta.get('page-id'));
                const url = new URL(contextPath() + data.page.url, window.location.origin);
                url.searchParams.append('converted', 'true');
                window.location.href = url.toString();
            })
            .catch((err) => {
                error(AJS.I18n.getText('convert.page.to.blogpost.ajax.error'));
                logger.debug(err.message);
            });
    }

    function ConvertToBlogDialog(opts) {
        const options = {
            spaceKey: Meta.get('space-key'),
            spaceName: Meta.get('space-name'),
            convertHandler: function () {
                logger.debug('No convertHandler defined. Closing dialog.');
                dialog.remove();
            },
            cancelHandler: function (dialog) {
                dialog.hide();
                return false;
            },
            ...opts,
        };

        function checkChildExistence() {
            const url = contextPath() + '/rest/page-hierarchy/internal/subtreeCount/' + Meta.get('page-id') + '?totalCountOnly=true';
            fetch(url, {
                credentials: 'same-origin'
            }).then(response => {
                if (!response.ok) {
                    errorHandler(AJS.I18n.getText('convert.page.to.blogpost.ajax.error'));
                }
                return response.json();
            }).then(data => {
                if (data.totalCount > 1) {
                    dialog.$el.find('#convert-warning-container').show();
                    dialog.$el.removeClass("aui-dialog2-small").addClass("aui-dialog2-medium");
                } else {
                    dialog.$el.find('#convert-warning-container').hide();
                    dialog.$el.removeClass("aui-dialog2-medium").addClass("aui-dialog2-small");
                }
            }).catch((e) => {
                errorHandler(AJS.I18n.getText('convert.page.to.blogpost.ajax.error'));
            });
        }

        checkChildExistence();

        const dialog = dialog2(Templates.ConvertToBlogPost.convertToBlogPostDialog(options));
        dialog.show();
        document.querySelector('#convert-button').addEventListener('click', convertButtonHandler);
        document.querySelector('#cancel-button').addEventListener('click', cancelButtonHandler);

        $('#new-space').val(Meta.get('space-key')).confluenceSelect(spacePicker.build());

        const convertFunction = function (dialog) {
            const newSpaceKey = document.querySelector('#new-space').value;
            options.convertHandler(dialog, newSpaceKey, errorHandler);
        };

        function errorHandler(error) {
            document.querySelector('#convert-error-container').style.display = 'block';
            document.querySelector('#convert-error-container .convert-errors').innerHTML = error;
        }

        function convertButtonHandler() {
            convertFunction(dialog);
            analytics.triggerConvertPageToBlogModalConvertClickedEvent(Meta.get('page-id'));
        }

        function cancelButtonHandler() {
            options.cancelHandler(dialog);
        }

        dialog.on('hide', function () {
            document.querySelector('#action-menu-link').focus();
        });

        return dialog;
    }

    return {
        initialiser: function (e) {
            e.preventDefault();

            new ConvertToBlogDialog({
                convertHandler: viewConvertToBlogHandler,
            });

            return false;
        },
    };
});

require('confluence/module-exporter').safeRequire(
    'confluence/convert-page-to-blogpost-dialog',
    function (ConvertToBlogDialog) {
        'use strict';

        const Event = require('confluence/api/event');
        Event.bind('deferred.convert-page-to-blogpost.tools-menu', ConvertToBlogDialog.initialiser);
    }
);
