/**
 * @module confluence/flag
 */
define('confluence/flag', [
    'ajs',
    'jquery'
],
/**
 * A fork of AUI Flags (https://docs.atlassian.com/aui/latest/docs/flag.html) from 2015
 * Confluence flag provides two additional capabilities to AUI Flags.
 * 1. It allows several flags to be stacked on top of each other to consolidate multiple flags in a small space.
 *    This allows frequent notifications such as new comments, inline comments, page notifications to take up less space
 *    In the options, the user passes a 'stack' option with an identifier. All flags created with the same identifier
 *    will be stacked. The user can also specify optionally specify a boolean 'fifo' option. If true, the flag will
 *    be placed at the top of the stack. If false or omitted, the flag will be placed at the bottom of the stack.
 * 2. Triggers an 'aui-flag-show' event anytime a flag is added to the DOM
 * 3. Allows the user to specify extra classes to append to the Flag element
 *
 * Much of the functionality is implemented using styling. Make sure to check the associated CSS file.
 *
 * Note that AUI 9.0+ uses `open` attribute on the flag instead of `aria-hidden` to control its visibility.
 * This fork supports both for backward compatibility (the 'open' attribute has AUI styles bound to it).
 *
 * From 7.11, the close button selector changes from `span.aui-icon.icon-close` to `button.aui-close-button`
 *
 * @tainted AJS._internal
 */
function(
    AJS,
    $
) {
    'use strict';

    var AUTO_CLOSE_TIME = 5000;
    var ID_FLAG_CONTAINER = 'aui-flag-container';
    var defaultOptions = {
        body: '',
        close: 'manual',
        title: '',
        type: 'info',
        fifo: false
    };

    var closeButtonLabel = '';
    try {
        // The price to pay of a fork
        // eslint-disable-next-line atlassian-wrm-i18n/matching-i18n-key-in-properties-file
        closeButtonLabel = AJS.I18n.getText('aui.words.close');
    } catch (e) {
        // Ignore error (in unit test)
    }

    var CLOSE_BUTTON_CLASS = 'aui-close-button';
    var CLOSE_BUTTON = '<button type="button" class="' + CLOSE_BUTTON_CLASS + '" aria-label="' + closeButtonLabel + '">';

    function flag(options) {
        options = $.extend({}, defaultOptions, options);

        var $flag = renderFlagElement(options);
        extendFlagElement($flag);

        if (options.close === 'auto') {
            makeCloseable($flag);
            makeAutoClosable($flag);
        } else if (options.close === 'manual') {
            makeCloseable($flag);
        }

        pruneFlagContainer();

        return insertFlag($flag, options);
    }

    function extendFlagElement($flag) {
        var flagElement = $flag[0];

        flagElement.close = function() {
            closeFlag($flag);
        };
    }

    function renderFlagElement(options) {
        // confluence-flag.soy template
        // eslint-disable-next-line no-undef
        var template = Confluence.Templates.AuiStaging.flag({
            title: options.title,
            body: options.body,
            type: options.type,
            closeable: options.close !== 'never',
            extraClasses: options.extraClasses
        });

        return $(template);
    }

    function makeCloseable($flag) {
        var $icon = $(CLOSE_BUTTON);

        $icon.on('click', function() {
            closeFlag($flag);
        });

        $icon.on('keypress', function(e) {
            if ((e.which === AJS.keyCode.ENTER) || (e.which === AJS.keyCode.SPACE)) {
                closeFlag($flag);
                e.preventDefault();
            }
        });

        return $flag.find('.aui-message').append($icon)[0];
    }

    function makeAutoClosable($flag) {
        $flag.find('.aui-message').addClass('aui-will-close');
        setTimeout(function() {
            $flag[0].close();
        }, AUTO_CLOSE_TIME);
    }

    function closeFlag($flagToClose) {
        var flagElement = $flagToClose.get(0);

        flagElement.removeAttribute('open');
        flagElement.setAttribute('aria-hidden', 'true');
        var $parent = $flagToClose.parent();
        markTopFlagInStack($parent);
        flagElement.dispatchEvent(new CustomEvent('aui-flag-close', {
            bubbles: true
        }));

        return flagElement;
    }

    function pruneFlagContainer() {
        var $container = findContainer();
        var $allFlags = $container.find('.aui-flag');

        $allFlags.get().forEach(function(flagElement) {
            var isFlagAriaHidden = flagElement.getAttribute('aria-hidden') === 'true';

            if (isFlagAriaHidden) {
                $(flagElement).remove();
            }
        });
    }

    function findContainer() {
        return $('#' + ID_FLAG_CONTAINER);
    }

    function markTopFlagInStack($stack) {
        if ($stack.hasClass('aui-flag-stack') && $stack.children('.aui-flag:not([aria-hidden="true"])').length) {
            $stack.children('.aui-flag-stack-top-item').removeClass('aui-flag-stack-top-item');
            $stack.children('.aui-flag:not([aria-hidden="true"])').last().addClass('aui-flag-stack-top-item');
        }
    }

    function recomputeStyle(el) {
        el = el.length ? el[0] : el;
        window.getComputedStyle(el, null).getPropertyValue('left');
    }

    function insertFlag($flag, options) {
        var $flagContainer = findContainer();

        if (!$flagContainer.length) {
            $flagContainer = $('<div id="' + ID_FLAG_CONTAINER + '"></div>');
            $('body').prepend($flagContainer);
        }

        if (options.stack) {
            var $stackContainer = $('[data-aui-flag-stack="' + options.stack + '"]');
            if (!$stackContainer.length) {
                $stackContainer = $('<div data-aui-flag-stack="' + options.stack + '" class="aui-flag-stack"></div>');
                $stackContainer.appendTo($flagContainer);
            }
            if (!$stackContainer.find('[aria-hidden="false"]').length) {
                $stackContainer.detach().appendTo($flagContainer);
            }

            if (options.fifo) {
                $flag.appendTo($stackContainer);
            } else {
                $flag.prependTo($stackContainer);
            }
            markTopFlagInStack($stackContainer);
        } else {
            $flag.appendTo($flagContainer);
        }
        recomputeStyle($flag);

        var flagElement = $flag.attr('open', '').attr('aria-hidden', 'false')[0];
        flagElement.dispatchEvent(new CustomEvent('aui-flag-show', {
            bubbles: true
        }));

        return flagElement;
    }

    return flag;
});
