/**
 * @module confluence-sortable-tables/sortable-tables
 */
define('confluence-sortable-tables/sortable-tables', [
    'jquery',
    'ajs',
    'document',
    'confluence-sortable-tables/hooks'
], function($,
    AJS,
    document,
    Hooks) {
    'use strict';

    /**
     * We work hard here to keep the cheap and expensive parts separated.
     * We call enable on a delay but keep both methods exposed globally so that we can re-init/enable
     * sortable tables on a page should we modify the dom dramatically.
     */
    var tables;

    /**
     * isTableSorterAlreadyEnabled will return a boolean for the already enabled sortable state of a table
     *
     * @param {HTMLElement} table Html table element
     * @return {Boolean} SortableTables enabled state for table
     */
    var isTableSorterAlreadyEnabled = function(table) {
        return table.className.indexOf('tablesorter') > -1;
    };

    /**
     * isTheadDirectVisibleChild - is thead present as the visible direct child of a table
     *
     * @param {HTMLElement} table Html table element
     * @return {Boolean}
     */
    var isTheadDirectChild = function(table) {
        return $(table).find('> thead').length > 0;
    };

    /**
     * removeDefaultAriaLabels - removes default area labels from <table> element
     *
     * @param {HTMLElement} table Html table element
     */
    var updateDefaultAriaLabels = function(table) {
        var tHead = table.children('thead');
        var tBody = table.children('tbody');
        tHead.children('tr').children('th').each(function() {
            var thElement = $(this);
            thElement.removeAttr('aria-label');
            if (thElement && thElement.attr('aria-sort') === 'none') {
                thElement.removeAttr('aria-sort');
            }
            // a11y hint for sorting button
            thElement.children('div').children('button').attr('aria-roledescription', AJS.I18n.getText('sortableTables.a11y.sortButton'));
        });

        tBody.removeAttr('aria-live');
        tBody.removeAttr('aria-relevant');
    };

    /**
     * addScreenReaderNotification - adds/removes a11y notification block element to the document
     *
     * @param {HTMLElement} targetedElement Html table element
     * @param {String} notificationText for sortEnd event
     */
    var addScreenReaderNotification = function(targetedElement, notificationText) {
        var notificationPlaceholder;
        var timer;
        var notificationAreaTTL = 3000; // 3 sec to give chance for the developer to notice dom changes while debugging.
        var elementId = 'a11y-notification-' + Math.round(Math.random() * 100000000); // eight-digit uniq id
        // Add placeholder for a11y notification
        $('<div class="assistive" aria-live="assertive" role="Alert" id="' + elementId + '"></div>').insertAfter($(targetedElement));
        // Add notification for screen reader
        notificationPlaceholder = document.getElementById(elementId);
        if (notificationPlaceholder) {
            $('<span>').text(notificationText).appendTo(notificationPlaceholder);
            // auto-remove notification placeholder
            timer = setTimeout(function() {
                notificationPlaceholder.remove();
                clearTimeout(timer);
            }, notificationAreaTTL);
        }
    };
    /**
     * announceSortChange - adds a11y notification when table sort happened
     *
     * @param {EventTarget} eventTarget for sortEnd event
     */
    var announceSortChange = function(eventTarget) {
        var targetedTable = eventTarget;
        var configObj;
        var currentSort;
        var columnNum;
        var sortingOrder;
        var columnName;
        if (targetedTable.config && targetedTable.config.sortList) {
            configObj = targetedTable.config;
            currentSort = configObj.sortList;
            // target the first sorted column
            if (Array.isArray(currentSort) && currentSort.length && Array.isArray(currentSort[0]) && currentSort[0].length > 1) {
                columnNum = currentSort[0][0];
                sortingOrder = currentSort[0][1] ? AJS.I18n.getText('sortableTables.a11y.descSort') : AJS.I18n.getText('sortableTables.a11y.ascSort'); // 0 -> ascending , 1 -> descending
                if (Array.isArray(configObj.headerList) && configObj.headerList.length >= columnNum) {
                    columnName = $(configObj.headerList[columnNum]).text();
                    addScreenReaderNotification(targetedTable, AJS.I18n.getText('sortableTables.a11y.changeSortingOrderNotification', columnName, sortingOrder));
                }
            }
        }
    };

    return {
        /**
         * Setup a reference for the sortable table elements
         */
        init: function() {
            var tableSearchResult = $('table');
            tables = tableSearchResult.filter(function() {
                var thisTable = $(this);
                var cells = thisTable.find('td, th');
                var firstRowCells = this.rows.length && $(this.rows[0].cells);
                var cell;
                var eventData;

                // if disabled or already enabled
                if (thisTable.attr('data-sortable') === 'false' || isTableSorterAlreadyEnabled(this)) {
                    return false;
                }

                // this exists for legacy reasons, we use a data attribute to opt out now
                eventData = $.Event('makeSortable.SortableTables');
                thisTable.trigger(eventData);
                if (eventData.isDefaultPrevented()) {
                    return false;
                }

                if (!firstRowCells || firstRowCells.length === 0) { // there are no rows or no cells
                    return false;
                }
                // no colspan or rowspans > 1
                for (var i = 0, len = cells.length; i < len; i++) {
                    cell = cells[i];
                    if (cell.rowSpan != 1 || cell.colSpan != 1) {
                        return false;
                    }
                }

                // Header contains an inner table, not sortable
                if ($(this.rows[0]).find('table').length) {
                    return false;
                }

                // at least one cell in first row is not a th or have a class of nohighlight, not sortable
                if (firstRowCells.filter('th').length !== firstRowCells.length || firstRowCells.hasClass('nohighlight')) {
                    return false;
                }

                // single row tables are not sortable
                return this.rows[1];
            });

            tableSearchResult.bind('sortEnd', function(e, table) {
                var tableElement = $(table);
                // removing default a11y attributes that were added by tablesorter lib after sort update
                updateDefaultAriaLabels(tableElement);
                // get sorted column text
                if (e.target) {
                    announceSortChange(e.target);
                }
            });
        },
        /**
         * Enables sorting for the stored table reference, needs to be called after init has run at least once.
         */
        enable: function() {
            tables.each(function() {
                // If we already enabled tablersorter we should skip
                if (isTableSorterAlreadyEnabled(this) || isTheadDirectChild(this)) {
                    return true;
                }

                // Changing this to use live dom nodes or something other than a document fragment has a
                // significant performance penalty. Modify with care.
                var body = this.removeChild(this.tBodies[0]);
                var rows = $(body.children);
                var firstRow = Array.prototype.shift.call(rows);
                var fragment = document.createDocumentFragment();
                var head = document.createElement('thead');

                fragment.appendChild(head);
                head.appendChild(firstRow);
                fragment.appendChild(body);
                this.appendChild(fragment);
            });

            Hooks.beforeInitHooks().forEach(function(hook) {
                if (hook && typeof hook === 'function') {
                    try {
                        hook($.tablesorter);
                    } catch (e) {
                        AJS.info('Failed to run tablesorter beforeInit hook.', e);
                    }
                }
            });

            $.tablesorter.addParser({
                id: 'european-date',
                is: function(s, table, cell) {
                    var dateElement = cell.querySelector('.content-wrapper time');
                    return dateElement && !dateElement.previousSibling && dateElement.parentNode && !dateElement.parentNode.previousSibling && dateElement.dateTime;
                },
                format: function(s, table, cell) {
                    var dateElement = cell.querySelector('time');
                    return dateElement
                        ? new Date(dateElement.dateTime).getTime()
                        : 0;
                },
                type: 'numeric'
            });

            tables.tablesorter({
                cssHeader: 'sortableHeader',
                delayInit: true,
                textExtraction: function(node) {
                    // Do the text trim manually since our version of jQuery fails at this.
                    return $(node).text().trim();
                },
                tabIndex: false,
                // do something after tablesorter has initialized
                initialized: function(table) {
                    // removing default a11y attributes that were added by tablesorter lib
                    var tableElement = $(table);
                    updateDefaultAriaLabels(tableElement);
                },
                headerTemplate: '<button class="headerButton">{content}</button>',
                cssIcon: '',
            });
        },
        /**
         * Allows re-init/enable sortable tables when the DOM has changed
         */
        refresh: function() {
            this.init();
            this.enable();
        }
    };
});

require('confluence/module-exporter').safeRequire('confluence-sortable-tables/sortable-tables', function(SortableTables) {
    'use strict';

    require('ajs').toInit(function() {
        SortableTables.init();
        setTimeout(SortableTables.enable, 100);
    });
});
