define('confluence/share-page/popup/generate-popup', [
    'ajs',
    'jquery',
    'confluence/legacy',
    'confluence/share-page/autocomplete/autocomplete-user',
    'confluence/share-page/autocomplete/setup-autocomplete',
    'confluence/share-page/popup/setup-share-link',
    'confluence/share-page/form/submit',
    'confluence/share-page/service/analytics-service',
    'confluence/share-page/util/show-message',
    'confluence/share-page/util/find-recipients',
    'confluence/share-page/popup/setup-restriction-warning'
], function (AJS,
             $,
             Confluence,
             autocompleteUser,
             setupAutocomplete,
             setupShareLink,
             submit,
             analyticsService,
             showMessage,
             findRecipients,
             setupRestrictionWarning) {
    // Claim this value staticly so that multiple usages of generatePopup will still work correctly.
    var MAIL_SERVER_CONFIGURED = WRM.data.claim("com.atlassian.confluence.plugins.share-page:_private_share-page-resources.mail-server-configured");
    var WHEEL_EVENT = 'wheel.share-page';
    var KEYUP_EVENT = 'keyup.share-page';

    /**
     * Confines scrolling in the recipients area when it overflows and becomes scrollable.
     * Because the share dialog disappears as you scroll down the page, we confine scrolling
     * inside the recipients &lt;ul&gt; so that it doesn't escape the ul and propagate to the window.
     * Instead, it scrolls only the recipients div and doesn't propagate.
     * @param $contents
     * @private
     */
    function _confineRecipientsScrolling($contents) {
        $contents.find('.recipients')
            .off(WHEEL_EVENT)
            .on(WHEEL_EVENT, function (e) {
                var $this = $(this);
                // Stop the scroll from propagating to the window which causes the share dialog to disappear.
                // Only do it if the element currently needs to be scrolled.
                if ($this.prop('scrollHeight') > $this.innerHeight()) {
                    $this.scrollTop($this.scrollTop() + e.originalEvent.deltaY);
                    e.preventDefault();
                    e.stopPropagation();
                    return false;
                }
            });
    }

    /**
     * Bind editor click and keydown events for closing the invite to edit dialog when a user
     * clicks in the editor while the dialog is open.ø
     * @param {string} onOrOff Only 'on' to bind the events, and 'off' to unbind them.
     * @param {object} parameters Passed down from the view-init.js or edit-init.js
     * @param {function} handler
     * @private
     */
    function _bindEditorEvents(onOrOff, parameters, handler) {
        if (parameters.shareType !== 'edit') {
            return;
        }
        var editor = AJS.Rte.getEditor();
        try {
            editor[onOrOff]('Click', handler);
            editor[onOrOff]('KeyUp', handler);
        } catch (e) {
            AJS.warn('Unknown bind method:', onOrOff, 'for binding to editor events.',
                'Only "on" or "off" are expected.');
        }
    }

    /**
     * Bind handlers for when layers are added or removed from the document.
     * @param trigger
     * @param parameters
     * @param {function} doHidePopup
     * @private
     */
    function _layers(trigger, parameters, doHidePopup) {
        $(document).bind("showLayer", function (e, type, dialog) {
            if (type === "inlineDialog" && dialog.popup === $(trigger).data('dialog')) {
                dialog.popup.find("#share-popup-header").focus();
                _bindEditorEvents('on', parameters, doHidePopup);
            }
        }).bind("hideLayer", function (e, type, dialog) {
            if (type === "inlineDialog" && dialog.popup === $(trigger).data('dialog')) {
                _bindEditorEvents('off', parameters, doHidePopup);
            }
        });
    }

    /**
     * Binds submit actions to the dialog for when the submit button is clicked.
     * @param $contents
     * @param trigger
     * @param parameters
     * @param {function} doHidePopup
     * @private
     */
    function _submitHandler($contents, trigger, parameters, doHidePopup) {
        var $submitButton = $contents.find(".submit.button");

        function _submitClick() {
            if (findRecipients.findAllRecipients($contents).length === 0) {
                analyticsService.publish(analyticsService.SHARE_NO_USERS_SELECTED, parameters.entityId, parameters.shareType);
                showMessage($contents.find('.autocomplete-user-target'), 'recipientSelectionError', parameters, true, function () {
                    $submitButton.off('click', _submitClick);
                }, function () {
                    $submitButton.click(_submitClick);
                });
                $(this).blur();
                $contents.find('#share-invite-users-input').addClass('field-error').focus();
                return false;
            }
        }

        $submitButton.click(_submitClick);

        $contents.find("form").submit(function () {
            return submit($contents, trigger, doHidePopup, parameters);
        });
    }

    /**
     * Binds keypress handlers for the document, such as the Escape key to trigger the document to close.
     * @param {function} doHidePopup
     * @private
     */
    function _keypressHandler(doHidePopup) {
        $(document).off(KEYUP_EVENT).on(KEYUP_EVENT, function (e) {
            if (e.keyCode == 27) { // escape
                doHidePopup(true);
                $(document).unbind(KEYUP_EVENT, arguments.callee);
                return false;
            }
            return true;
        });
    }

    /**
     * Binds the handler for when the cancel button is clicked.
     * @param $contents
     * @param parameters
     * @param {function} doHidePopup
     * @private
     */
    function _closeHandler($contents, parameters, doHidePopup) {
        $contents.find(".close-dialog").click(function () {
            analyticsService.publish(analyticsService.SHARE_CANCEL_CLICKED, parameters.entityId, parameters.shareType);
            return doHidePopup(true)
        });
    }

    /**
     * Scroll to the top of the page so the user can see the dialog
     * We need to walk up the tree because the scrollable area is not always document.window
     * but can be other elements.
     * @param trigger
     * @private
     */
    function _scrollToTop(trigger) {
        $(trigger).parents().filter(function () {
            return this.scrollTop > 0;
        }).scrollTop(0);
    }

    /**
     * Generates the share popup if it hasn't been generated yet.
     * @param $contents of the dialog
     * @param trigger DOM element that is the trigger for this dialog
     * @param doShowPopup function that runs after the popup has been generated
     * @returns {undefined}
     */
    return function generatePopup($contents, trigger, doShowPopup) {
        var parameters = this.parameters;
        analyticsService.publish(analyticsService.SHARE_STARTED, parameters.entityId, parameters.shareType);
        if ($contents.find("input").length) {
            doShowPopup();
            parameters.onShow && typeof parameters.onShow === 'function' && parameters.onShow($contents);
            return;
        }

        var submitButtonText = parameters.copyOption === 'share'
            ? AJS.I18n.getText('share.dialog.form.share')
            : AJS.I18n.getText('share.invite.to.edit.invite');

        var peopleHeading = parameters.copyOption === 'share'
            ? AJS.I18n.getText('share.dialog.add.people')
            : AJS.I18n.getText('share.invite.to.edit.invite.people');

        // load template
        $contents.append(Confluence.Templates.Share.Dialog.shareContentPopup({
            heading: parameters.heading,
            peopleHeading: peopleHeading,
            submitButtonText: submitButtonText,
            mailServerConfigured: MAIL_SERVER_CONFIGURED,
            notePlaceholder: parameters.notePlaceholder
        }));

        // hide popup hooks
        var doHidePopup = function (reset) {
            parameters.onHide && typeof parameters.onHide === 'function' && parameters.onHide();
            $(trigger).data('dialog').hide();
            if (reset) {
                // hiding the dialog is animated. we want to wait until the animation is over before
                // we clear the content of the dialog
                setTimeout(function () {
                    $contents.empty();
                }, 300)
            }
            return false;
        };

        autocompleteUser();
        setupAutocomplete($contents, trigger, parameters);
        setupShareLink($contents, parameters);
        setupRestrictionWarning($contents, parameters);

        _confineRecipientsScrolling($contents);
        _keypressHandler(doHidePopup);
        _submitHandler($contents, trigger, parameters, doHidePopup);
        _closeHandler($contents, parameters, doHidePopup);
        _layers(trigger, parameters, doHidePopup);
        _scrollToTop(trigger);

        doShowPopup();
        parameters.onShow && typeof parameters.onShow === 'function' && parameters.onShow($contents);
    }
});
