/**
 * @module confluence-page-restrictions-dialog/dialog-model
 */
define('confluence-page-restrictions-dialog/dialog-model', [
    'ajs',
    'backbone',
    'underscore',
    'confluence-page-restrictions-dialog/explicit-restrictions',
    'confluence-page-restrictions-dialog/inherited-restrictions'
],
function(
    AJS,
    Backbone,
    _,
    ExplicitRestrictions,
    InheritedRestrictions
) {
    'use strict';

    // Given a set of restrictions from the server determine the "mode" the dialog is in.
    function detectMode(data, contentId) {
        if (dataContainsExplicitRestriction(data.permissions, 'View', contentId)) {
            return 'viewedit';
        }

        if (dataContainsExplicitRestriction(data.permissions, 'Edit', contentId)) {
            return 'edit';
        }

        return 'none';
    }

    function dataContainsExplicitRestriction(permissions, type, contentId) {
        return _.some(permissions, function(permission) {
            var permissionType = permission[0];
            var permissionOwningContentId = permission[3];
            return permissionType === type && permissionOwningContentId === contentId;
        });
    }

    var DialogModel = Backbone.Model.extend({
        initialize: function(options) {
            this.set('content-id', options.contentId);
            this.set('parent-page-id', options.parentPageId);
            this.set('space-key', options.spaceKey);
            this.set('context-path', options.contextPath);
            this.set('explicit-restrictions', new ExplicitRestrictions());
            this.set('inherited-restrictions', new InheritedRestrictions());
        },

        fetch: function() {
            var deferred = AJS.safe.ajax({
                url: this.get('context-path') + '/pages/getcontentpermissions.action',
                data: {
                    contentId: this.get('content-id'),
                    parentPageId: this.get('parent-page-id'),
                    spaceKey: this.get('space-key')
                },
                contentType: 'application/json',
                dataType: 'json'
            });

            deferred.done(function(data) {
                this.get('explicit-restrictions').populateWithData(data, this.get('content-id'));
                this.get('inherited-restrictions').populateWithData(data, this.get('content-id'));
                this.set('view-only', !data.userCanEditRestrictions);
                this.set('mode', detectMode(data, this.get('content-id')));
                this.set('original-mode', this.get('mode')); // This is purely for analytics purposes.
                this.set('get-permissions-req-error', data.errorMessage);
            }.bind(this));

            return deferred;
        },

        save: function() {
            var permissions = this.get('explicit-restrictions').formatForSubmission(this.get('mode'));

            permissions.contentId = this.get('content-id');

            var deferred = AJS.safe.post(
                this.get('context-path') + '/pages/setcontentpermissions.action',
                permissions,
                null,
                'json'
            );

            return deferred;
        },

        populateWithCurrentUser: function() {
            if (this.hasExplicitRestrictions()) {
                return;
            }

            if (!this.get('view-only')) {
                // We do this even if the mode is "none" to avoid reactively adding it later.
                this.get('explicit-restrictions').populateWithCurrentUser();
            }
        },

        hasExplicitRestrictions: function() {
            return this.get('mode') !== 'none';
        },

        hasExplicitViewRestrictions: function() {
            return this.get('mode') === 'viewedit';
        },

        hasInheritedViewRestrictions: function() {
            return !this.get('inherited-restrictions').isEmpty();
        }
    });

    return DialogModel;
});
