/**
 * @module confluence-page-restrictions-dialog/dialog-view
 */
define('confluence-page-restrictions-dialog/dialog-view', [
    'ajs',
    'jquery',
    'backbone',
    'confluence-page-restrictions-dialog/table-view',
    'confluence-page-restrictions-dialog/inherited-view',
    'confluence/legacy',
    'confluence/templates',
    'confluence/form-state-control'
],
function(
    AJS,
    $,
    Backbone,
    TableView,
    InheritedView,
    Confluence,
    Templates,
    FormStateControl
) {
    'use strict';

    function isEditMode() {
        // This dialog can also be accessed from the view page, this function asserts whether or not the editor UI is present.
        return $('#rte-button-restrictions').parent().is(':visible');
    }

    var modeSelector;

    var DialogView = Backbone.View.extend({
        events: {
            'click #page-restrictions-dialog-save-button': 'save',
            'click #page-restrictions-dialog-close-button': 'closeButtonHandler',
            'click .dialog-help-link': 'triggerHelpAnalyticsEvent',
            'click .call-to-action': 'triggerLearnAboutAnalyticsEvent'
        },

        initialize: function(options) {
            var hasInherited = !this.model.get('inherited-restrictions').isEmpty();

            this.modes = {
                none: {
                    title: AJS.I18n.getText('option.none'),
                    description: hasInherited
                        ? '' // There is no description as we are just using the inherited reminder.
                        : AJS.I18n.getText('option.none.description'),
                    byline: AJS.I18n.getText('option.none.byline'),
                    classes: 'aui-iconfont-unlocked'
                        + (hasInherited ? ' red' : '')
                },
                edit: {
                    title: AJS.I18n.getText('option.edit'),
                    description: hasInherited
                        ? AJS.I18n.getText('option.edit.inherited.description')
                        : AJS.I18n.getText('option.edit.description'),
                    byline: AJS.I18n.getText('option.edit.byline'),
                    classes: 'aui-iconfont-unlocked'
                    + (hasInherited ? ' red' : '')
                },
                viewedit: {
                    title: AJS.I18n.getText('option.viewedit'),
                    description: hasInherited
                        ? AJS.I18n.getText('option.viewedit.inherited.description')
                        : AJS.I18n.getText('option.viewedit.description'),
                    byline: AJS.I18n.getText('option.viewedit.byline'),
                    classes: 'aui-iconfont-locked red'
                }
            };

            this.dialog2 = options.dialog2;
            this.applyButton = this.$('#page-restrictions-dialog-save-button');
            this.spinner = this.$('.button-spinner');

            this.disableApply();

            this.inlineDialog2 = document.querySelector('#inherited-restrictions-popup');

            this.initializeModeSelector();

            // If the mode is currently no-restrictions, we do not need to listen to the collection.
            if (this.model.get('mode') !== 'none') {
                this.listenTo(this.model.get('explicit-restrictions'), 'add remove', this.enableApply);
            }

            this.model.populateWithCurrentUser();

            // We always want to listen for a change in the mode to enable the save button.
            this.listenTo(this.model, 'change:mode', this.enableApply);

            this.tableView = new TableView({
                model: this.model,
                el: this.$('.page-restrictions-dialog-entities-container')
            });

            // Hide the inline dialog whenever select2 is interacted with.
            $('#page-restrictions-dialog-selector').on('select2-opening', this.hideInlineDialog.bind(this));
            $('#restrictions-dialog-auto-picker')
                .on('select2-opening', this.hideInlineDialog.bind(this))
                .on('select2-opening', this.triggerSearchAnalyticsEvent)
                .on('select2-selecting', this.enableApply.bind(this));

            new InheritedView({
                collection: this.model.get('inherited-restrictions'),
                el: this.$('#inherited-restrictions-popup .inherited-restrictions-contents')
            });

            this.render();

            // We bind the re-render here, to avoid unnecessary rendering beforehand.
            this.listenTo(this.model, 'change:mode', this.render);
        },

        render: function() {
            this.hideInlineDialog();

            var mode = this.model.get('mode');
            var hasInheritedRestrictions = this.model.hasInheritedViewRestrictions();

            var $entityContainer = this.$('.page-restrictions-dialog-entities-container');
            var $emptyPlaceholder = this.$('.page-restrictions-dialog-empty-placeholder');
            var $inheritedReminder = this.$('.inherited-restrictions-explanation');

            modeSelector.select2('val', mode);

            this.$('.page-restrictions-dialog-explanation').html(this.modes[mode].description);

            var hasExplicitRestrictions = this.model.hasExplicitRestrictions();

            $entityContainer.toggle(hasExplicitRestrictions);
            $emptyPlaceholder.toggle(!hasExplicitRestrictions);

            $inheritedReminder.toggle(hasInheritedRestrictions);

            if (!hasExplicitRestrictions) {
                $emptyPlaceholder.find('.default').toggle(!hasInheritedRestrictions);
                $emptyPlaceholder.find('.inherited').toggle(hasInheritedRestrictions);
            }
        },

        enableApply: function() {
            FormStateControl.enableElement(this.applyButton);
            this.spinner.spinStop();
        },

        disableApply: function(spin) {
            FormStateControl.disableElement(this.applyButton);
            if (spin) {
                this.spinner.spin();
            }
        },

        hideInlineDialog: function() {
            // Clean up any existing, skated dialogs.
            if (this.inlineDialog2.isVisible && this.inlineDialog2.isVisible()) {
                this.inlineDialog2.hide();
            }
        },

        // Initialize select2.
        initializeModeSelector: function() {
            modeSelector = this.$('#page-restrictions-dialog-selector').auiSelect2({
                minimumResultsForSearch: -1, // Hide the search box.
                formatSelection: (this.formatSelection).bind(this),
                formatResult: (this.formatSelection).bind(this),
                dropdownCssClass: 'page-restrictions-dialog-drop-down'
            });

            var model = this.model;

            if (model.get('get-permissions-req-error')) {
                this.showErrorMessage(AJS.I18n.getText('restrictions.dialog.load.error'));
                FormStateControl.disableElement(modeSelector);
            } else if (model.get('view-only')) {
                this.showInfoMessage(AJS.I18n.getText('restrictions.dialog.noedit'));
                FormStateControl.disableElement(modeSelector);
            } else {
                FormStateControl.enableElement(modeSelector);
                modeSelector.on('change', function() {
                    model.set('mode', modeSelector.val());
                });
            }
        },

        formatSelection: function(selection) {
            var mode = this.modes[selection.id];

            return Templates.Restrictions.restrictionsDropDownItem({
                classes: mode.classes,
                title: mode.title,
                byline: mode.byline
            });
        },

        showInfoMessage: function(message) {
            this.showMessage(message, 'info');
        },

        showErrorMessage: function(message) {
            this.showMessage(message, 'warning');
        },

        showMessage: function(message, type) {
            var errorDivId = 'page-restrictions-dialog-error-div';
            this.$('#' + errorDivId).remove();

            AJS.messages[type]('#update-page-restrictions-dialog .aui-dialog2-content', {
                title: message,
                id: errorDivId,
                insert: 'prepend'
            });
        },

        triggerRestrictionsUpdateEvent: function(restrictionsHash) {
            if (isEditMode()) {
                AJS.trigger('edit-page-restrictions-updated', {
                    hasExplicitRestrictions: this.model.hasExplicitViewRestrictions(),
                    hasInheritedRestrictions: this.model.hasInheritedViewRestrictions(),
                    hasAnyExplicitRestrictions: this.model.hasExplicitRestrictions(),
                    restrictionsHash: restrictionsHash
                });
            } else {
                AJS.trigger('system-content-metadata.toggled-restrictions', {
                    hasExplicitRestrictions: this.model.hasExplicitViewRestrictions(),
                    hasInheritedRestrictions: this.model.hasInheritedViewRestrictions(),
                    hasAnyExplicitRestrictions: this.model.hasExplicitRestrictions()
                });
            }
        },

        save: function() {
            if (this.applyButton.attr('aria-disabled') === 'true') { return; }

            this.disableApply(true);

            // We need to commit all the pending values in the Select2 field.
            this.tableView.processSelectedEntities();

            var dialogView = this;
            var deferred = this.model.save();

            deferred.done(function(data) {
                dialogView.spinner.spinStop();
                dialogView.triggerRestrictionsUpdateEvent(data.restrictionsHash);
                dialogView.close();
                dialogView.triggerSaveAnalyticsEvent();
            });

            deferred.fail(function() {
                dialogView.showErrorMessage(AJS.I18n.getText('restrictions.dialog.submitting.error'));
                dialogView.enableApply();
            });
        },

        notifyUserOfExternalChange: function() {
            this.showErrorMessage(AJS.I18n.getText('restrictions.dialog.reload'));
        },

        close: function() {
            this.inlineDialog2.open = false;
            setTimeout(function() { // Here we use a timeout to avoid removing the dialog
                this.dialog2.remove(); // from the DOM before raphael removes the fade screen.
            }.bind(this), 0);
        },

        closeButtonHandler: function() {
            this.close();
            Confluence.Analytics.publish('confluence.restrictions.dialog.cancel');
        },

        triggerHelpAnalyticsEvent: function() {
            Confluence.Analytics.publish('confluence.restrictions.dialog.help');
        },

        triggerLearnAboutAnalyticsEvent: function() {
            Confluence.Analytics.publish('confluence.restrictions.dialog.learnaboutrestrictions');
        },

        triggerSearchAnalyticsEvent: function() {
            Confluence.Analytics.publish('confluence.restrictions.dialog.search');
        },

        triggerSaveAnalyticsEvent: function() {
            var mode = this.model.get('mode');
            var original = this.model.get('original-mode');

            Confluence.Analytics.publish('confluence.restrictions.dialog.saved.' + mode);

            if (original !== mode) {
                Confluence.Analytics.publish('confluence.restrictions.dialog.' + original + '.changed.' + mode);
            }
        }
    });

    return DialogView;
});
