/**
 * @module confluence-page-restrictions-dialog/explicit-restrictions
 */
define('confluence-page-restrictions-dialog/explicit-restrictions', [
    'ajs',
    'underscore',
    'confluence-page-restrictions-dialog/restriction-model',
    'confluence/meta',
    'backbone'
],
function(
    AJS,
    _,
    RestrictionModel,
    Meta,
    Backbone
) {
    'use strict';

    var ExplicitRestrictions = Backbone.Collection.extend({
        model: RestrictionModel,

        // Load user and group data from the server supplied JSON into a collection of User and Group models.
        populateWithData: function(data, contentId) {
            if (!data) { return; }

            // Empty the collection before inserting a new set of data.
            this.reset();

            var explicitRestrictions = this;

            _.each(data.permissions, function(permission) {
                var permissionType = permission[0].toLowerCase();
                var entityType = permission[1];
                var entityName = permission[2];
                var owningContentId = permission[3];
                var isUserEntity = entityType === 'user';

                var entity = isUserEntity ? data.users[entityName].entity : data.groups[entityName].entity;

                if (+owningContentId <= 0) {
                    // We should never encounter an non-positive owningContentId.
                    AJS.log('Encountered restriction with non-positive owningContentId: ' + entityName
                    + ', ' + entityType + ' has owningContentId = ' + owningContentId + '.');
                    return;
                }

                // Here we only add restrictions relevant to the current content object.
                if (owningContentId === contentId) {
                    explicitRestrictions.addEntity(entity, permissionType, false);
                }
            });

            // We need to ensure to upgrade view only to view-and-edit if there are no users with edit restrictions.
            if (this.size() > 0 && this.detectLegacyViewOnlyState(data)) {
                this.each(function(entity) {
                    entity.changeEntityPermission('edit');
                });
            }
        },

        detectLegacyViewOnlyState: function(data) {
            // If all of the permissions in the provided data imply only "View" restrictions exist,
            //  we have successfully detected the legacy view-restriction only state.
            return _.every(data.permissions, function(permission) { return permission[0] === 'View'; });
        },

        addRawModel: function(rawModel) {
            var existingEntity = this.get(rawModel.get('id'));

            // Here we specifically want to ensure that permissions with "edit" are always added in place of those with "view".
            if (rawModel.get('permission') === 'edit' && (existingEntity && existingEntity.get('permission') !== 'edit')) {
                this.remove(rawModel.get('id'));
            }

            // Adding a model to the collection that already exists is a no-op, so no special checks are required here.
            this.add(rawModel);
        },

        addEntity: function(entity, permission, isClientModel) {
            var entityId = entity.type === 'user' ? entity.userKey : entity.name;

            this.addRawModel(new RestrictionModel({
                id: entityId,
                entity: entity,
                permission: permission,
                isClientModel: isClientModel
            }));
        },

        removeEntity: function(entityId) {
            this.remove(entityId);
        },

        populateWithCurrentUser: function() {
            this.addEntity(
                {
                    username: Meta.get('remote-user'),
                    title: Meta.get('current-user-fullname'),
                    thumbnailLink: {
                        href: Meta.get('current-user-avatar-uri-reference')
                    },
                    type: 'user',
                    userKey: Meta.get('remote-user-key')
                }, 'edit', true);
        },

        // Filter the entities we want to display for the current dialog mode.
        filterForMode: function(mode) {
            var explicitRestrictions = this.clone();

            if (mode === 'edit') {
                explicitRestrictions.models = explicitRestrictions.filter(function(restriction) {
                    return restriction.get('permission') === 'edit';
                });
            }

            return explicitRestrictions;
        },

        // Sort the entities so that entities that cannot be changed are at the top
        sortByCanRemove: function() {
            this.models = this.sortBy(function(restriction) {
                return restriction.get('canRemove') ? 1 : 0;
            });

            return this;
        },

        formatForSubmission: function(mode) {
            function iteratee(memo, entity) {
                if (hasViewRestrictions) {
                    if (entity.isUser()) {
                        memo.viewPermissionsUserList.push(entity.get('id'));
                    } else {
                        memo.viewPermissionsGroupList.push(entity.get('id'));
                    }
                }

                if (entity.get('permission') === 'edit') {
                    if (entity.isUser()) {
                        memo.editPermissionsUserList.push(entity.get('id'));
                    } else {
                        memo.editPermissionsGroupList.push(entity.get('id'));
                    }
                }

                return memo;
            }

            var result = {
                viewPermissionsUserList: [],
                editPermissionsUserList: [],
                viewPermissionsGroupList: [],
                editPermissionsGroupList: []
            };

            var hasViewRestrictions = mode === 'viewedit';

            if (mode !== 'none') {
                result = this.reduce(iteratee, result);
            }

            return result;
        }
    });

    return ExplicitRestrictions;
});
