/**
 * @module confluence-page-restrictions-dialog/inherited-restrictions
 */
define('confluence-page-restrictions-dialog/inherited-restrictions', [
    'ajs',
    'underscore',
    'confluence-page-restrictions-dialog/restriction-model',
    'backbone'
],
function(
    AJS,
    _,
    RestrictionModel,
    Backbone
) {
    'use strict';

    var InheritedRestrictions = Backbone.Collection.extend({
        model: RestrictionModel,

        // Load user and group data from the server supplied JSON into a collection of User and Group models.
        populateWithData: function(data, contentId) {
            if (!data) { return; }

            // Empty the collection before inserting a new set of data.
            this.reset();

            var inheritedRestrictions = this;

            _.each(data.permissions, function(permission) {
                var entityType = permission[1];
                var entityName = permission[2];
                var owningContentId = permission[3];
                var owningContentTitle = permission[4];
                var isUserEntity = entityType === 'user';

                var entity = isUserEntity ? data.users[entityName].entity : data.groups[entityName].entity;

                if (+owningContentId <= 0) {
                    // We should never encounter an non-positive owningContentId.
                    AJS.log('Encountered restriction with non-positive owningContentId: ' + entityName
                    + ', ' + entityType + ' has owningContentId = ' + owningContentId + '.');
                    return;
                }

                // Here we only add restrictions that have been inherited, and do not share a contentId with the current object.
                if (owningContentId !== contentId) {
                    var existingEntity = inheritedRestrictions.get(isUserEntity ? entity.userKey : entity.name);

                    if (existingEntity === undefined) {
                        inheritedRestrictions.add(new RestrictionModel({
                            id: (entity.type === 'user' ? entity.userKey : entity.name) + owningContentId,
                            entity: entity,
                            permission: 'view', // Inherited restrictions can only be view restrictions.
                            owningContentId: owningContentId,
                            owningContentTitle: owningContentTitle,
                            isClientModel: false
                        }));
                    }
                }
            });
        },

        formatForPresentation: function() {
            var inheritedRestrictions = this.toJSON();
            return _.groupBy(inheritedRestrictions, 'owningContentId');
        }
    });

    return InheritedRestrictions;
});
