/**
 * @module confluence-page-restrictions-dialog/table-view
 */
define('confluence-page-restrictions-dialog/table-view', [
    'jquery',
    'backbone',
    'confluence-ui-components/js/user-group-select2',
    'confluence/templates',
    'confluence/legacy',
    'underscore',
    'confluence/meta',
    'ajs'
],
function(
    $,
    Backbone,
    UserGroupSelect2,
    Templates,
    Confluence,
    _,
    Meta,
    AJS
) {
    'use strict';

    var TableView = Backbone.View.extend({
        events: {
            'click .restrictions-dialog-entity-remove': 'removeEntity',
            'change .restrictions-dialog-entity-permission': 'restrictionChanged',
            'click #page-restrictions-add-button': 'processSelectedEntitiesEvent'
        },

        initialize: function() {
            this.collection = this.model.get('explicit-restrictions');

            this.$el.html(Templates.Restrictions.restrictionsTable({
                viewOnly: this.model.get('view-only')
            }));

            this.listenTo(this.model, 'change:mode', this.render);
            this.listenTo(this.collection, 'add remove', this.render);
            this.listenTo(this.model, 'change:mode', this.renderDropdown);

            this.renderDropdown();
            UserGroupSelect2.bind('#update-page-restrictions-dialog', {
                searchLabel: AJS.I18n.getText('restrictions.dialog.entity.search.placeholder'),
            });
            this.userGroupSelector = this.$('#s2id_restrictions-dialog-auto-picker');
            this.render();
        },

        render: function() {
            // Avoid rendering the table view for no restrictions.
            if (this.model.get('mode') === 'none') {
                return;
            }

            var entities = this.collection;

            entities = entities.filterForMode(this.model.get('mode'));
            entities = entities.sortByCanRemove();

            $('.restrictions-dialog-table .entity-row').remove();

            this.$el.find('.restrictions-dialog-table tbody').append(Templates.Restrictions.entityList({
                entities: entities.toJSON(),
                viewOnly: this.model.get('view-only'),
                mode: this.model.get('mode'),
                groupAvatarUrl: Meta.get('context-path') + '/images/icons/avatar_group_48.png'
            }));
        },

        renderDropdown: function() {
            // Avoid rendering the renderDropdown for no restrictions.
            if (this.model.get('mode') === 'none') {
                return;
            }

            this.$el.find('#restrictions-dialog-permission-picker').html(Templates.Restrictions.addPermissionPicker({
                viewOnly: this.model.get('view-only'),
                mode: this.model.get('mode')
            }));
            $('#page-restrictions-permission-selector').auiSelect2({
                minimumResultsForSearch: -1
            });
        },

        getSelectedPermission: function() {
            if (this.model.get('mode') === 'edit') {
                return 'edit';
            }

            return $('#page-restrictions-permission-selector').val();
        },

        processSelectedEntitiesEvent: function(event) {
            if ($(event.target).attr('aria-disabled') === 'true') {
                return;
            }

            if (this.processSelectedEntities()) {
                Confluence.Analytics.publish('confluence.restrictions.dialog.add');
            }
        },

        processSelectedEntities: function() {
            var userGroupData = this.userGroupSelector.select2('data');
            var selectedPermission = this.getSelectedPermission();
            var tableView = this;

            if (userGroupData.length > 0) {
                _.each(userGroupData, function(data) {
                    tableView.addEntity(data.entity, selectedPermission);
                });
            }

            // Here we empty the select2 field by setting the selected values to "".
            this.userGroupSelector.select2('val', '');

            return userGroupData.length > 0;
        },

        removeEntity: function(event) {
            // Check if 'button' is diabled.
            if (this.model.get('view-only')) {
                return;
            }

            var elementId = $(event.target).attr('data-entity-id');
            this.collection.removeEntity(elementId);
            Confluence.Analytics.publish('confluence.restrictions.dialog.remove');
        },

        addEntity: function(entity, permission) {
            this.collection.addEntity(entity, permission, true);
        },

        restrictionChanged: function(event) {
            var permission = $(event.target).val();
            var entityId = $(event.target).attr('data-entity-id');

            this.collection.get(entityId).changeEntityPermission(permission);

            // Here we trigger "add" as if a member of the collection had been directly set.
            this.collection.trigger('add');
        }
    });

    return TableView;
});
