/**
 * Represents the destination panel containing the Link location (internal or external) and the link Alias.
 *
 * @module confluence-link-browser/link-browser-location
 */
define('confluence-link-browser/link-browser-location', [
    'jquery',
    'confluence/legacy',
    'ajs',
    'confluence/dialog-breadcrumbs'
], function($,
    Confluence,
    AJS,
    DialogBreadcrumbs) {
    'use strict';

    return function(lbController) {
        var OPEN_IN_NEW_WINDOW_CHECKED_EVENT_NAME = 'editor.linkBrowser.location.openInNewWindow.checked';
        var OPEN_IN_NEW_WINDOW_UNCHECKED_EVENT_NAME = 'editor.linkBrowser.location.openInNewWindow.unchecked';
        var container; // the top-level container holding the location elements
        var breadcrumbsEl; // the element containing any Breadcrumbs
        var breadcrumbController; // the controller for updating and rendering Breadcrumbs
        var currentLink; // the link object currently selected or entered
        var defaultAliasClass; // class that indicates the default alias has been used
        var linkOpenInNewWindowCheckbox;
        var linkTextField;
        var linkTextRow;
        var linkImageRow;
        var linkMixedRow;
        var linkMixedContent;
        var linkImageName;
        var finalBody; // the body of the link when an image or range is selected - it cannot be changed
        var movedPanel; // when panels are moved, stores the panel that was moved

        container = $(Confluence.Templates.LinkBrowser.locationPanel());
        breadcrumbsEl = container.find('#breadcrumbs-container');
        breadcrumbController = DialogBreadcrumbs.Breadcrumbs(breadcrumbsEl, DialogBreadcrumbs.getBreadcrumbsLegacy);
        defaultAliasClass = 'default-alias';
        linkOpenInNewWindowCheckbox = container.find('#link-open-in-new-window');
        linkImageRow = container.find('.link-image');
        linkMixedRow = container.find('.link-mixed');
        linkImageName = container.find('#link-image-filename');
        linkMixedContent = container.find('#link-mixed-content');
        linkTextRow = container.find('.link-text');
        linkTextField = linkTextRow.find('input');

        // If the user types in the link text field, we remove the default alias class to indicate this.
        linkTextField.change(function(e) {
            e.keyCode = e.keyCode || e.which;
            if (e.keyCode && e.keyCode !== 13) {
                linkTextField.removeClass(defaultAliasClass);
            }
        });

        linkOpenInNewWindowCheckbox.change(function() {
            if (linkOpenInNewWindowCheckbox.prop('checked')) {
                if (currentLink.getResourceType() !== 'attachment') {
                    currentLink.setTarget('_blank');
                }
                _triggerAnalyticsEvent(OPEN_IN_NEW_WINDOW_CHECKED_EVENT_NAME, currentLink);
            } else {
                currentLink.removeTarget();
                _triggerAnalyticsEvent(OPEN_IN_NEW_WINDOW_UNCHECKED_EVENT_NAME, currentLink);
            }
        });

        /**
         * Returns the text in the link text field.
         */
        function getLinkText() {
            return AJS.escapeHtml(getRawLinkText());
        }

        function getRawLinkText() {
            return linkTextField.val();
        }

        /**
         * Sets the link text in the link text field only if the field is empty
         * or has the default alias class (signifying that the text currently in the field
         * was generated from the link details and is not user-specified).
         * @param text The text to display in the link text field
         */
        function setLinkText(text) {
            if ((getLinkText() === '') || linkTextField.hasClass(defaultAliasClass)) {
                linkTextField.addClass(defaultAliasClass);
                linkTextField.val(text);
            }
        }

        /**
         * Sets the link text from a link object. Uses the data-linked-resource-default-alias if
         * available, otherwise defaults to the body html.
         */
        function setLinkTextFromLink(linkObj) {
            var alias = linkObj.attrs['data-linked-resource-default-alias'] || linkObj.getHtml();
            setLinkText(alias);
        }

        /**
         * Returns the name of the image being linked.
         */
        function getLinkImageName() {
            return linkImageName.text();
        }

        /**
         * Updates the link body fields to reflect current link body information. Sets and toggles
         * the link text, link image or link mixed content as appropriate.
         *
         * @param body - a body object
         */
        function setLinkBody(body) {
            // Don't modify the link body if the original selection should persist.
            if (finalBody) {
                return;
            }

            if (body.isEditable) {
                linkTextField.val(body.text);
            } else if (body.isImage) {
                linkImageName.text(body.imgName);
            } else {
                linkMixedContent.text(body.text);
            }

            finalBody = body.isEditable ? null : body;

            linkTextRow.toggleClass('hidden', !body.isEditable);
            linkImageRow.toggleClass('hidden', !body.isImage);
            linkMixedRow.toggleClass('hidden', body.isEditable || body.isImage);
        }

        /**
         * If true, show the Breadcrumbs element, else hide it. After the element is shown/hidden, resize the location
         * container by setting a CSS class.
         *
         * @param show true if breadcrumbs should be displayed
         */
        function showBreadcrumbsElement(show) {
            breadcrumbsEl.closest('.row').toggleClass('hidden', !show);
            container.toggleClass('has-breadcrumbs', !!show); // must pass a boolean to toggleClass to force!
        }

        /**
         * Show or hide the Open In New Tab Checkbox row
         * @param show Boolean true if the checkbox should be shown, false to hide it.
         */
        function showOpenInNewWindowCheckbox(show) {
            var $row = linkOpenInNewWindowCheckbox.closest('.row');
            $row.toggleClass('hidden', !show);
        }

        function hasBreadcrumbs(crumbTexts) {
            if (!breadcrumbsEl.is(':visible')) {
                return false;
            }

            var numCrumbs = crumbTexts.length;
            var crumbItems = breadcrumbsEl.find('li');
            if (crumbItems.length !== numCrumbs) {
                return false;
            }

            for (var i = 0; i < numCrumbs; i++) {
                if (crumbItems.eq(i).text() !== crumbTexts[i]) {
                    return false;
                }
            }

            return true;
        }

        /**
         * Updates the breadcrumbs for the passed Link object, retrieving breadcrumb data from the back end and display
         * it.
         */
        function updateBreadcrumbs(linkObj) {
            var controls = {
                clearErrors: function() {
                    // do nothing - errors will appear in the breadcrumbs element
                },
                error: function(errMsg) {
                    // do nothing - errors will appear in the breadcrumbs element
                },

                // called when a destination is selected on one of the panels
                select: function(options) {
                    breadcrumbController.update(options, controls);
                }
            };

            var options = {
                id: linkObj.getResourceId(),
                type: linkObj.getResourceType()
            };
            breadcrumbController.update(options, controls);
        }

        /**
         * Updates the link destination in the Location panel of the Link Browser.
         *
         * @param linkObj a linkable ContentEntity with an id and title, or some other object with an href property
         * @param showBreadcrumbs
         */
        function setLink(linkObj, showBreadcrumbs) {
            AJS.debug('Link Browser: setting link : ' + linkObj);

            setLinkTextFromLink(linkObj);

            // id and type is required for update Breadcrumbs so we also check if they have value
            showBreadcrumbs && linkObj.getResourceId() && linkObj.getResourceType() && updateBreadcrumbs(linkObj);
            // Show/hide breadcrumbs element immediately (before any potential AJAX requests). This allows the breadcrumb
            // controller to show loading messages.
            showBreadcrumbsElement(showBreadcrumbs);

            if (AJS.DarkFeatures.isEnabled(lbController.OPEN_IN_NEW_WINDOW_DARK_FEATURE)) {
                // if the box is already checked, it means it was previously checked, so need to make sure that the link
                // is in sync with that
                if (linkOpenInNewWindowCheckbox.prop('checked')) {
                    linkObj.setTarget('_blank');
                }
                linkOpenInNewWindowCheckbox.prop('checked', linkObj.getTarget() === '_blank');
            } else {
                linkObj.removeTarget();
            }

            currentLink = linkObj;
            lbController.linkValid(currentLink && currentLink.isHrefValid());
        }

        /**
         * Returns the link object currently stored in the panel.
         */
        function getLink() {
            if (!currentLink) {
                return null;
            }

            var body = finalBody;
            if (!body) {
                // Not an image/mixed content link. Perhaps there is alias text entered in the field?
                var linkText = getLinkText();
                if (!linkText) {
                    // If there is no link text, use a defaultAlias in the case of internal Confluence
                    // links, or the href for an external link.
                    linkText = currentLink.getDefaultAlias() || currentLink.getHref();
                }
                body = {
                    html: linkText
                };
            }
            currentLink.body = body;
            currentLink.attrs.href = encodeURLSafely(currentLink.attrs.href);
            return currentLink;
        }

        /**
         * @param url the URL want to be encoded special characters
         * @returns {string} the url after encoding special character
         */
        function encodeURLSafely(url) {
            /**
             This regex return 10 segments
             $0: full url
             $1: scheme + :
             $2: scheme
             $3: // + authority
             $4: authority
             $5: path
             $6: query with ?
             $7: query
             $8: # + hash
             $9: hash
             *
             */
            // This regex is from RFC-3986. If you need to change it, think about it some more.
            var urlElement = url.match(/^(([^:/?#]+):)?(\/\/([^/?#]*))?([^?#]*)(\?([^#]*))?(#(.*))?/);

            // This algorithm is from RFC-3986: https://tools.ietf.org/html/rfc3986#section-5.3
            var scheme = urlElement[2];
            var authorityIncludingSlashes = urlElement[3];
            var path = urlElement[5];
            var query = urlElement[7];
            var fragment = urlElement[9];

            var newUrl = '';

            if (scheme) {
                newUrl = scheme.toLowerCase() + ':';
            }
            if (authorityIncludingSlashes) {
                newUrl += authorityIncludingSlashes;
            }

            newUrl += _encodeSafely(path);

            if (query) {
                newUrl += '?' + _encodeSafely(query);
            }
            if (fragment) {
                newUrl += '#' + _encodeSafely(fragment);
            }
            return newUrl;
        }

        function _encodeSafely(encodeString) {
            var characters = {
                '\'': '%27', '\\[': '%5B', '\\]': '%5D', '`': '%60', '\\\\': '%5C'
            };
            var safeEncodedString = encodeString;
            for (var character in characters) {
                if (characters.hasOwnProperty(character)) {
                    var encoded = characters[character];
                    var regex = new RegExp(character, 'g');
                    safeEncodedString = safeEncodedString.replace(regex, encoded);
                }
            }
            return safeEncodedString;
        }

        /**
         * Puts the cursor in the 'Link Text' field, selecting the current text. Returns true on success, else false.
         */
        function focusLinkText() {
            if (isLinkTextVisible()) {
                AJS.debug('LinkInfoPresenter.focusLinkText focusing alias');
                linkTextField.select();
                return true;
            }
            return false;
        }

        function isLinkTextVisible() {
            return linkTextField.is(':visible');
        }

        function isLinkImageVisible() {
            return linkImageName.is(':visible');
        }

        function isLinkMixedContentVisible() {
            return linkMixedContent.is(':visible');
        }

        function isNewWindowCheckboxVisible() {
            return linkOpenInNewWindowCheckbox.is(':visible');
        }

        function isNewWindowCheckboxChecked() {
            return linkOpenInNewWindowCheckbox.prop('checked');
        }

        /**
         * Called when the Link Browser tab is changed, has the location panel refresh itself based on required size
         * and whether breadcrumbs should be shown.
         */
        function refresh(tabShowsBreadcrumbs) {
            if (currentLink) {
                showBreadcrumbsElement(tabShowsBreadcrumbs);
            }
        }

        /**
         * Returns the container with the Location elements.
         */
        function getContainer() {
            return container;
        }

        /**
         * Moves the location panel to be the last element inside the new parent. Used for styling.
         * @param newParent
         */
        function moveLocationPanel(newParent) {
            if (!movedPanel) {
                movedPanel = container.find('.row:not(.hidden) .field-group');
                movedPanel.each(function(i, panel) {
                    $(panel).data('original-parent', $(panel).parent());
                });
            }
            movedPanel.appendTo(newParent);
            container.hide();
        }

        /**
         * Moves the location panel back to its normal position
         */
        function restoreLocationPanel() {
            if (movedPanel) {
                movedPanel.each(function(i, panel) {
                    var $panel = $(panel);
                    if ($panel.data('original-parent')) {
                        $panel.appendTo($panel.data('original-parent'));
                        $panel.removeData('original-parent');
                    }
                });
            }
            container.show();

            movedPanel = null;
        }

        function _triggerAnalyticsEvent(name, link) {
            AJS.trigger('analyticsEvent', {
                name: name,
                data: {
                    resourceType: link.getResourceType(),
                    isExternal: link.isExternalLink()
                }
            });
        }

        return {
            setLink: setLink,
            getLink: getLink,
            refresh: refresh,
            setLinkBody: setLinkBody,
            getContainer: getContainer,
            isLinkTextVisible: isLinkTextVisible,
            isLinkImageVisible: isLinkImageVisible,
            isNewWindowCheckboxVisible: isNewWindowCheckboxVisible,
            isNewWindowCheckboxChecked: isNewWindowCheckboxChecked,
            isLinkMixedContentVisible: isLinkMixedContentVisible,
            focusLinkText: focusLinkText,
            getLinkText: getLinkText,
            getRawLinkText: getRawLinkText,
            getLinkImageName: getLinkImageName,
            moveLocationPanel: moveLocationPanel,
            restoreLocationPanel: restoreLocationPanel,
            hasBreadcrumbs: hasBreadcrumbs,
            showOpenInNewWindowCheckbox: showOpenInNewWindowCheckbox,
            encodeURLSafely: encodeURLSafely
        };
    };
});

require('confluence/module-exporter').exportModuleAsGlobal('confluence-link-browser/link-browser-location', 'Confluence.Editor.LinkBrowser.LinkInfoPresenter');
