/**
 * @module confluence-link-browser/link-browser-tab-attachment
 */
define('confluence-link-browser/link-browser-tab-attachment', [
    'jquery',
    'ajs',
    'confluence/legacy'
],
function(
    $,
    AJS,
    Confluence
) {
    'use strict';

    return {
        dialogCreatedLinkBrowserAttachment: function(e, linkBrowser) {
            var restUrl = AJS.REST.makeUrl('content/' + AJS.Meta.get('attachment-source-content-id') + '/attachments.json');
            var key = 'attachments';
            var controller;
            var uploaderController;
            var messageHandler;
            var resultGrid;

            // Returns the controller for this panel
            var getController = function(context, getOverrideController) {
                var uploadForm = context.baseElement.find('.attach-file-form');

                return $.extend({
                    getUploaderController: function() {
                        var uploaderContext = {
                            baseElement: uploadForm
                        };
                        var uploaderControllerOverride = function(context) {
                            return {
                                onUploadSuccess: function(attachmentsAdded) {
                                    // Jsonated object properties should match REST ones.
                                    for (var i = 0, len = attachmentsAdded.length; i < len; i++) {
                                        attachmentsAdded[i].type = 'attachment';
                                    }
                                    resultGrid.prependAndSelect(attachmentsAdded);
                                }
                            };
                        };
                        return Confluence.AttachmentUploader(uploaderContext, uploaderControllerOverride);
                    }
                },
                getOverrideController && getOverrideController(context)
                );
            };

            linkBrowser.tabs[key] = {
                hasBreadcrumbs: false,

                // Called by the main Link Browser controller on startup to initialize the by-now-existing panel DOM
                // element.
                createPanel: function(context, getOverrideController) {
                    var tab = this;

                    controller = getController(context, getOverrideController);
                    uploaderController = controller.getUploaderController(context);
                    messageHandler = uploaderController.getMessageHandler();

                    var columns = [

                        // File column has a link to the file and a class for styling with an icon
                        AJS.SelectGrid.Column({
                            key: 'title',
                            heading: AJS.I18n.getText('link.browser.table.file.name'),
                            getHref: function(rowData) {
                                if (rowData.link) {
                                    // REST object returned by attachment search
                                    return AJS.REST.findLink(rowData.link);
                                }
                                // JSON returned by upload attachment action
                                return rowData.url;
                            },
                            getInnerClass: function(rowData) {
                                return rowData.iconClass;
                            }
                        }),
                        // Size column
                        AJS.SelectGrid.Column({
                            key: 'size',
                            heading: AJS.I18n.getText('link.browser.table.file.size'),
                            getText: function(rowData) {
                                return rowData.niceFileSize;
                            }
                        }),

                        // Comment column
                        AJS.SelectGrid.Column({
                            key: 'comment',
                            heading: AJS.I18n.getText('link.browser.table.file.comment')
                        })
                    ];

                    resultGrid = new AJS.ResultGrid({
                        baseElement: context.baseElement,
                        columns: columns,

                        selectionCallback: function(rowElement, data) {
                            var linkObj = Confluence.Link.fromREST(data);
                            // The attachment will be on the same page so set the data-linked-resource-container-id.
                            linkObj.attrs['data-linked-resource-container-id'] = AJS.Meta.get('content-id');

                            // Need to check that the href leads to the download path for the attachment.
                            if ($.isArray(data.link)) {
                                for (var i = 0, ii = data.link.length; i < ii; i++) {
                                    var link = data.link[i];
                                    if (link.rel === 'download') {
                                        linkObj.attrs.href = link.href;
                                    }
                                }
                            }
                            linkBrowser.setLink(linkObj);
                            linkBrowser.focusLinkText();
                        },
                        noResultMessage: AJS.I18n.getText('link.browser.files.none')
                    });
                },

                // When the Attachment tab is selected the current attachments are refreshed.
                onSelect: function() {
                    // Store a ref in scope to last across the AJAX request - the tab.openedLink ref is dropped by the
                    // main controller after the tab.select() call is made.
                    var openedLink = this.openedLink;
                    var selectedLink = linkBrowser.getLink();

                    resultGrid.loading();
                    AJS.getJSONWrap({
                        url: restUrl,
                        messageHandler: messageHandler,
                        successCallback: function(data) {
                            resultGrid.update(data.attachment);

                            if (selectedLink) {
                                if (selectedLink.getResourceType() == 'attachment') {
                                    resultGrid.select(selectedLink.getResourceId());
                                }
                            } else if (openedLink) {
                                // If this panel has been opened automatically to edit an existing attachment link,
                                // select that link.
                                resultGrid.select(openedLink.getResourceId());
                            } else if (data.attachment.length) {
                                resultGrid.select(data.attachment[0].id);
                            }
                            AJS.debug('Loaded attachments');
                        }
                    });
                },

                // The Attachment panel opens existing links for attachments that are on the current page or blogpost.
                handlesLink: function(linkObj) {
                    return linkObj.isToAttachmentOnSamePage(AJS.Meta.get('content-id'));
                }
            };
        }
    };
});

require('confluence/module-exporter').safeRequire('confluence-link-browser/link-browser-tab-attachment', function(LinkBrowserTabAttachment) {
    'use strict';

    var AJS = require('ajs');

    // Not supported for templates
    if (AJS.Meta.get('content-type') === 'template') {
        return;
    }

    AJS.bind('dialog-created.link-browser', LinkBrowserTabAttachment.dialogCreatedLinkBrowserAttachment);
});
