/**
 * @module confluence-link-browser/link-browser-tab-history
 */
define('confluence-link-browser/link-browser-tab-history', [
    'ajs',
    'confluence/legacy'
],
function(
    AJS,
    Confluence
) {
    'use strict';

    // The History tab registers itself when the Link Browser is created.
    return {
        dialogCreatedLinkBrowserHistory: function(e, linkBrowser) {
            /**
             Link Dialog tab object is a method that initializes the tab and returns an object
             with these methods:
             1. onSelect - code that should run every time the panel is selected
             2. getLinkDetails - returns the current link represented by this panel's state
             The init method takes the following arguments:
             a. linkDialog - the JS link dialog controller
             */
            var restSearchUrl = AJS.REST.makeUrl('session/history.json?max-results=20');
            var resultGrid;
            var messageHandler;
            var key = 'recentlyviewed';

            linkBrowser.tabs[key] = {

                createPanel: function(context, getOverrideController) {
                    var rowSelected = function(rowElement, data) {
                        var linkObj = Confluence.Link.fromREST(data);
                        linkBrowser.setLink(linkObj);
                        linkBrowser.focusLinkText();
                    };

                    var columns = [

                        // File column has a link to the file and a class for styling with an icon
                        AJS.SelectGrid.Column({
                            key: 'title',
                            heading: AJS.I18n.getText('result.grid.column.content.title'),
                            getHref: function(rowData) {
                                return AJS.REST.findLink(rowData.link);
                            },
                            getInnerClass: function(rowData) {
                                return rowData.iconClass || ('content-type-' + rowData.type);
                            }
                        }),
                        // Space column
                        AJS.SelectGrid.Column({
                            key: 'space',
                            heading: AJS.I18n.getText('result.grid.column.content.space'),
                            getText: function(rowData) {
                                return rowData.space && rowData.space.title || '';
                            }
                        }),

                        // Last Modified column
                        AJS.SelectGrid.Column({
                            key: 'last-modified',
                            heading: AJS.I18n.getText('result.grid.column.content.last.modified'),
                            getText: function(rowData) {
                                return rowData.lastModifiedDate && rowData.lastModifiedDate.friendly || '';
                            },
                            getTitle: function(rowData) {
                                return rowData.lastModifiedDate && rowData.lastModifiedDate.date || '';
                            }
                        })
                    ];
                    var getRowId = function(rowData) {
                        return rowData.attachmentId;
                    };

                    messageHandler = AJS.MessageHandler({
                        baseElement: context.baseElement.find('.message-panel')
                    });

                    resultGrid = new AJS.ResultGrid({
                        baseElement: context.baseElement,
                        columns: columns,
                        getRowId: getRowId,
                        selectionCallback: rowSelected,
                        messageHandler: messageHandler,
                        noResultMessage: AJS.I18n.getText('link.browser.recently.viewed.none')
                    });
                },

                // The History list is refreshed each time the user selects the "Recently Viewed" tab
                onSelect: function() {
                    resultGrid.loading();

                    AJS.getJSONWrap({
                        url: restSearchUrl,
                        messageHandler: messageHandler,
                        successCallback: function(data) {
                            resultGrid.update(data.content);
                            AJS.trigger('updated.link-browser-recently-viewed');
                        }
                    });
                }
            };
        }
    };
});

require('confluence/module-exporter').safeRequire('confluence-link-browser/link-browser-tab-history', function(LinkBrowserTabHistory) {
    'use strict';

    var AJS = require('ajs');

    AJS.bind('dialog-created.link-browser', LinkBrowserTabHistory.dialogCreatedLinkBrowserHistory);
});
