/**
 * @module confluence-link-browser/link-browser-tab-search
 */
define('confluence-link-browser/link-browser-tab-search', [
    'jquery',
    'ajs',
    'document',
    'confluence/legacy',
    'confluence/page-location'
],
function(
    $,
    AJS,
    document,
    Confluence,
    PageLocation
) {
    'use strict';

    // The Search tab registers itself when the Link Browser is created.
    return {
        dialogCreatedLinkBrowserSearch: function(e, linkBrowser) {
            var searchTextField;
            var searchSpaceField;
            var resultGrid;
            var messageHandler;
            var key = 'search';
            var restSearchUrl = AJS.REST.getBaseUrl() + 'search.json';

            var setLink = function(linkObj, updateSearchField) {
                if (!Confluence.Link.isLink(linkObj)) {
                    linkObj = Confluence.Link.fromREST(linkObj);
                }
                updateSearchField && searchTextField.val(linkObj.attrs['data-linked-resource-default-alias']);
                linkBrowser.setLink(linkObj, true);
                linkBrowser.focusLinkText();
            };

            // Perform a search on the passed text
            var doSearch = function(callback) {
                AJS.debug('link-browser-tab-search.js: doing search');

                searchTextField.trigger('hide.autocomplete');
                var searchText = $.trim(searchTextField.val());

                if (searchText) {
                    resultGrid.loading();

                    AJS.getJSONWrap({
                        url: restSearchUrl,
                        data: {
                            search: 'site',
                            query: searchText,
                            spaceKey: searchSpaceField.val()
                        },
                        successCallback: function(data) {
                            if ($.isFunction(callback)) {
                                resultGrid.update(data.result);
                                callback();
                            } else {
                                resultGrid.updateAndSelect(data.result);
                            }

                            // If the autocomplete result has appeared, hide it.
                            searchTextField.trigger('hide.autocomplete');
                        },
                        messageHandler: messageHandler
                    });
                } else {
                    $.isFunction(callback) && callback();
                }
            };

            linkBrowser.tabs[key] = {

                hasBreadcrumbs: true,

                createPanel: function(context) {
                    var $panel = context.baseElement;
                    searchTextField = $panel.find('#link-search-text');
                    searchSpaceField = $panel.find('#search-panel-space');

                    // Space may have changed since page load if page has been moved.
                    var currentLocation = PageLocation.get();
                    searchSpaceField.find('option:eq(1)').text(currentLocation.spaceName).val(currentLocation.spaceKey);

                    searchSpaceField.change(function() {
                        // When the space changes update the attribute on the autocomplete field so it can re-loaded when the
                        // field gets focus.
                        searchTextField.attr('data-spacekey', searchSpaceField.val());
                        searchTextField.trigger('clearCache.autocomplete');
                    });

                    Confluence.Binder.autocompleteSearch(searchTextField.parent());

                    searchTextField.bind('selected.autocomplete-content', function(e, data) {
                        // If the user selects the "Search for" link from the autocomplete, perform the search.
                        if (data.searchFor) {
                            doSearch();
                        } else {
                            resultGrid.clear();
                            setLink(data.content, false);
                        }
                    });

                    var columns = [

                        // File column has a link to the file and a class for styling with an icon
                        AJS.SelectGrid.Column({
                            key: 'title',
                            heading: AJS.I18n.getText('link.browser.table.title'),
                            getHref: function(rowData) {
                                return AJS.REST.findLink(rowData.link);
                            },
                            getInnerClass: function(rowData) {
                                return rowData.iconClass || ('content-type-' + rowData.type);
                            }
                        }),

                        // Space column
                        AJS.SelectGrid.Column({
                            key: 'space',
                            heading: AJS.I18n.getText('result.grid.column.content.space'),
                            getText: function(rowData) {
                                return rowData.space && rowData.space.title || '';
                            }
                        }),

                        // Last Modified column
                        AJS.SelectGrid.Column({
                            key: 'last-modified',
                            heading: AJS.I18n.getText('result.grid.column.content.last.modified'),
                            getText: function(rowData) {
                                return rowData.lastModifiedDate && rowData.lastModifiedDate.friendly || '';
                            },
                            getTitle: function(rowData) {
                                return rowData.lastModifiedDate && rowData.lastModifiedDate.date || '';
                            }
                        })
                    ];

                    messageHandler = AJS.MessageHandler({
                        baseElement: context.baseElement.find('.message-panel')
                    });

                    resultGrid = new AJS.ResultGrid({
                        baseElement: context.baseElement,
                        columns: columns,

                        // Called when a row of the search results is selected by mouse or keyboard.
                        selectionCallback: function(rowElement, data) {
                            setLink(data);
                        },

                        noResultMessage: AJS.I18n.getText('link.browser.search.results.none'),

                        // Disable the up/down keys of the grid when the search field or space select have the focus
                        dontShiftFocus: function() {
                            return $(searchTextField).add(searchSpaceField).is(':focus');
                        }
                    });

                    // Performs the search when the form at the top of the panel is submitted.
                    // The form may be submitted by the user clicking the Search button or pressing
                    // 'Enter' in one of the form components.
                    $panel.find('.search-form').submit(function() {
                        doSearch();
                        return false;
                    }).keydown(function(e) {
                        if (e.keyCode === 13) {
                            // If the search autocomplete has a dropdown visible with an item selected we should let it handle
                            // the Enter - if nothing is selected we should close the dropdown and do the search.
                            var dropdownWithSelection = $('.aui-dropdown:visible .active', this).length;
                            if (!dropdownWithSelection) {
                                $('#search-panel-button').focus();
                                e.stopPropagation(); // Don't let the dialog try to Submit itself when it catches the enter.
                            }
                        }
                    });
                },

                onSelect: function() {
                    AJS.debug('Link Browser Search panel selected');

                    var openedLink = this.openedLink;
                    if (openedLink) {
                        AJS.debug('Link Browser Search panel setting link info');
                        setLink(openedLink, true);
                    } else {
                        searchTextField.focus();
                    }
                },

                // The Search panel handles links that have a Confluence resource id, unless they're attachment links on the
                // same page or links with anchors.
                handlesLink: function(linkObj) {
                    return linkObj.isCustomAtlassianContentLink() && !linkObj.hasAnchor() && !linkObj.isShortcutLink() && !linkObj.isToAttachmentOnSamePage(AJS.Meta.get('content-id'));
                },

                /*
                 * Perform the search by setting the search field value and reproducing a search-form submit.
                 *
                 * @param searchText - the text to set in the search field and search on
                 * @param callback (optional) - a function to call after the search results have been populated
                 */
                doSearch: function(searchText, callback) {
                    searchTextField.val(searchText);
                    doSearch(callback);
                },

                isResultGridVisible: function() {
                    return resultGrid.isVisible();
                },

                getSearchTextField: function() {
                    return searchTextField;
                }
            };
        }
    };
});
require('confluence/module-exporter').safeRequire('confluence-link-browser/link-browser-tab-search', function(LinkBrowserTabSearch) {
    'use strict';

    var AJS = require('ajs');

    AJS.bind('dialog-created.link-browser', LinkBrowserTabSearch.dialogCreatedLinkBrowserSearch);
});
