/**
 * @module confluence-link-browser/link-browser-tab-weblink
 */
define('confluence-link-browser/link-browser-tab-weblink', [
    'ajs',
    'jquery',
    'confluence/legacy'
], function(AJS,
    $,
    Confluence) {
    'use strict';

    // The Web Link tab registers itself when the Link Browser is created.
    return {
        dialogCreatedLinkBrowserWeblink: function(e, linkBrowser) {
            var urlField;
            var key = 'weblink';
            var thisPanel;
            var tab;
            var protocolDelimRegex = new RegExp('[:/]');

            function getURL() {
                return $.trim(urlField.val());
            }

            // Checks that the URL has a protocol, fixes if it can.
            function fixUrl() {
                var url = getURL();
                if (AJS.Validate.url(url) || url.indexOf('mailto:') === 0) {
                    // A complete URL - leave it alone
                    return;
                }
                // If the URL is not an absolute URL it might be a http URL missing the http or a mailto: missing the mailto.
                var defaultAlias = url;
                if (AJS.Validate.email(url)) {
                    // Email link is missing mailto - add it
                    url = 'mailto:' + url;
                } else if (!protocolDelimRegex.test(url)) {
                    // *Probably* just a web address like 'www.atlassian.com' that needs an http:// prefix
                    url = 'http://' + url;
                } else {
                    // Not a valid URL that we know of - might have deleted half of the protocol or something... as they wish!
                    return;
                }

                AJS.debug('Updating Link Browser Web Link URL to: ' + url);
                urlField.val(url);

                // Pass the original URL as the title so that the alias can be set to the non-prefixed URL if it
                // isn't already set.
                var linkObj = Confluence.Link.makeExternalLink(url);
                linkBrowser.setLink(linkObj);
            }

            function checkSubmitButton() {
                var url = getURL();
                var linkObj = url ? Confluence.Link.makeExternalLink(url) : null;
                if (linkObj) {
                    linkBrowser.setLink(linkObj); // will enable the Submit button when a URL is added
                }
            }

            function checkPasteUrl() {
                fixUrl();
                checkSubmitButton();
            }

            tab = linkBrowser.tabs[key] = {
                createPanel: function(context) {
                    thisPanel = context.baseElement;
                    urlField = thisPanel.find('input[name=\'destination\']');
                    urlField.keyup(function(e) {
                        checkSubmitButton();
                    });

                    urlField.change(fixUrl);
                    urlField.bind('paste', function() {
                        AJS.debug('Link Browser web link url pasted');
                        // CONF-22287 - use setTimeout to wait for the paste event to update the field value
                        setTimeout(checkPasteUrl, 0);
                    });
                },

                // Called when the panel is selected - may be called before the LB dialog has its 'show' method called
                onSelect: function() {
                    AJS.DarkFeatures.isEnabled(linkBrowser.OPEN_IN_NEW_WINDOW_DARK_FEATURE) && linkBrowser.showOpenInNewWindowCheckbox(true);
                    // Put the location panel inside the web-link panel for easier styling.
                    linkBrowser.moveLocationPanel(thisPanel.find('form:first'));

                    if (this.openedLink) {
                        tab.setURL(this.openedLink.attrs.href);
                        linkBrowser.setLink(this.openedLink);
                    }

                    // Defer focus to after LB is shown, gets around AJS.Dialog tabindex issues
                    setTimeout(function() {
                        urlField.focus();
                    });
                },

                onDeselect: function() {
                    linkBrowser.restoreLocationPanel();
                    linkBrowser.showOpenInNewWindowCheckbox(false);
                },

                // Called when the submit button is clicked, before the location is retrieved from the Location controller.
                // Re-validate and prefix the URL.
                preSubmit: fixUrl,

                // The Web link panel handles non Confluence links
                handlesLink: function(linkObj) {
                    return !linkObj.isCustomAtlassianContentLink();
                },

                setURL: function(url) {
                    urlField.val(url);
                    urlField.keyup();
                    urlField.change();
                },

                getURL: getURL
            };
        }
    };
});

require('confluence/module-exporter').safeRequire('confluence-link-browser/link-browser-tab-weblink', function(LinkBrowserTabWeblink) {
    'use strict';

    var AJS = require('ajs');

    AJS.bind('dialog-created.link-browser', LinkBrowserTabWeblink.dialogCreatedLinkBrowserWeblink);
});
