/**
 * Replaces tokens in a string with arguments, similar to Java's MessageFormat.
 * Tokens are in the form {0}, {1}, {2}, etc.
 * @method format
 * @param message the message to replace tokens in
 * @param arg (optional) replacement value for token {0}, with subsequent arguments being {1}, etc.
 * @return {String} the message with the tokens replaced
 * @usage MW.format("This is a {0} test", "simple");
 */
MW.format = function (message) {
    var token = /^((?:(?:[^']*'){2})*?[^']*?)\{(\d+)\}/, // founds numbers in curly braces that are not surrounded by apostrophes
        apos = /'(?!')/g; // founds "'", bot not "''"
    // we are caching RegExps, so will not spend time on recreating them on each call
    MW.format = function (message) {
        var args = arguments,
            res = "",
            match = message.match(token);
        while (match) {
            message = message.substring(match[0].length);
            res += match[1].replace(apos, "") + (args.length > ++match[2] ? args[match[2]] : "");
            match = message.match(token);
        }
        return res += message.replace(apos, "");
    };
    return MW.format.apply(MW, arguments);
};

/*
 * Similar to AJS.format(), but uses a map instead of ordinal parameters.
 *
 * eg.
 * MW.format("{user} commented on {key}", {user: 'admin', key: 'TST-1'}) == "admin commented on TST-1"
 */
MW.formatString = function (str, map) {
    return str.replace(/\{\{|\}\}|\{(\w+)\}/g, function (m, n) {
        if (m == "{{") { return "{"; }
        if (m == "}}") { return "}"; }
        return map[n];
    });
};

MW.getParameterByNameFromHash = function(name) {
    return MW.getParameterByName(name, window.location.hash);
};

MW.getParameterByName = function(name, search) {
    name = name.replace(/[\[]/, "\\[").replace(/[\]]/, "\\]");
    var regexS = "[\\?\\#&]" + name + "=([^&#]*)";
    var regex = new RegExp(regexS);
    var results = regex.exec(search || window.location.search);
    if(results === null) {
        return "";
    } else {
        return decodeURIComponent(results[1].replace(/\+/g, " "));
    }
};

MW.setRequestTime = function(time) {
    MW.timeDifference = new Date().getTime() - parseInt(time, 10);
};

/**
 * Formats given time using Confluence's translations for relative dates
 * falling back to operating system's default date format for dates more than
 * one day ago.
 */
MW.getPrettyTime = function(time) {
    var date = new Date(time + MW.timeDifference),
        now = new Date(),
        delta = now - date,
        minute = 60000,
        hour = 60 * minute;


    if (date >= now) {
        return AJS.I18n.getText("date.friendly.now");
    } else if (delta < minute) {
        return AJS.I18n.getText("date.friendly.a.moment.ago");
    } else if (delta < 2 * minute) {
        return AJS.I18n.getText("one.min.ago");
    } else if (delta < 50 * minute) {
        return AJS.I18n.getText("x.mins.ago", Math.floor(delta / minute));
    } else if (delta < 90 * minute) {
        return AJS.I18n.getText("date.friendly.about.one.hour.ago");
    } else if (delta < 2 * hour) {
        // Without this, if we are between 90 mins and 2 hours, we get "1 hours"
        return AJS.I18n.getText("date.friendly.about.x.hours.ago", Math.ceil(delta / hour));
    } else if (delta < 24 * hour) {
        return AJS.I18n.getText("date.friendly.about.x.hours.ago", Math.floor(delta / hour));
    } else if (delta < 48 * hour) {
        return AJS.I18n.getText("modified.yesterday");
    } else {
        return AJS.I18n.getText("com.atlassian.mywork.notification.x.days.ago", Math.floor(delta / (24*hour)));
    }
};

MW.escapeHTML = function(str) {
    return str ? str.replace(/&/g,'&amp;').replace(/</g,'&lt;').replace(/>/g,'&gt;') : str;
};

// TODO Remove when we upgrade Underscore
MW.pick = function(obj) {
    var result = {};
    _.each(_.flatten(Array.prototype.slice.call(arguments, 1)), function(key) {
        if (key in obj) {
            result[key] = obj[key];
        }
    });
    return result;
};

MW.Deferred = function() {
    var deferred = MW.$.Deferred();
    return  {
        done: _.bind(deferred.done, deferred),
        isResolved: function() {
            return deferred.state() === 'resolved';
        },
        resolveOnSuccess: function(options) {
            return _.extend({}, options, {success: function() {
                if (options && options.success) {
                    options.success();
                }
                deferred.resolve();
            }});
        }
    };
};


MW.makeTextAreaExpandable = function (container, callback) {
    // http://www.alistapart.com/articles/expanding-text-areas-made-elegant/
    var area = container.find('.expandingArea textarea');
    var span = container.find('.expandingArea span');
    span.text(area.val());
    area.bind('input', function(){
        span.text(area.val());
        if (callback) {
            callback();
        }
    });

    // Add 'active' class via js so if there is no js will resort to plain textarea with scrollbars
    container.find(".expandingArea").addClass('active');
};

MW.getBaseUrl = function() {
    return window.location.href.split('/', 3).join('/') + MW.contextPath;
};

MW.createMiniviewCallback = function(parentUrl, aggregateKey) {
    var url = document.createElement('a');
    url.href = parentUrl;
    var search = url.search;
    if (search.length === 0) {
        search = "?show-miniview";
    } else if (!/[?&]show-miniview/.test(search)) {
        search += "&show-miniview";
    }
    if (aggregateKey) {
        search += '=' + encodeURIComponent(aggregateKey);
    }
    return url.pathname + search + url.hash;
};

MW.createCallback = function(baseUrl, aggregateKey) {
    if (MW.App.parseView() !== "chrome") {
        if (MW.App.parseView() === "mobile") { // mobile
            return baseUrl + '/plugins/servlet/mobile?#mywork/notification';
        } else if (!!MW.parentConfig.parentUrl) { // miniview
            return MW.createMiniviewCallback(MW.parentConfig.parentUrl, aggregateKey);
        } else { // full-screen
            return baseUrl + '/plugins/servlet/notifications-miniview/notification';
        }
    }
};

// TODO We probably should extract the _second_ redirectUrl and do this properly
MW.appendCallback = function(url, callbackUrl) {
    var delim = url.indexOf('%3F') < 0 ? '?' : '&';
    return encodeURIComponent(delim + "callback=" + encodeURIComponent(callbackUrl));
};

MW.onJsonMessage = function(jsonHandler) {
    function eventHandler(event) {
        // Save data here, as IE8 will return "" from message.data later on
        var myOriginTarget = location.protocol + '//' + location.host;
        if (event.origin !== myOriginTarget) {
            return;
        }
        var data = JSON.parse(event.data);
        jsonHandler(data, event);
    }

    if (!window.addEventListener) {
        window.attachEvent("onmessage", eventHandler);
    } else {
        window.addEventListener("message", eventHandler, false);
    }
};

MW.$(function() {
    /**
     * Converts a form into a JSON object.
     */
    MW.$.fn.serializeObject = function() {
        var o = {};
        var a = this.serializeArray();
        _.each(a, function(item) {
            if (o[item.name] !== undefined) {
                if (!o[item.name].push) {
                    o[item.name] = [o[item.name]];
                }
                o[item.name].push(item.value || '');
            } else {
                o[item.name] = item.value || '';
            }
        });
        return o;
    };

});
