MW.NotificationEntryAbstractView = Backbone.View.extend({

    className: "mw-drilldown-list",

    events: {
        "click": "focusSelected",
        "click .authorise": "authorise",
        "click .discard": "discard"
    },

    initialize: function(options) {
        this.globalEvents = options.globalEvents;

        this.model.bind('change drilldownFocused', this.rerenderFocus, this);
    },

    cleanup: function() {
        this.model.off(null, null, this);
    },

    render: function() {
        var el = MW.$(this.el);
        var notification = this.model;

        el.toggleClass("focused", notification.get("drilldownFocused") || false);

        if (notification.get("read")) {
            el.addClass("collapsed");
        }

        // HACK for mobile to display username instead of title
        var drilldownTitle = "";

        drilldownTitle = this.createNotificationTitle(notification);

        this.renderTemplate(el, {notification: notification.toJSON(), htmlTitle: drilldownTitle});

        this.fixRelativeUrls();

        if (MW.Notifications.isMyWorkAuth(notification.get('parent').attributes)) {
            el.find('.mw-drilldown-excerpt').append(
                MW.$('<button class="authorise">' + AJS.I18n.getText('com.atlassian.mywork.oauth.authorise') + '</button>' +
                      '<button class="discard">' + AJS.I18n.getText('com.atlassian.mywork.oauth.discard') + '</button>'));
        }

        // Use a default avatar if we have problems getting the user's actual avatar
        this.ensureNoMissingImage(el.find('.mw-drilldown-avatar'), 'useravatar');

        this.actionMessageView = new MW.ActionMessageView({
            el: el.find('.message')[0],
            aggregateKey: notification.get('parent').get('aggregateKey')
        });

        this.addActions();

        return this;
    },

    /**
     * Replace any relative links (to things like issues) with absolute URLs.
     */
    fixRelativeUrls: function() {
        // Just the host:port, no context
        var url = this.model.get('parent').config.getUrl(),
            baseUrl = url.split('/', 3).join('/');
        this.$el.find("a").each(function(i, tag) {
            var href = MW.$(tag).attr("href");
            if (!!href && href.indexOf("/") === 0) {
                MW.$(tag).attr("href", baseUrl + href);
            }
        });
    },

    rerenderFocus: function(){
        MW.$(this.el).toggleClass("focused", this.model.get("drilldownFocused") || false);
    },

    focusSelected: function() {
        this.globalEvents.trigger("focusClickedNotification", this.model);
    },

    // Make the username link to the user profile
    htmlTitle: function() {
        // We actually want "this" as the title eg. "Darth Vader commented on this"
        var thisMsg = this.model.get("parent").entityI18n("this") || AJS.I18n.getText('com.atlassian.mywork.this'),
            metadata = MW.$.extend({title: thisMsg}, this.model.get('metadata'));
        return this.model.get("parent").i18nHtml('title', metadata) || MW.escapeHTML(this.model.get("title"));
    },

    ensureNoMissingImage: function($img, defaultImage) {
        $img.on('error', function() {
            var $span = MW.$('<span />');
            $span.addClass(defaultImage);
            $img.after($span);
            $img.remove();
        });
    },

    addActions: function() {
        var $el = MW.$(this.el),
            notification = this.model,
            $actions = $el.find('.mw-drilldown-actions ul'),
            $actionForm = $el.find('.action-form'),
            actionList = notification.get('actions').reverse();

        var that = this;
        MW.$.each(actionList, function(i, action) {
            if (!MW.Configurations.getFromNotification(notification.attributes)) {
                return;
            }
            var view = new MW.ActionView({
                model: action,
                url: notification.get("url"),
                itemUrl: notification.get("parent").get("url"),
                actionFormElement: $actionForm,
                notification: notification,
                actions: actionList,
                target: "notification",
                actionMessageView: that.actionMessageView,
                globalEvents: that.globalEvents
            });
            $actions.prepend(view.render().el);
        });
        actionList.reverse();
        this.addTime($actions, notification.getPrettyTime());
    },

    authorise: function() {
        window.top.location.href = this.model.get('url');
    },

    discard: function() {
        var notification = this.model;
        notification.executeInlineAction({
            actionName: 'dismissAuth',
            target: 'notification',
            success: function(result) {
                if (result.successful) {
                    MW.Tasks.remove(MW.Tasks.getByGlobalId(notification.get('globalId')));
                    MW.Notifications.remove(notification.get("parent"));
                    MW.App.gotoMain();
                } else {
                    var msg = AJS.I18n.getText('com.atlassian.mywork.action.unknown.error');
                    this.actionMessageView.showErrorMessage(msg);
                }
            },
            error: function(xhr, textStatus, message) {
                if (xhr.status === 401) {
                    this.actionMessageView.showLoginErrorMessage(AJS.I18n.getText('com.atlassian.mywork.action.unauthorised'));
                } else {
                    this.actionMessageView.showErrorMessage(message);
                }
            }
        });
    }
});
