// The main Notifications view which holds the list of notifications (notification-view.js)
MW.NotificationMainAbstractView = Backbone.View.extend({

    className: "mw-notifications card",

    events: {
        "click .mw-show-more" : "showMoreNotificationsEvent",
        "click .mw-button-tasks" : "gotoTasks"
    },

    initialize: function(options){
        this.globalEvents = options.globalEvents;

        this.globalEvents.on("showNotificationCard", this.showCard, this);
        this.globalEvents.bind("highlightNotificationAbove", this.highlightNotificationAbove, this);
        this.globalEvents.bind("highlightNotificationBelow", this.highlightNotificationBelow, this);
        this.globalEvents.bind("unblockKeyboardShortcut", this.syncFocusPosition, this);

        MW.Notifications.bind('reset', this.addNotificationsToContainer, this);
        MW.Notifications.bind('remove', this.addNotificationsToContainer, this);
        MW.Notifications.bind('add', this.addOneNewNotification, this);
        MW.Notifications.bind('modified', this.updateNotification, this);

        var kb = MW.KeyboardShortcuts.setupContext("notification", "showNotificationCard", this);
        kb.addShortcut(107, this.highlightNotificationBelow); // j
        kb.addShortcut(106, this.highlightNotificationAbove); // k
        kb.addShortcut(13, this.renderFocusedDrilldown); // enter
    },

    syncFocusPosition: function () {
        var $el = MW.$(this.el);
        var focused = $el.find('.focused');
        if (focused[0]) {
            focused[0].focus();
        }
    },

    cleanUp: function() {
        MW.Notifications.off();
    },

    renderFocusedDrilldown: function() {
        MW.App.globalEvents.trigger("renderDrilldown", MW.Notifications.focused("mainFocused"));
    },

    highlightNotificationAbove: function() {
        if (!MW.Notifications.focusMain("down", this.isShowingAll())) {
            this.showMoreNotifications('keyboard');
        } else {
            MW.App.scrollToFocusedMain();
        }
        this.syncFocusPosition();
    },

    highlightNotificationBelow: function() {
        MW.Notifications.focusMain("up", this.isShowingAll());
        MW.App.scrollToFocusedMain();
        this.syncFocusPosition();
    },

    isShowingAll: function() {
        return !MW.$(".mw-show-more").filter(":visible").length;
    },

    render: function() {
        var $el = MW.$(this.el);

        this.renderTemplate($el);

        this.notificationList = $el.find(".mw-notifications-list");

        return this;
    },

    addNotificationsToContainer: function() {
        var $notificationsListContainer = this.notificationList;

        var notifications = MW.Notifications.main();

        $notificationsListContainer.empty();
        if (notifications.length === 0) {
            // There are no notifications.
            this.renderEmptyNotificationsTemplate($notificationsListContainer);
        } else {
            // If there are no new notifications, add all notifications
            this.addAllNotifications(notifications);
        }

        // Hide if total nested notifications count is less than MW.MAX_RESULTS (we have run out of notifications)
        var $showMore = MW.$(MyWork.Templates.showMore());
        if (MW.Notifications.countNested(notifications) < MW.MAX_RESULTS) {
            $showMore.addClass("hidden");
        }
        $notificationsListContainer.append($showMore);
    },

    addAllNotifications: function(notifications) {
        var that = this;
        _.each(notifications, function(notification) {
            that.addOneNewNotification(notification);
        });
    },

    updateNotification: function(notification) {
        // Otherwise can't trust indexOf
        setTimeout(_.bind(function(){
            var element = this.notificationList.find('span[aggregate-key="' + notification.get('aggregateKey') +'"]');
            this.addOneNotification(notification, element);
        }, this));
    },

    addOneNewNotification: function(notification) {
        this.$el.find('.mw-no-notifications').remove();
        var view = new MW.NotificationView({
            model: notification,
            globalEvents: this.globalEvents,
            attributes: {
                "aggregate-key": notification.get("aggregateKey")
            }});
        var $li = MW.$('<li>', {
            class: 'mw-notification-li'
        });
        $li.append(view.render().$el);
        this.addOneNotification(notification, $li);
    },

    // Need to ensure notifications have been sorted, see setTimeout above.
    addOneNotification: function(notification, $el) {
        var $list = this.$el.find(".mw-notifications-list");
        var index = MW.Notifications.indexOf(notification);
        var $item = $list.find(".mw-notification-item").eq(index);
        if ($item.length === 0) {
            $list.append($el);
        } else if ($item[0] !== $el[0]) {
            $item.before($el);
        }
    },

    showMoreNotificationsEvent: function(event) {
        this.showMoreNotifications('click');
        event.preventDefault();
    },

    showMoreNotifications: function(from) {
        var $showMore = MW.$('.more-trigger');
        var $loading = MW.$('<li class="mw-loading-more" />');
        $showMore.before($loading);
        $showMore.addClass("hidden").remove();

        var successCallback = _.bind(function(collection, results) {
            $loading.remove();
            var count = _(results).reduce(function(memo, item) {
                return memo + item.notifications.length;
            }, 0);
            this.notificationList.append($showMore);
            if (count === MW.MAX_RESULTS) {
                $showMore.removeClass("hidden");
            }
        }, this);

        MW.Notifications.fetchWithLimit({add: true, data: {before: MW.Notifications.beforeId}, success: successCallback});

        this.fireShowMoreAnalyticsEvent(from);
    },

    fireShowMoreAnalyticsEvent: function(from) {
        AJS.EventQueue.push({name:'mywork.showmore', properties:{
            'count': MW.Notifications.main().length,
            'from': from || 'click' // This is used to indicate whether it came in from a keyboard shortcut or click
        }});
    },

    /*
     * Showing/Hiding of cards when navigating between cards
     */
    gotoTasks: function() {
        this.globalEvents.trigger("showTaskCard", 1);
    }
});
