// An individual list element that is displayed in the main notification view
// When clicked on, will slide away to show the drilldown (detailed view) of the notification
MW.NotificationAbstractView = Backbone.View.extend({

    tagName: "span",
    className: "mw-notification-item",

    events: {
        "mousedown": "renderDrilldownClick",
        "focus": "setFocusedItem"
    },

    initialize: function(options) {
        this.globalEvents = options.globalEvents;
        this.globalEvents.bind("renderDrilldown", this.renderSpecifiedDrilldown, this);

        this.model.bind('change', this.render, this);

        _.bindAll(this, "renderSpecifiedDrilldown");
    },

    setFocusedItem: function() {
        this.globalEvents.trigger("removeMainFocused");
        this.model.set("mainFocused", true);
    },

    render: function() {
        // TODO: render seems to be unnecessarily called multiple times (mainly due to change of 'mainFocused' state)
        var el = MW.$(this.el);
        el.attr("tabindex", 0);
        el.attr("role", "button");
        el.toggleClass("unread", this.model.hasNewNotifications());
        el.toggleClass("focused", this.model.get("mainFocused") || false);

        if (this.model.get("pinned")) {
            el.addClass("pinned");
        }

        this.renderTemplate(el);
        this.updateTitle();

        return this;
    },

    renderDrilldownClick: function() {
        // Need to handle click otherwise the MouseEvent is passed in as the 'instant' argument
        this.globalEvents.trigger("renderDrilldown", this.model, false);
    },

    renderSpecifiedDrilldown: function(notification, instant, reverseSlide) {
        if (this.model.get("aggregateKey") == notification.get("aggregateKey")) {
            this.renderDrilldown(instant, reverseSlide);
        }
    },

    renderDrilldown: function(instant, reverseSlide) {
        // Generate HTML for the drilldown view
        var that = this,
            drilldownView = new MW.DrilldownView({
            model: this.model,
            globalEvents: that.globalEvents,
            reverseSlide: reverseSlide
        });

        this.model.resetFocused(_(this.model.getUnreadNotifications()).last());
        //TODO: do we want to create MW.DrilldownView within the specific fns? (might be needed?)
        this.renderDrilldownTemplate(drilldownView, instant);

        // Make needed changes to the main notification view
        this.globalEvents.trigger("removeMainFocused");
        this.model.set("mainFocused", true);

        // Mark all notifications associated with this group as 'read'
        MW.Notifications.setNotificationsAsRead(this.model.notifications, true);
        this.updateTitle();
        this.$el.removeClass("unread");
    },


    updateTitle: function() {
        var mainTitleDiv = this.$el.find(".mw-notification-text");

        var groupNotifications = this.model.notifications,
            notificationModel = this.model;
        var unreadCount = _.filter(groupNotifications, function(item) {
            return !item.get("read");
        }).length;

        var newTitle = "",
            title = this.model.get("title") || groupNotifications[0].get("title") || "";

        if (groupNotifications.length === 1) {
            newTitle = notificationModel.i18nWithMap('title', {title: title});
        } else {
            newTitle = notificationModel.i18ns("aggregate", groupNotifications.length, title);
            if (unreadCount) {
                newTitle = notificationModel.i18ns("aggregatenew", unreadCount, title) || newTitle;
            }
        }
        mainTitleDiv.text(newTitle || title).attr("title", newTitle);
    }
});
