/**
 * This card represents the main task list view, where a user can add new tasks or see
 * existing ones.
 */
MW.TaskMainAbstractView = Backbone.View.extend({
    tagName: "div",
    className: "mw-tasks card",
    completed: false,

    events: {
        "keypress #new-personal-task" : "createOnEnter",
        "keyup #new-personal-task" : "disableAddTask",
        "input #new-personal-task" : "disableAddTask",
        "click .add-task" : "addPersonalTask",
        "click .view-completed" : "toggleCompleted",
        "click .mw-button-notifications" : "gotoNotifications",

        "click .view-completed-mobile" : "showCompleted",
        "click .view-uncompleted-mobile" : "showUncompleted"
    },

    gotoNotifications: function() {
        this.globalEvents.trigger("showNotificationCard", 1);
    },

    initialize: function(options) {
        this.globalEvents = options.globalEvents;
        this.globalEvents.bind("addPersonalTask", this.addPersonalTask, this);
        this.globalEvents.bind("showTaskDetail", this.showTaskDetail, this);
        this.globalEvents.bind("showTaskByGlobalId", this.showTaskByGlobalId, this);
        this.globalEvents.bind("showTaskCard", this.showTaskCard, this);
        this.globalEvents.bind("showTaskCardBack", this.showCard, this);

        MW.Tasks.bind('fetch', this.loaded, this);
        MW.KeyboardShortcuts.setupContext("task", "showTaskCard", this);

        // Any keypresses should go to the input.
        MW.$(document).bind("keydown", _.bind(this.focusInput, this));
    },

    cleanUp: function(){
        MW.Tasks.off();
    },

    focusInput: function() {
        this.input.focus();
    },

    showTaskByGlobalId: function(globalId) {
        this.showTaskDetail(MW.Tasks.getByGlobalId(globalId));
    },

    showTaskDetail: function(task, instant) {
        var taskDetailView = new MW.TaskDetailView({
            model: task,
            globalEvents: this.globalEvents
        });

        this.renderShowTaskDetailView(taskDetailView, instant);
        taskDetailView.showCard();
    },

    showTaskCard:function(slideTime) {
        this.completed = false;
        this.showCard(slideTime);
    },

    addTodo: function() {
        if (!this.taskList) return; // don't bother adding to DOM if no element exists

        var that = this;
        this.taskList.empty();
        var todoTasks = MW.Tasks.todo();
        if (todoTasks.length === 0) {
           this.renderEmptyTaskTemplate(this.taskList);
        } else {
            // add in all todo tasks
            _.each(todoTasks, function(todoTask) {
                that.addOneTask(todoTask);
            });

            this.makeTasksSortable();
        }
        if (this.viewCompleted) {
            this.viewCompleted.toggle(MW.Tasks.done().length !== 0);
        }
    },

    render: function() {
        var $el = MW.$(this.el);
        this.renderTemplate($el);

        this.input = $el.find("#new-personal-task");
        this.viewCompleted = $el.find(".view-completed");
        this.addTask = $el.find('.add-task');
        this.taskList = $el.find(".mw-tasks-list");

        this.disableAddTask();
        return this;
    },

    showLoading: function() {
        this.taskList.empty();
        var loading = new ConfluenceMobile.Notification(this.taskList);
        loading.showLoading();
    },

    /**
     * Called when the Backbone fetch is complete - show the correct card.
     */
    loaded: function () {
        if (this.completed)
            this.addDone();
        else
            this.addTodo();
    },


    addOneTask: function(item, prepend) {
        MW.$(".mw-no-tasks").hide();
        var view = new MW.TaskView({model:item, globalEvents: this.globalEvents,
            attributes: {
                "task-cid": item.cid
            }});
        if (prepend) {
            this.taskList.prepend(view.render().el);
        } else {
            this.taskList.append(view.render().el);
        }

    },

    createOnEnter : function(e) {
        if (e.which != 13) {
            return;
        }
        this.addPersonalTask();
    },

    addPersonalTask: function() {
        var text = this.input.val();

        if(!MW.$.trim(text)) {
            return;
        }

        if (this.completed) {
            this.showUncompleted();
        }

        var newPersonalTask = this.createPersonalTask(text);
        newPersonalTask.set("displayName", text);
        this.taskList.scrollTop(0); // Scroll to top when task is added.
        this.input.val('');
        this.disableAddTask();
    },

    createPersonalTask: function(text) {
        var newPersonalTask = MW.Tasks.create({
            "application":"com.atlassian.mywork.host.provider",
            "entity":"notes",
            "status": "TODO",
            "title": text
        }, {at: 0});
        this.addOneTask(newPersonalTask, true);

        return newPersonalTask;
    },

    show: function() {
        if(MW.Tasks.hasFetched.state() === "resolved") {
            this.completed ? this.showCompleted() : this.showUncompleted();
        }
    },

    toggleCompleted: function() {
        !this.completed ? this.showCompleted() : this.showUncompleted();
    },

    addDone: function() {
        this.taskList.empty();

        if (MW.Tasks.done().length === 0) {
            this.renderEmptyCompletedTemplate(this.taskList);
        } else {
            var now = new Date();
            var today = new Date(now.getFullYear(), now.getMonth(), now.getDate());
            var yesterday = new Date(now.getFullYear(), now.getMonth(), now.getDate()-1);

            // Stuff completed today
            this.addListWithCategory(MW.Tasks.doneNewerThan(today), AJS.I18n.getText('com.atlassian.mywork.task.completed.today'));

            // Stuff completed yesterday
            this.addListWithCategory(MW.Tasks.doneBetween(yesterday, today), AJS.I18n.getText('com.atlassian.mywork.task.completed.yesterday'));

            // Old stuff (items older than yesterday)
            this.addListWithCategory(MW.Tasks.doneOlderThan(yesterday), AJS.I18n.getText('com.atlassian.mywork.task.completed.older'));
        }
    },

    addListWithCategory: function(list, category) {
        var that = this;
        if (!!list.length) {
            this.taskList.append(MyWork.Templates.completedCategory({category: category}));
            _.each(list, function(item){
                that.addOneTask(item);
            });
        }
    }
});
