/**
 * @module confluence/flag-scroll
 */
define('confluence/flag-scroll', [
    'jquery',
    'ajs',
    'confluence/meta',
    'document'
],
/**
 * Adds a scroll handler for updating the top offset of Flag notifications as the user scrolls.
 * Initially, the top offset is sufficient to prevent Flag notifications from covering the application header
 * and page actions. When the user scrolls the application header off the page, the Flag notifications shift up
 * to reclaim the space.
 */
function(
    $,
    AJS,
    Meta,
    document
) {
    'use strict';

    var TOP_MARGIN = 20;
    var DEFAULT_TOP = 71; // fragile. duplicated from AUI flag CSS. won't auto update
    var ID_FLAG_CONTAINER = 'aui-flag-container'; // duplicated from confluence-flag.js. won't sync.
    var isDocumentationTheme = false;
    var $scrollingSection;

    function setFlagContainerTopOffset() {
        var $flagContainer = $('#' + ID_FLAG_CONTAINER);
        // the flag container does not contain any visible flags
        if (!$flagContainer.find('.aui-flag[aria-hidden="false"]').length) {
            return;
        }

        var windowScrollTop;
        var newFlagContainerTop = 0;

        if (isDocumentationTheme) {
            // place flags 20px below the application header
            newFlagContainerTop = $('#header').height() + TOP_MARGIN;
        } else {
            windowScrollTop = $scrollingSection.scrollTop();
            if (windowScrollTop <= (DEFAULT_TOP - TOP_MARGIN)) {
                // bouncy scrolling makes scrollTop take on negative values. ensure the minimum scrollTop value is 0.
                newFlagContainerTop = DEFAULT_TOP - Math.max(windowScrollTop, 0);
            } else {
                newFlagContainerTop = TOP_MARGIN;
            }
        }

        $flagContainer.css('top', newFlagContainerTop);
    }

    return function($) {
        isDocumentationTheme = $('body').hasClass('theme-documentation');
        if (isDocumentationTheme) {
            $scrollingSection = $('#splitter-content');
        } else {
            $scrollingSection = $(window);
        }
        $scrollingSection.on('scroll', setFlagContainerTopOffset);
        // adjust the flag position anytime a new flag is shown
        document.addEventListener('aui-flag-show', setFlagContainerTopOffset);

        // Dismiss the flags when the user enters the page editor, but not if entering the comment/inline comment editor
        AJS.bind('rte-ready', function() {
            if (Meta.get('content-type') === 'page') {
                $('#' + ID_FLAG_CONTAINER).find('.aui-flag[aria-hidden="false"]').each(function() {
                    this.close();
                });
            }
        });
    };
});

require('confluence/module-exporter').safeRequire('confluence/flag-scroll', function(ConfluenceFlagScroll) {
    'use strict';

    var AJS = require('ajs');

    AJS.toInit(ConfluenceFlagScroll);
});
