/**
 * Manager to store data in localStorage, only if supported by the browser.
 * It ensures that the keys are namespaced with a given prefix to avoid clashing with anything else.
 *
 * @param prefix the prefix to be added to the namespace. i.e. 'atlassian'
 * @param id of the storageManager to be returned. This is used to create a unique namespace for keys.
 * @module confluence/storage-manager
 */
define('confluence/storage-manager', [
    'confluence/meta',
    'confluence/api/logger',
    'window'
], function(
    Meta,
    logger,
    window
) {
    'use strict';

    var ConfluenceStorageManager = function(prefix, id) {
        var user = Meta.get('remote-user');
        var namespace = prefix + '.' + (user ? user + '.' : '') + id;
        var delimiter = '#';
        var localStorageSupported = false;
        var prefixMatch = /\d+#/;
        var getPrefix = function(seconds) {
            var milliseconds = (seconds || 0) * 1000;
            if (!milliseconds) {
                return '';
            }
            return +new Date() + milliseconds + delimiter;
        };
        var getItem = function(key) {
            if (!localStorageSupported) { return null; }
            var match;
            var item = window.localStorage.getItem(namespace + '.' + key);
            if (match = prefixMatch.exec(item)) {
                item = item.replace(match[0], '');
                if (+new Date() > match[0].replace('#', '')) {
                    window.localStorage.removeItem(namespace + '.' + key);
                    return null;
                }
            }
            return item;
        };

        try {
            localStorageSupported = 'localStorage' in window && window.localStorage !== null;
        } catch (e) {
            logger.log('Browser does not support localStorage, Confluence.storageManager will not work.');
        }

        return {
            isLocalStorageSupported: function() {
                return localStorageSupported;
            },
            /**
             * Gets the item stored in local storage for the given key. null is returned if it doesn't exist.
             * Note that this method will always return a string representation of what is stored.
             *
             * @param key
             */
            getItem: getItem,
            /**
             * Returns a boolean to let you know if we contain a key that matches, and has not expired.
             * @param key
             */
            doesContain: function(key) {
                return !!getItem(key);
            },
            /**
             * Gets the item stored in local storage for the given key and returns the boolean value of it.
             * It correctly convert the "true" and "false" strings to return true/false booleans.
             *
             * @param key
             */
            getItemAsBoolean: function(key) {
                var value = getItem(key);
                if (value == 'false') {
                    return false;
                }
                if (value == 'true') {
                    return true;
                }

                return !!value;
            },
            setItem: function(key, value, expire) {
                if (!localStorageSupported) { return; }
                value = getPrefix(expire) + value;

                window.localStorage.setItem(namespace + '.' + key, value);
            },
            /**
             * Writes an entry to local storage if possible. If any errors occur, the write fails and the error is ignored
             * so normal execution continues. No exception is thrown. This is useful when you don't want the normal flow
             * of your script to be interrupted if the write fails due to something like localStorage being full.
             *
             * @since 5.6
             *
             * @param key The key of the entry
             * @param value The value of the entry
             * @param expire The number of seconds for which the entry is valid
             */
            setItemQuietly: function(key, value, expire) {
                try {
                    this.setItem(key, value, expire);
                } catch (e) {
                    logger.debug('Unable to write to localStorage. key: ', key, ', value: ', value, ', cause: ', e);
                }
            },
            removeItem: function(key) {
                if (!localStorageSupported) { return; }

                window.localStorage.removeItem(namespace + '.' + key);
            }
        };
    };

    return ConfluenceStorageManager;
});

require('confluence/module-exporter').exportModuleAsGlobal('confluence/storage-manager', 'AJS.storageManager');
