define('confluence-editor-reliable-save/ajax-login', [
    'confluence/dark-features',
    'confluence/meta',
    'ajs',
    'jquery',
    'document'
], function(
    DarkFeatures,
    Meta,
    AJS,
    $,
    document
) {
    'use strict';

    /**
     * Replaces the regular $.ajax function with an authenticating ajax function
     * @constructor
     * @exports confluence-editor-reliable-save/ajax-login
     */
    function AjaxLogin() {
        if (!DarkFeatures.isEnabled('ajax.login')) {
            return;
        }

        AJS.debug('AJAX login support enabled');

        // State
        var loginActive = false;
        var loginDialog;
        var loginIframe;
        var loginSuccess;
        var pendingXhr = [];

        // Intercept jQuery ajax calls
        var jqAjax = $.ajax;

        // These shouldn't change - so cache them
        var remoteUser = Meta.get('remote-user');
        var baseUrl = Meta.get('base-url');
        var contextPath = Meta.get('context-path');
        var loginCompleteAction = 'ajaxlogincomplete.action';
        var loginUrl = Meta.get('context-path') + '/login.action?os_destination=%2F' + loginCompleteAction;

        function shouldIntercept(url, settings) {
            var urlVal = url && url.url;
            return remoteUser // only do it for logged in users
                    && (!settings || settings && !settings.preventAjaxLogin) // allow specific ajax requests to prevent ajax login
                    && urlVal
                    && (baseUrl && urlVal.indexOf(baseUrl) === 0 || contextPath && urlVal.indexOf(contextPath) === 0) // only intercept requests to the Confluence instance
            ;
        }

        function authenticatingPromise(url, settings, jqxhr) {
            AJS.debug('Using authenticatingPromise');
            // Our fake version of the jquery xhr ;)
            var fakexhr = {
                abort: function() {
                    jqxhr.abort.apply(jqxhr, arguments);
                },
                getAllResponseHeaders: function() {
                    jqxhr.getAllResponseHeaders.apply(jqxhr, arguments);
                },
                getResponseHeader: function() {
                    jqxhr.getResponseHeader.apply(jqxhr, arguments);
                },
                overrideMimeType: function() {
                    jqxhr.overrideMimeType.apply(jqxhr, arguments);
                },
                readyState: jqxhr.readyState,
                setRequestHeader: function(name, value) {
                    jqxhr.setRequestHeader.apply(jqxhr, arguments);
                },
                state: function() {
                    jqxhr.state.apply(jqxhr, arguments);
                },
                status: jqxhr.status,
                statusCode: function() {
                    jqxhr.statusCode.apply(jqxhr, arguments);
                },
                statusText: jqxhr.statusText
            };

            var deferred = $.Deferred();
            deferred.promise(fakexhr);
            fakexhr.success = fakexhr.done;
            fakexhr.error = fakexhr.fail;

            function checkAuthenticated(jqXHR) {
                var userName = jqXHR.getResponseHeader('X-AUSERNAME');
                return remoteUser === userName;
            }

            /* Reruns any queued xhr's */
            function execPendingXhrs(xhrs) {
                for (var i = 0, l = xhrs.length; i < l; i++) {
                    var x = xhrs[i];
                    AJS.debug('Rerunning ajax query to: ' + x.url, x);
                    jqAjax.call($, x.url, x.settings).done(function() {
                        AJS.debug('Rerunning done');
                        x.deferred.resolve(arguments);
                    }).fail(function() {
                        AJS.debug('Rerunning failed');
                        x.deferred.reject(arguments);
                    });
                }
            }

            /* Fallbacks to original callbacks with the original response */
            function execDefaultCallbacks(xhrs) {
                for (var i = 0, l = xhrs.length; i < l; i++) {
                    var x = xhrs[i];
                    AJS.debug('Executing with original response for: ' + x.url, x);
                    x.defaultCallback && x.defaultCallback();
                }
            }

            function authenticateAndReAjax(defaultCallback) {
                pendingXhr.push({
                    deferred: deferred,
                    url: url,
                    settings: settings,
                    defaultCallback: defaultCallback
                });
                if (loginActive) {
                    return;
                }

                loginSuccess = false;

                if (!loginDialog) {
                    loginDialog = new AJS.Dialog(840, 700);
                    loginDialog.addPanel('ajax-login', 'ajax-login-content', 'ajax-login-panel', 1);
                    var panel = $('.ajax-login-panel');
                    loginIframe = $('<iframe></iframe>');
                    loginIframe.bind('load', function(e) {
                        var path = loginIframe[0].contentDocument.location.pathname;
                        if (path && path.indexOf('/' + loginCompleteAction) >= 0) {
                            // Logged in
                            loginActive = false;
                            loginSuccess = true;
                            loginDialog.hide();
                            loginIframe.attr('src', 'about:blank');
                        }
                    });

                    panel.append(loginIframe);

                    $(document).bind('hideLayer', function(e, type, popup) {
                        if (type === 'popup' && popup === loginDialog.popup) {
                            AJS.debug('hiding login dialog');
                            if (loginSuccess) {
                                // re-exec xhr's pending
                                execPendingXhrs(pendingXhr);
                                pendingXhr = [];
                            } else {
                                // didn't login (i.e. cancelled via escape) - clear queue of outstanding xhr
                                execDefaultCallbacks(pendingXhr);
                                pendingXhr = [];
                            }
                        }
                    });
                }
                loginIframe.attr('src', loginUrl);

                loginDialog.show();
                loginActive = true;
            }

            function interceptResponse(jqxhr, callback, callbackContext, callbackArgs) {
                function defaultCallback() {
                    fakexhr.readyState = jqxhr.readyState;
                    fakexhr.status = jqxhr.status;
                    fakexhr.statusText = jqxhr.statusText;
                    if (jqxhr.responseXML) {
                        fakexhr.responseXML = jqxhr.responseXML;
                    } else {
                        fakexhr.responseText = jqxhr.responseText;
                    }
                    callback.apply(callbackContext, callbackArgs);
                }
                if (checkAuthenticated(jqxhr)) {
                    defaultCallback();
                } else {
                    authenticateAndReAjax(defaultCallback);
                }
            }

            jqxhr.done(function(data, textStatus, jqxhr) {
                interceptResponse(jqxhr, deferred.resolve, deferred, arguments);
            });
            jqxhr.fail(function(jqxhr, textStatus, errorThrown) {
                interceptResponse(jqxhr, deferred.reject, deferred, arguments);
            });

            return fakexhr;
        }

        /**
         * Replacement for $.ajax that deals with loss of user sessions.
         */
        function _authenticatingAjax(url, settings) {
            var jqxhr = jqAjax.call($, url, settings);
            if (shouldIntercept(url, settings)) {
                return authenticatingPromise(url, settings, jqxhr);
            }
            return jqxhr;
        }

        $.ajax = _authenticatingAjax;
    }

    return AjaxLogin;
});

require('confluence/module-exporter').safeRequire('confluence-editor-reliable-save/ajax-login', function(AjaxLogin) {
    'use strict';

    require('ajs').toInit(AjaxLogin);
});
