/**
 * @module confluence-editor-reliable-save/reliable-save
 */
define('confluence-editor-reliable-save/reliable-save', [
    'ajs',
    'confluence/legacy',
    'confluence/meta',
    'underscore',
    'jquery',
    'window',
    'document',
    'confluence/api/constants',
    'confluence-editor/editor/page-editor-message',
    'confluence-editor/editor/page-editor-quit-dialog'
], function(
    AJS,
    Confluence,
    Meta,
    _,
    $,
    window,
    document,
    CONSTANTS,
    Message,
    QuitDialog
) {
    'use strict';

    var cleanupFunctions = [];

    // used to identify the source of and the action triggering an entity.stop or start
    var PUBLISH = 'confluence.editor.publish';

    // is used to indicate critical error state, which cannot be recovered in the current editing session
    // depends on this flag, reliable-save behaves differently (see comments below)
    // Please refer to the jsdoc in synchrony-handlers.ls in collab-editor plugin (handleError function) for more
    // details about how eviction errors are processed on frontend
    var unrecoverableEditorError = false;

    function isSharedDraftsEnabled() {
        return Meta.get('shared-drafts');
    }

    function isNewPage() {
        return Meta.getBoolean('new-page');
    }

    function enableBar() {
        Confluence.Editor.UI.toggleSavebarBusy(false);
    }

    function displayGenericSaveMessage(error) {
        var errorMessage = AJS.I18n.getText('editor.offline.save.error.generic');
        Message.closeMessages(['generic-error']);
        Message.handleMessage('generic-error', {
            type: 'error',
            message: errorMessage
        }, bindEventsToDraftDialog);

        AJS.logError('Generic error: ' + JSON.stringify(error));
    }

    function bindEventsToDraftDialog() {
        var $draft = $('#draft-messages');
        if ($draft.length > 0) {
            if ($draft.is(':visible')) {
                AJS.Confluence.Analytics.publish('rte.notification.draft');
            }

            $draft.find('a.use-draft').click(function(e) {
                e.stopPropagation();
                e.preventDefault();
                Confluence.Editor.Drafts.useDraft();
                AJS.Confluence.Analytics.publish('rte.notification.draft.resume');
            });
            $draft.find('a.discard-draft').click(function(e) {
                e.stopPropagation();
                e.preventDefault();
                SafeSave.Draft.discardDraft(Meta.get('page-id'), Meta.get('existing-draft-id'))
                    .done(SafeSave.Draft.onSuccessDiscardDraft)
                    .fail(SafeSave.Draft.onErrorDiscardDraft);
            });
        }
    }

    function bindEventsToRestoreFlag() {
        $('#editor-restore-title-link').click(function(e) {
            e.stopPropagation();
            e.preventDefault();
            $('#content-title').val(Meta.get('latest-published-page-title'));
            Message.closeMessages(['rename-during-limited-mode']);
        });
    }

    function transitionToReadOnlyMode() {
        Meta.set('access-mode', 'READ_ONLY');
        Message.closeMessages(['read-only-mode']);
        Message.handleMessage('read-only-mode', {
            title: AJS.I18n.getText('editor.read.only.mode.error.message.title'),
            type: 'error',
            message: AJS.I18n.getText('editor.read.only.mode.error.message.text')
        });
        // Re-enable all editor buttons
        Confluence.Editor.UI.setButtonsState(true);
        Confluence.Editor.UI.toggleSavebarBusy(false);
        // disable Publish button
        Confluence.Editor.UI.setButtonState(false, Confluence.Editor.UI.saveButton);
        Confluence.Editor.UI.setButtonState(false, Confluence.Editor.UI.cancelButton);
    }

    function displayNoAuthorizedSaveMessage(triggerInvalidXsrfToken) {
        var errorMessage = AJS.I18n.getText('editor.offline.save.noauthorized');
        Message.closeMessages(['noauthorized']);
        Message.handleMessage('noauthorized', {
            title: AJS.I18n.getText('editor.offline.save.noauthorized.title'),
            type: 'error',
            message: errorMessage
        }, bindEventsToDraftDialog);
        if (triggerInvalidXsrfToken) {
            AJS.trigger('rte.safe-save.invalid-xsrf-token');
        }
    }

    function displayServerOfflineSaveMessage() {
        var errorMessage = AJS.I18n.getText('editor.offline.save.error');
        Message.closeMessages(['server-offline']);
        Message.handleMessage('server-offline', {
            type: 'error',
            message: errorMessage
        }, bindEventsToDraftDialog);
    }

    var SafeSave = {};

    SafeSave.Draft = {
        discardDraft: function(pageId, draftId) {
            var draftData = {
                draftId: draftId,
                pageId: pageId,
                type: Meta.get('draft-type'),
                spaceKey: Meta.get('space-key')
            };

            return $.ajax({
                type: 'DELETE',
                url: CONSTANTS.CONTEXT_PATH + '/rest/tinymce/1/drafts/discard',
                data: $.toJSON(draftData),
                contentType: 'application/json',
                dataType: 'json'
            });
        },

        onSuccessDiscardDraft: function() {
            // Set draft-id to zero when discard draft in edit mode to make sure
            // if there is no draft at the time save button is clicked later on, we update content directly
            if (!isNewPage()) {
                Meta.set('draft-id', '0');
            }

            Message.closeMessages(['draft-message']);
            Message.handleMessage('discarding-successfull', {
                type: 'info',
                message: AJS.I18n.getText('editor.page.draft.discarding.info'),
                close: 'auto'
            }, bindEventsToDraftDialog);
            AJS.Confluence.Analytics.publish('rte.notification.draft.discard');
        },
        onErrorDiscardDraft: function(errorData) {
            switch (errorData.status) {
            case 403:
                displayNoAuthorizedSaveMessage(true);
                break;
            case 404:
                Message.handleMessage('draft-deleted', {
                    type: 'info',
                    message: AJS.I18n.getText('editor.offline.draft.is.deleted')
                }, bindEventsToDraftDialog);
                break;
            case 405:
                transitionToReadOnlyMode();
                break;
            case 422:
                Message.handleMessage('discarding-invalid', {
                    type: 'error',
                    message: AJS.I18n.getText('editor.offline.draft.is.invalid')
                }, bindEventsToDraftDialog);
                break;
            default:
                Message.handleMessage('discarding-error', {
                    type: 'error',
                    message: AJS.I18n.getText('discard.draft.unknown.error')
                }, bindEventsToDraftDialog);
                break;
            }
        }
    };

    SafeSave._internal = Confluence.SafeSafe && Confluence.SafeSave._internal ? Confluence.SafeSave._internal : {};

    // successful http response with no validation errors
    SafeSave._internal.onSuccessfulResponse = function(responseJSON) { // exposed for testing purposes
        $('#rte-button-overwrite').unbind('click.overwrite');

        var data = {
            dataType: 'json',
            contentId: Meta.get('content-id'),
            draftType: Meta.get('draft-type')
        };

        AJS.safe.post(CONSTANTS.CONTEXT_PATH + '/json/stopheartbeatactivity.action', data, function() {
            AJS.log('Stop heartbeat activity on', data.draftType, 'id', data.contentId);
        }, 'json').fail(function(xhr, status, err) {
            AJS.logError('Server error on stop heartbeat activity request:');
            AJS.log(err);
        }).always(function() {
            var newLocation = responseJSON._links.webui;
            if (!newLocation) {
                Confluence.Editor.isPublishing(false);
                enableBar();
                return;
            }
            if (newLocation.indexOf('/') !== 0) { // relative address to current path
                window.location = newLocation;
            } else { // full virtual path, needs to be joined to context path.
                window.location = CONSTANTS.CONTEXT_PATH + newLocation;
            }
        });
    };

    /**
     * For testing purposes only
     * @private
     */
    SafeSave.resetUnrecoverableEditorError = function() {
        unrecoverableEditorError = false;
    };

    SafeSave.initialize = function() {
        // Makes sure we are not in a comment
        if ($('#editpageform').length === 0 && $('#createpageform').length === 0) {
            return;
        }

        // errors in body of 400 status response
        var allowedSaveErrorMessage = {
            duplicatedTitle: 'A page with this title already exists',
            titleTooLong: 'Title cannot be longer than 255 characters.',
            versionCommentTooLong: 'Version comment exceeds maximum length of 100000 characters',
            publishNewDraftDeprecated: 'Unsupported call to publishNewDraft',
            existingDraftNotFound: 'Could not find existing draft, perhaps you\'re trying to publish a personal draft?',
            renameDuringLimitedMode: 'Unable to perform a page rename when limited mode is enabled',
            utf8ValidationFailed: 'Unsupported character found in content: ',
            createdDateInFuture: 'CREATED_DATE_IN_FUTURE',
            pageHasBeenConvertedToBlog: 'Type is required and must match existing entity type of blogpost, but received page',
            draftParent: 'Parent page is either deleted or not published.'
        };

        var draftData = {
            existingDraftId: Meta.get('existing-draft-id') ? Meta.get('existing-draft-id') : 0,
            pageId: Meta.get('page-id'),
            type: Meta.get('draft-type'),
            spaceKey: Meta.get('space-key')
        };

        if (Meta.get('show-draft-message') === true) {
            if ($('#conflict-diffs').length > 0) {
                return;
            }
            $.ajax({
                type: 'GET',
                url: CONSTANTS.CONTEXT_PATH + '/rest/tinymce/1/drafts/message',
                data: draftData,
                contentType: 'application/json',
                dataType: 'text json',
                success: function(data) {
                    if (data && data.draftData) {
                        var content = Confluence.Templates.Editor.Reliable.draftMessage({
                            existingDraft: data.draftData,
                            conflictFound: data.conflictFound,
                            mergeRequired: data.mergeRequired,
                            isNewPage: data.newPage,
                            pageId: Meta.get('page-id'),
                            spaceKey: Meta.get('space-key')

                        });
                        Message.handleMessage('draft-message', {
                            type: 'info',
                            message: content
                        }, bindEventsToDraftDialog);
                    }
                }
            });
        }

        // This is only a temporary solution for legacy drafts: CONFDEV-37357
        // When reliable save is enabled, after saving page with conflicts case and automatically falling back to xwork actions,
        // if there is any action error messages displayed (like token validation...), we restore the default save mechanism (form submit => go through xwork action)
        // for save button to ensure merging conflict will be handled properly after that if user click save again.
        var isActionErrorMessagesDisplayed = $('#editor-notifications-container #all-messages .aui-message-error').length > 0;

        if (!isSharedDraftsEnabled()) {
            if (isActionErrorMessagesDisplayed) {
                Confluence.Editor.restoreDefaultSave();
            } else {
                Confluence.Editor.overrideSave(onSave);
            }
        } else {
            QuitDialog.init({ saveHandler: onSave, cancelErrorHandler: showAppropriateMessageOnSavingError });
            Confluence.Editor.overrideSave(QuitDialog.process);
        }

        $('#rte-button-overwrite').bind('click.overwrite', onSave);

        var retries = 0;
        var MAX_RETRIES = 3;
        var RETRY_DELAY = 1000;
        var retrySaveTimeouts = [];

        function clearAllTimeouts() {
            while (retrySaveTimeouts.length) {
                clearTimeout(retrySaveTimeouts.shift());
            }
        }

        function showAppropriateMessageOnSavingError(xhr) {
            var supplementaryCharacter;
            enableBar();
            switch (xhr.status) {
            case 400:
                Message.closeMessages(['empty-title', 'duplicate-title', 'title-too-long', 'version-comment-too-long', 'legacy-draft-deprecated',
                    'utf8-validation-failed', 'created-date-input-validation', 'draft-parent']);

                if (xhr.responseText.indexOf(allowedSaveErrorMessage.duplicatedTitle) >= 0) {
                    Message.handleMessage('duplicate-title', {
                        type: 'error',
                        message: AJS.I18n.getText('editor.offline.save.page.exist.content', AJS.escapeHtml($('#content-title').val()))
                    });
                } else if (xhr.responseText.indexOf(allowedSaveErrorMessage.titleTooLong) >= 0) {
                    Message.handleMessage('title-too-long', {
                        type: 'error',
                        message: AJS.I18n.getText('editor.title.too.long')
                    }, bindEventsToDraftDialog);
                } else if (xhr.responseText.indexOf(allowedSaveErrorMessage.versionCommentTooLong) >= 0) {
                    Message.handleMessage('version-comment-too-long', {
                        type: 'error',
                        message: AJS.I18n.getText('editor.version.comment.too.long')
                    }, bindEventsToDraftDialog);
                }
                else if (xhr.responseText.indexOf(allowedSaveErrorMessage.draftParent) >= 0) {
                    Message.closeMessages(['draft-parent']);
                    Message.handleMessage('draft-parent', {
                        title: AJS.I18n.getText('editor.parent.is.draft.error.title'),
                        type: 'error',
                        message: AJS.I18n.getText('editor.parent.is.draft.error')
                    }, bindEventsToDraftDialog);
                } else if (
                    xhr.responseText.indexOf(allowedSaveErrorMessage.publishNewDraftDeprecated) >= 0 // TODO SHARED DRAFTS Should be removed once we 100% collab
                        || xhr.responseText.indexOf(allowedSaveErrorMessage.existingDraftNotFound) >= 0) {
                    // CONFDEV-47278 in case extra content is added after CE enabled, we can't migrate them to the new shared draft
                    // so we should ask user to copy their content and refresh the editor instead.
                    if (Meta.get('new-page') && isSharedDraftsEnabled() && !Confluence.Editor.hasContentChanged()) {
                        AJS.trigger('rte.legacy-draft-can-be-migrated');

                        Message.handleMessage('legacy-draft-deprecated', {
                            type: 'error',
                            message: AJS.I18n.getText('editor.draft.can.be.migrated',
                                CONSTANTS.CONTEXT_PATH + '/pages/resumedraft.action?draftId=' + Meta.get('draft-id'))
                        }, bindEventsToDraftDialog);
                    } else {
                        AJS.trigger('rte.legacy-draft-cannot-be-migrated');

                        Message.handleMessage('legacy-draft-deprecated', {
                            type: 'error',
                            message: AJS.I18n.getText('editor.page.legacy.draft.deprecated')
                        }, bindEventsToDraftDialog);
                    }
                } else if (xhr.responseText.indexOf(allowedSaveErrorMessage.utf8ValidationFailed) >= 0) {
                    supplementaryCharacter = xhr.responseJSON.message.split(allowedSaveErrorMessage.utf8ValidationFailed)[1];
                    Message.handleMessage('utf8-validation-failed', {
                        title: AJS.I18n.getText('mysql.utf8.content.validation.failed.title'),
                        type: 'error',
                        message: AJS.I18n.getText('mysql.utf8.content.validation.failed.message', supplementaryCharacter)
                    }, bindEventsToDraftDialog);
                } else if (xhr.responseText.indexOf(allowedSaveErrorMessage.createdDateInFuture) >= 0) {
                    Message.handleMessage('created-date-input-validation', {
                        type: 'error',
                        message: AJS.I18n.getText('news.date.in.future')
                    });
                } else if (xhr.responseText.indexOf(allowedSaveErrorMessage.pageHasBeenConvertedToBlog) >= 0) {
                    Message.handleMessage('page-has-been-converted-to-blog', {
                        type: 'error',
                        message: AJS.I18n.getText('page.has.been.converted.to.blogpost')
                    });
                } else {
                    displayGenericSaveMessage(xhr);
                }
                break;
            case 403:
                displayNoAuthorizedSaveMessage(true);
                break;
            case 404:
                Message.closeMessages(['page-not-accessible', 'noauthorized']);
                Message.handleMessage('page-not-accessible', {
                    title: AJS.I18n.getText('editor.page.can.not.be.accessed.title'),
                    type: 'error',
                    message: AJS.I18n.getText('editor.page.can.not.be.accessed.message', Meta.get('space-key'))
                }, bindEventsToDraftDialog);
                break;
            case 405:
                transitionToReadOnlyMode();
                break;
                // 409
            case 410:
                Message.closeMessages(['page-deleted']);
                Message.handleMessage('page-deleted', {
                    title: AJS.I18n.getText('editor.page.is.trashed.title'),
                    type: 'error',
                    message: AJS.I18n.getText('editor.page.is.trashed.message', Meta.get('space-key'))
                }, bindEventsToDraftDialog);
                break;
            case 413:
                Message.closeMessages(['page-too-big']);
                Message.handleMessage('page-too-big', {
                    type: 'error',
                    message: AJS.I18n.getText('editor.page.is.too.big')
                }, bindEventsToDraftDialog);
                break;
            case 0:
            case 500:
            case 503:
                displayServerOfflineSaveMessage();
                break;
            case 501:
                if (xhr.responseText.indexOf(allowedSaveErrorMessage.renameDuringLimitedMode) >= 0) {
                    Message.handleMessage('rename-during-limited-mode', {
                        type: 'error',
                        message: AJS.I18n.getText('editor.title.rename.during.limited.mode')
                    }, bindEventsToRestoreFlag);
                } else {
                    displayGenericSaveMessage(xhr);
                }
                break;
            default:
                displayGenericSaveMessage(xhr);
                break;
            }
        }

        function onSave(e) {
            // any colored text entered in the current session and was not converted to tokens is
            // using the new themes. Calling this function as this point will make use of the theme information and convert new
            // colors based on that
            if (AJS.Rte.getEditor().hasPlugin('colorTransformer')) {
                const background = getComputedStyle(AJS.Rte.getEditor().getBody()).backgroundColor;
                AJS.Rte.getEditor().plugins.colorTransformer.convertHardCodedColorsToTokens(AJS.Rte.getEditor().getBody(), background);
            }
            AJS.trigger('synchrony.stop', { id: PUBLISH });

            if (e) {
                e.preventDefault();
            }

            function computeParentData() {
                var parentData = {};
                parentData.id = Meta.get('parent-page-id') || '0';
                parentData.type = Meta.get('content-type');

                var parentPageIdInput = Meta.get('parent-page-id');

                // If space has been changed
                if (!parentPageIdInput || jsonData.space.key !== Meta.get('space-key')) {
                    parentData.id = '0';
                }

                // If parent page id is not empty and has been changed
                if (parentPageIdInput && parentData.id !== parentPageIdInput) {
                    parentData.id = ($('#parentPageString').val() === Meta.get('from-page-title'))
                        ? parentData.id
                        : parentPageIdInput;
                }

                return parentData;
            }

            // Run any cleanup functions that have been registered.
            function cleanupContent(content) {
                cleanupFunctions.forEach(function(cleanupFunc) {
                    content = cleanupFunc(content);
                });

                return content;
            }

            function retrySave(xhr) {
                if (retries < MAX_RETRIES) {
                    retries++;

                    // remember timeout ID so we can clear all timeout later
                    retrySaveTimeouts.push(setTimeout(function() {
                        onSave();
                    }, RETRY_DELAY));
                } else {
                    // TODO SHARED-DRAFTS. Once content reconciliation is working we could ask the server to retry
                    // reconciliation at this point.
                    AJS.trigger('analyticsEvent', { name: 'editor.save.error.conflict' });
                    retries = 0;

                    Message.handleMessage('page-conflict', {
                        title: AJS.I18n.getText('editor.page.conflict.title'),
                        type: 'error',
                        message: AJS.I18n.getText('editor.page.conflict.message', jsonData.space.key)
                    }, bindEventsToDraftDialog);

                    afterFailedSaveAttempt(xhr);
                }
            }

            function afterFailedSaveAttempt(xhr) {
                AJS.trigger('rte.safe-save.error', { status: xhr.status });
            }

            var $contentTitle = $('#content-title');
            if ($contentTitle.hasClass('placeholded') || $contentTitle.val().trim() === '') {
                AJS.trigger('rte.safe-save.error');
                AJS.trigger('synchrony.start', { id: PUBLISH });

                Message.closeMessages(['title-too-long', 'duplicate-title']);
                Message.handleMessage('empty-title', {
                    title: AJS.I18n.getText('editor.title.is.empty.title'),
                    type: 'error',
                    message: AJS.I18n.getText('editor.title.is.empty')
                }, bindEventsToDraftDialog);
                enableBar();
                return;
            }

            Confluence.Editor.Drafts.unBindUnloadMessage();

            var action;
            var jsonData = {};
            var draftId = Meta.get('draft-id');
            var contentId = Meta.get('content-id');
            var url = CONSTANTS.CONTEXT_PATH + '/rest/api/content';
            var sourceTemplateId = $('#sourceTemplateId').val();

            jsonData.status = 'current';
            jsonData.title = $contentTitle.val();
            jsonData.space = { key: Meta.get('space-key') };
            jsonData.body = {
                editor: {
                    value: cleanupContent(AJS.Rte.getEditor().getContent()),
                    representation: 'editor'
                }
            };

            if (sourceTemplateId) {
                jsonData.extensions = { sourceTemplateId: sourceTemplateId };
            }

            function pad2(num) {
                return (num < 10 ? '0' : '') + num;
            }

            // offset - the difference in milliseconds from UTC time
            function getDisplayOffset(offsetStr) {
                let offset = parseInt(offsetStr, 10);
                if (Number.isNaN(offset)) { return ''; }

                const sign = offset < 0 ? '-' : '+';
                offset = Math.abs(offset);
                // Keep minutes only.
                // There are no timezones with seconds and milliseconds offsets.
                offset = Math.floor(offset / 1000 / 60);

                const minutes = offset % 60;
                offset = (offset - minutes) / 60;

                const hours = offset;

                return sign + pad2(hours) + ':' + pad2(minutes);
            }

            const postingDate = $('#PostingDate')[0];
            let publishedDate = postingDate && postingDate.value;

            if (draftData.type === 'blogpost' && publishedDate) {
                const postingTime = $('#PostingTime')[0];
                let publishedTime = postingTime && postingTime.value;

                if (!publishedTime) {
                    const localTime = new Date();
                    publishedTime = pad2(localTime.getHours()) + ':' +
                            pad2(localTime.getMinutes());
                }

                publishedTime += ':00'; // add seconds part

                // Add current time; create ISO 8601 compatible string
                publishedDate += 'T' + publishedTime +
                        getDisplayOffset(Meta.get('user-timezone-offset'));

                if (Number.isNaN(new Date(publishedDate).getDate())) {
                    Message.closeMessages(['created-date-input-validation']);
                    Message.handleMessage('created-date-input-validation', {
                        type: 'error',
                        message: AJS.I18n.getText('page.posting.date.invalid.general')
                    });
                    enableBar();
                    return;
                }

                // check if the date is older than tomorrow
                // eslint-disable-next-line vars-on-top
                var thisTimeTomorrow = new Date();
                // eslint-disable-next-line vars-on-top
                var createdDate = new Date(publishedDate);

                thisTimeTomorrow.setDate(thisTimeTomorrow.getDate() + 1);
                if (createdDate > thisTimeTomorrow) {
                    Message.closeMessages(['created-date-input-validation']);
                    Message.handleMessage('created-date-input-validation', {
                        type: 'error',
                        message: AJS.I18n.getText('news.date.in.future')
                    });
                    enableBar();
                    return;
                }

                jsonData.history = {
                    createdDate: createdDate.toISOString()
                };
            }

            // @Deprecated: `POST` is issued only for legacy drafts
            if (isNewPage() && !Meta.get('shared-drafts')) {
                action = 'POST';

                if (Meta.get('is-blueprint-page')) {
                    url = url + '/blueprint/instance/' + draftId;
                }

                url += '?status=draft';

                jsonData.id = draftId;
                jsonData.type = Meta.get('content-type');
                jsonData.body.editor.content = { id: draftId };
            } else {
                action = 'PUT';

                if (isNewPage() && Meta.get('is-blueprint-page')) {
                    url += '/blueprint/instance';
                }

                url = url + '/' + contentId;

                // `draft-id` is content id with `draft` status for shared drafts
                if (isNewPage() && isSharedDraftsEnabled()) {
                    jsonData.id = draftId;
                    jsonData.body.editor.content = { id: draftId };
                } else {
                    jsonData.id = Meta.get('page-id');
                    jsonData.body.editor.content = { id: Meta.get('page-id') };
                }

                // if there is no draft at the time save button is clicked, we update content directly
                if (draftId === '0') {
                    url += '?status=current';
                } else {
                    url += '?status=draft';
                }

                url += '&asyncReconciliation=true';

                var versionNumber = Meta.getNumber('page-version') || 0;

                jsonData.type = Meta.get('content-type');
                jsonData.version = {
                    number: versionNumber + 1,
                    message: $('#versionComment').val(),
                    minorEdit: !$('#notifyWatchers').is(':checked'),
                    syncRev: $('#syncRev').val()
                };
            }

            var parentData = computeParentData();
            if (parentData.id !== '0') {
                jsonData.ancestors = [parentData];
            }

            // Check if we are currently saving a draft and wait for it to be saved to
            // avoid race conditions in which finishes saving first.
            // We want a publish to always occur after a draft is saved, and never a
            // draft saved after the publish
            var draftSavingPromise = Confluence.Editor.Drafts.getDraftSavingPromise();
            if (draftSavingPromise) {
                draftSavingPromise.always(_makeRequest);
            } else {
                _makeRequest();
            }

            function _makeRequest() {
                // We don't want drafts to be saved if the user has initiated a publish.
                Confluence.Editor.isPublishing(true);
                $.ajax({
                    type: action,
                    url: url,
                    contentType: 'application/json; charset=utf-8',
                    dataType: 'json',
                    data: JSON.stringify(jsonData),
                    success: function(data) {
                        if (jsonData.type === 'page' || jsonData.type === 'blogpost') {
                            AJS.trigger('analytics', {
                                name: 'confluence.editor.close',
                                data: { source: 'publishButton' }
                            });
                        }
                        AJS.trigger('rte.safe-save.success', data);
                        SafeSave._internal.onSuccessfulResponse(data);
                    },
                    error: function(xhr) {
                        Confluence.Editor.isPublishing(false);
                        enableBar();
                        var retrying = false;
                        switch (xhr.status) {
                        case 409:
                            if (isSharedDraftsEnabled()) {
                                retrying = true;
                                retrySave(xhr);
                            } else {
                                // In case of having conflicts when saving, restore the default save mechanism (form submit)
                                // then rely on XworkAction for displaying error message and merging conflicts because of
                                // at the moment ContentAPI doesn't do the work merging conflicts, it only throws ConflictException
                                Confluence.Editor.restoreDefaultSave();
                                Confluence.Editor.UI.saveButton.click();
                            }
                            break;
                        case 0:
                        case 500:
                        case 503:
                            // CONFDEV-43951
                            clearAllTimeouts();
                            showAppropriateMessageOnSavingError(xhr);
                            break;
                        default:
                            showAppropriateMessageOnSavingError(xhr);
                            break;
                        }
                        if (!retrying) {
                            afterFailedSaveAttempt(xhr);
                        }
                        AJS.trigger('synchrony.start', { id: PUBLISH });
                    }
                });
            }
        }

        // -----------------------------
        // Heartbeat binding. Heartbeating takes care of enabling/disabling the save bar.
        // -----------------------------

        AJS.bind('rte.heartbeat-error', function(sender, err) {
            switch (err.status) {
            case 401:
            case 403:
                if (!Message.isDisplayed(['page-not-accessible'])) {
                    displayNoAuthorizedSaveMessage(false);
                }
                break;
            case 405:
                if (err.responseText) {
                    var data = JSON.parse(err.responseText);
                    if (!Message.isDisplayed(['read-only-mode']) && data.reason === 'READ_ONLY') {
                        transitionToReadOnlyMode();
                    }
                    break;
                }
                // fall through
            case 0:
            case 404:
            case 500:
            case 503:
                if (!Confluence.Editor.metadataSyncRequired()) {
                    displayServerOfflineSaveMessage();
                }
                break;
            default:
                AJS.logError('Heartbeat action error: ' + JSON.stringify(err));
            }
        });

        AJS.bind('rte.heartbeat', function(sender) {
            var isDisplayed = false;
            _.each(Message.displayedErrors(), function(error) {
                if (_.contains(['noauthorized', 'server-offline', 'page-not-accessible', 'read-only-mode'], error)) {
                    isDisplayed = true;
                    Message.closeMessages([error]);
                }
            });

            var isReadOnly = Meta.get('access-mode') === 'READ_ONLY';
            if (isReadOnly) {
                Meta.set('access-mode', 'READ_WRITE');
            }

            if (isDisplayed || isReadOnly) {
                Message.handleMessage('reconnect', {
                    type: 'info',
                    title: AJS.I18n.getText('editor.offline.save.info.title'),
                    message: AJS.I18n.getText('editor.offline.save.info.content'),
                    close: 'auto'
                }, bindEventsToDraftDialog);
            }
            // this code enables publish button on heartbeat
            // but according to the new requirements, publish button has to be disabled if synchronisation error
            // has happened (due to history eviction - pls. see CONFSRVDEV-9378 for more details)
            // that's why !unrecoverableEditorError check was added to this statement
            // Please refer to the jsdoc in synchrony-handlers.ls in collab-editor plugin (handleError function) for more
            // details about how eviction errors are processed on frontend
            if (!Confluence.Editor.UI.isButtonEnabled(Confluence.Editor.UI.saveButton) && !unrecoverableEditorError) {
                Confluence.Editor.UI.setButtonState(true, Confluence.Editor.UI.saveButton);
            }
            if (!Confluence.Editor.UI.isButtonEnabled(Confluence.Editor.UI.cancelButton) && !unrecoverableEditorError) {
                Confluence.Editor.UI.setButtonState(true, Confluence.Editor.UI.cancelButton);
            }
        });

        AJS.bind('synchrony.history.evicted', function() {
            unrecoverableEditorError = true;
        });
    };

    SafeSave.registerCleanupFunction = function(cleanupFunction) {
        cleanupFunctions.push(cleanupFunction);
    };

    return SafeSave;
});

require('confluence/module-exporter').safeRequire('confluence-editor-reliable-save/reliable-save', function(SafeSave) {
    'use strict';

    var AJS = require('ajs');
    var Meta = require('confluence/meta');

    // CONFDEV-38205
    var isCreatingPageFromRoadMapBar = window.document.referrer.indexOf('createDialog=true&flashId') > 0;

    if (Meta.get('remote-user') === ''
        || isCreatingPageFromRoadMapBar) {
        // DO NOT BIND MODULE OR EXPOSE GLOBALS
    } else {
        var Confluence = require('confluence/legacy');

        AJS.bind('rte.init.ui', function() {
            SafeSave.initialize();

            Confluence.Editor = Confluence.Editor || {};
            Confluence.Editor.SafeSave = Confluence.Editor.SafeSave || {};
            Confluence.Editor.SafeSave.Draft = SafeSave.Draft;
            Confluence.Editor.SafeSave._internal = SafeSave._internal || {};
        });
    }
});
