// eslint-disable-next-line @atlassian/confluence-server/matching-tests,@atlassian/confluence-server/web-resource-audit
define('confluence/attachments', [
    'ajs',
    'confluence/templates',
    'confluence/api/ajax',
    'confluence/api/constants',
    'confluence/api/event',
    'confluence/api/logger',
    'confluence/dark-features',
    'confluence/flag',
    'confluence/meta',
    'confluence/message-controller',
    'wrm/context-path'
], function(
    AJS,
    Templates,
    SafeAjax,
    CONSTANTS,
    Event,
    logger,
    DarkFeatures,
    Flag,
    Meta,
    MessageController,
    contextPath
) {
    'use strict';

    var BULK_DELETE_ATTACHMENT_FEATURE_FLAG_KEY = 'confluence.attachments.bulk.delete';

    /** @memberOf module:confluence/attachments */
    var AttachmentsComponent = {
        showOlderVersions: function($) {
            $('.attachment-history a').click(function(e) {
                var attachmentTable = $(this).parents('table.attachments');
                // Attachment table row ID is in the format attachment-${attachmentId} e.g. attachment-2147713048.
                // To get the attachment ID, we need to strip the prefix "attachment-" out of the table row element ID.
                var attachmentId = $(this)
                    .parents('tr:first')[0]
                    .id.replace('attachment-', '');
                // Use the parent container since there can be multiple macros on the same page
                var historyRows = $('.history-' + attachmentId, attachmentTable);
                $(this).toggleClass('aui-iconfont-chevron-down');
                $(this).toggleClass('aui-iconfont-chevron-right');
                historyRows.toggleClass('hidden');

                $(this).attr('aria-expanded', $(this).hasClass('aui-iconfont-chevron-down'));
                e.stopPropagation();
                return false;
            });
        }
    };

    var defaultFlagOptions = {
        close: 'manual',
        type: 'success',
        stack: 'attachments',
        fifo: true
    };

    // default number of text lines used for dialog height calculation
    var defaultNumberOfTextLines = 2;

    var recentlySubmitted = false;
    var recentlySubmittedTimer;

    function refreshPage() {
        window.location.reload(true);
    }

    function submitHandler(event) {
        clearTimeout(recentlySubmittedTimer);

        if (recentlySubmitted) {
            logger.log('Preventing submit due to recent form submission.');
            event.preventDefault();
        }

        recentlySubmitted = true;

        recentlySubmittedTimer = setTimeout(function() {
            recentlySubmitted = false;
        }, 2000);
    }

    function initialiser($) {
        var $attachmentsForm = $('#upload-attachments');
        $attachmentsForm.on('submit', submitHandler);

        // Show more attach more attachment fields.
        /* eslint-disable vars-on-top */
        var moreAttachmentsLink = $('#more-attachments-link');
        moreAttachmentsLink.click(function(e) {
            $('.more-attachments').removeClass('hidden');
            moreAttachmentsLink.addClass('hidden');
            Event.trigger('attachments.hide-more-attachments-link');
            e.stopPropagation();
            return false;
        });

        AttachmentsComponent.showOlderVersions($);

        /* eslint-disable vars-on-top */
        var templates = Templates.Attachments;

        function attachmentsDeletedSuccessFlag(numberOfDeletedAttachments) {
            var successFlag = new Flag(
                $.extend({}, defaultFlagOptions, {
                    title: AJS.I18n.getText('attachment.remove.notification.title.success', numberOfDeletedAttachments),
                    body: templates.deleteSuccessFlagBody(),
                    extraClasses: 'attachment-deleted-success-flag'
                })
            );

            $('#attachment-remove-success-notification-body-refresh-button').click(function() {
                refreshPage();
            });

            setTimeout(function() {
                successFlag.close();
            }, 8000);
        }

        function attachmentsDeletedErrorFlag(numberOfTotalAttachments) {
            var errorFlag = new Flag(
                $.extend({}, defaultFlagOptions, {
                    title: AJS.I18n.getText('attachment.remove.notification.title.error.all', numberOfTotalAttachments),
                    body: AJS.I18n.getText('attachment.remove.notification.body.error.all'),
                    type: 'error',
                    extraClasses: 'attachment-deleted-error-flag'
                })
            );

            setTimeout(function() {
                errorFlag.close();
            }, 8000);
        }

        function attachmentsDeletedPartialErrorFlag(numberOfDeletedAttachments, numberOfTotalAttachments) {
            var errorFlag = new Flag(
                $.extend({}, defaultFlagOptions, {
                    title: AJS.I18n.getText('attachment.remove.notification.title.error.partial', numberOfDeletedAttachments, numberOfTotalAttachments),
                    body: templates.partialFailedFlagBody(),
                    type: 'error',
                    extraClasses: 'attachment-deleted-partial-error-flag'
                })
            );

            $('#attachment-remove-fail-notification-body-refresh-button').click(function() {
                refreshPage();
            });

            setTimeout(function() {
                errorFlag.close();
            }, 8000);
        }

        function retrieveDataAttributeValueFromParents(elem, dataAttributeName) {
            return $(elem)
                .parents('[' + dataAttributeName + ']')
                .attr(dataAttributeName);
        }

        function retrieveAttachmentFilename(elem) {
            return retrieveDataAttributeValueFromParents(elem, 'data-attachment-filename');
        }

        function getVisibleAttachmentsMetadata() {
            return $('tr[id^=attachment-]')
                .toArray()
                .map(function(row) {
                    return {
                        fileName: row.attributes['data-attachment-filename'].value,
                        id: row.id,
                        isDeletable: $(row).find('.removeAttachmentLink').length > 0,
                        isSelected: $(row).find('[id^=selected-attachment]')[0].checked
                    };
                });
        }

        function retrieveSelectedAttachmentIds() {
            var attachmentAttributes = $('tr[id^=attachment-]')
                .toArray()
                .map(function(row) {
                    return {
                        attachmentId: $(row)[0].id.replace('attachment-', ''),
                        isSelected: $(row).find('[id^=selected-attachment]')[0].checked
                    };
                });
            return attachmentAttributes.filter(function(attachment) {
                return attachment.isSelected;
            }).map(function(attachment) {
                return attachment.attachmentId;
            });
        }

        function retrieveAllDisplayedAttachmentCount() {
            return $('[id^=selected-attachment]').length;
        }

        function retrieveSelectedAttachmentCount() {
            return $('[id^=selected-attachment]').filter(function(index, checkbox) {
                return checkbox.checked;
            }).length;
        }

        function retrieveAttachmentVersion(elem) {
            return retrieveDataAttributeValueFromParents(elem, 'data-attachment-version');
        }

        function postWithXsrfToken(url, data, successCallback, errorCallback) {
            // the token gets added to data -> body, if data is not instantiated the method will just fail silently
            var dataParsed = data || {};
            SafeAjax.ajax({
                type: 'POST',
                url: url,
                data: dataParsed,
                success: successCallback,
                error: errorCallback
            });
        }

        function processInSequence(promiseFactoryArray) {
            return promiseFactoryArray.reduce(
                function(promise, getPromise) {
                    return promise.then(function(result) {
                        return getPromise().then(Array.prototype.concat.bind(result));
                    });
                },
                // eslint-disable-next-line no-undef
                Promise.resolve([])
            );
        }

        /* eslint-disable no-shadow */
        function deleteAttachmentsInBatch(attachmentIds, confirmDialog) {
            var errorMessage = null;
            var numberOfDeletedAttachments = 0;

            var promiseFactoryArray = attachmentIds
                .filter(function(id) {
                    return (id !== undefined && id !== null);
                })
                .map(function(id) {
                    return id.replace('attachment-', '');
                })
                .filter(function(id) {
                    return !Number.isNaN(id);
                })
                .map(function(contentId) {
                    return function() {
                        return fetch(
                            CONSTANTS.CONTEXT_PATH + '/rest/api/content/' + contentId,
                            {
                                method: 'DELETE'
                            })
                            .then(function(response) {
                                if (response.status === 204) {
                                    numberOfDeletedAttachments++;
                                } else if (response.status === 400) {
                                    errorMessage = AJS.I18n.getText('attachment.remove.error.status.bad.request');
                                } else if (response.status === 404) {
                                    errorMessage = AJS.I18n.getText('attachment.remove.error.status.not.found');
                                } else if (response.status === 409) {
                                    errorMessage = AJS.I18n.getText('attachment.remove.error.status.conflict');
                                } else {
                                    errorMessage = AJS.I18n.getText('attachment.remove.error.status.default');
                                }
                            });
                    };
                });

            Event.trigger('analyticsEvent', {
                name: 'confluence.attachments.delete.selected',
                data: { count: attachmentIds.length }
            });

            confirmDialog.disable();

            // loop through attachment ID list, chain request for each attachment
            /* eslint-disable no-undef */
            Promise.resolve(processInSequence(promiseFactoryArray))
                .catch(function() {
                    errorMessage = AJS.I18n.getText('attachment.remove.error.status.network');
                    // returning the error to satisfy linter as it's not read anywhere
                    return errorMessage;
                })
                .finally(function() {
                    // This will run once all requests are finished
                    confirmDialog.enable();

                    if (numberOfDeletedAttachments === attachmentIds.length) {
                        attachmentsDeletedSuccessFlag(numberOfDeletedAttachments);
                    } else if (numberOfDeletedAttachments === 0) {
                        attachmentsDeletedErrorFlag(attachmentIds.length);
                    } else {
                        attachmentsDeletedPartialErrorFlag(numberOfDeletedAttachments, attachmentIds.length);
                    }

                    confirmDialog.remove();
                });
        }

        function createDialog(id) {
            var dialog = AJS.ConfluenceDialog({
                width: 600,
                height: 200,
                id: id
            });
            return dialog;
        }

        /* eslint-disable no-shadow */
        function createDynamicHeightDialog(id, numberOfTextLines) {
            // Dialog height starting from 200px
            // when bulk delete feature is enabled, dialog height will increase another 28px for each additional line
            // which will be capped at 620px at 16+ lines of text content
            var calculatedHeight = 200;
            var dialog;
            var heightForEachLine = 28;
            var maximumHeight = 620;
            if (numberOfTextLines > 1 && numberOfTextLines < 16) {
                calculatedHeight = 200 + (numberOfTextLines - 1) * heightForEachLine;
            } else if (numberOfTextLines >= 16) {
                calculatedHeight = maximumHeight;
            }
            dialog = AJS.ConfluenceDialog({
                width: 600,
                height: calculatedHeight,
                id: id
            });
            return dialog;
        }

        function confirmDialog(url, confirmationTitle, confirmationBody) {
            var confirmDialog;
            var submitButtonText;
            if (DarkFeatures.isEnabled(BULK_DELETE_ATTACHMENT_FEATURE_FLAG_KEY)) {
                confirmDialog = createDynamicHeightDialog(
                    'attachment-removal-confirm-dialog',
                    defaultNumberOfTextLines
                );
                submitButtonText = AJS.I18n.getText('remove.name');
            } else {
                confirmDialog = createDialog('attachment-removal-confirm-dialog');
                submitButtonText = AJS.I18n.getText('ok');
            }
            confirmDialog.addHeader(confirmationTitle);
            confirmDialog.addPanel('', confirmationBody);
            confirmDialog.addSubmit(submitButtonText, function() {
                var errorDialog;
                var successCallback = function() {
                    refreshPage();
                };
                var errorCallback = function(jqXHR) {
                    var messages = [];
                    messages.push(MessageController.parseError(jqXHR));
                    if (jqXHR.responseText) {
                        var response = jqXHR.responseText ? JSON.parse(jqXHR.responseText) : null;
                        if (response.actionErrors) {
                            messages = response.actionErrors;
                        }
                    }
                    if (DarkFeatures.isEnabled(BULK_DELETE_ATTACHMENT_FEATURE_FLAG_KEY)) {
                        errorDialog = createDynamicHeightDialog(
                            'attachment-removal-error-dialog',
                            defaultNumberOfTextLines
                        );
                    } else {
                        errorDialog = createDialog('attachment-removal-error-dialog');
                    }
                    errorDialog.addHeader(templates.removalErrorTitle());

                    errorDialog.addPanel(
                        '',
                        templates.removalErrorBody({
                            messages: messages
                        })
                    );
                    errorDialog.addButton(AJS.I18n.getText('close.name'), function() {
                        refreshPage();
                    });
                    errorDialog.show();
                    errorDialog.updateHeight();
                    confirmDialog.remove();
                };
                postWithXsrfToken(url, null, successCallback, errorCallback);
            });
            confirmDialog.addCancel(AJS.I18n.getText('cancel.name'), function() {
                confirmDialog.remove();
            });
            confirmDialog.show();
        }

        function createUrl(path, search) {
            return CONSTANTS.CONTEXT_PATH + path + search;
        }

        /**
         * Exposes the confirm dialog for external plugins to use.
         *
         * @param {HTMLAnchorElement} removeLink the link that triggers the dialog
         */
        AttachmentsComponent.showRemoveAttachmentConfirmDialog = function(removeLink) {
            var url = createUrl('/json/removeattachment.action', removeLink.search);
            var confirmationTitle;
            var confirmationBody;
            if (DarkFeatures.isEnabled(BULK_DELETE_ATTACHMENT_FEATURE_FLAG_KEY)) {
                confirmationTitle = templates.removalConfirmationTitleV2();
                confirmationBody = templates.removalConfirmationBodyV2();
            } else {
                confirmationTitle = templates.removalConfirmationTitle();
                confirmationBody = templates.removalConfirmationBody({
                    filename: retrieveAttachmentFilename(removeLink)
                });
            }
            confirmDialog(url, confirmationTitle, confirmationBody);
        };

        function confirmBatchRemovalDialog(
            attachmentIds,
            confirmationTitle,
            confirmationBody,
            numberOfTextLines
        ) {
            var confirmDialog;
            if (DarkFeatures.isEnabled(BULK_DELETE_ATTACHMENT_FEATURE_FLAG_KEY)) {
                confirmDialog = createDynamicHeightDialog(
                    'attachment-removal-confirm-dialog',
                    numberOfTextLines
                );
            } else {
                confirmDialog = createDialog('attachment-removal-confirm-dialog');
            }
            confirmDialog.addHeader(confirmationTitle);
            confirmDialog.addPanel('', confirmationBody);
            if (attachmentIds.length > 0) {
                confirmDialog.addSubmit(AJS.I18n.getText('remove.name'), function() {
                    deleteAttachmentsInBatch(attachmentIds, confirmDialog);
                });
            }
            confirmDialog.addCancel(AJS.I18n.getText('cancel.name'), function() {
                confirmDialog.remove();
            });

            confirmDialog.show();
        }

        $('.removeAttachmentLink').click(function() {
            AttachmentsComponent.showRemoveAttachmentConfirmDialog(this);
            return false;
        });

        $('.removeAttachmentLinkVersion').click(function() {
            confirmDialog(
                createUrl('/json/removeattachmentversion.action', this.search),
                templates.versionRemovalConfirmationTitle(),
                templates.versionRemovalConfirmationBody({
                    filename: retrieveAttachmentFilename(this),
                    version: retrieveAttachmentVersion(this)
                })
            );
            return false;
        });

        $('#download-all-attachments').click(function() {
            var currentPageId = Meta.get('page-id');
            window.location.href = contextPath() + '/pages/downloadallattachments.action?pageId=' + currentPageId;

            Event.trigger('analyticsEvent', {
                name: 'confluence.attachments.download.all',
                data: { pageID: currentPageId }
            });
            return false;
        });

        $('#download-selected-attachments').click(function() {
            var currentPageId = Meta.get('page-id');
            var attachmentIdsToDownload = retrieveSelectedAttachmentIds();

            window.location.href = contextPath() + '/pages/downloadallattachments.action?pageId=' + currentPageId +
                    '&attachmentIds=' + attachmentIdsToDownload.join(',');

            Event.trigger('analyticsEvent', {
                name: 'confluence.attachments.download.selected',
                data: { count: attachmentIdsToDownload.length }
            });
            return false;
        });

        $('#delete-all-attachments').click(function() {
            confirmBatchRemovalDialog(
                'all',
                templates.removalAllConfirmationTitle(),
                templates.allAttachmentsRemovalConfirmationBody(),
                defaultNumberOfTextLines
            );
            return false;
        });

        $('#delete-selected-attachments').click(function() {
            var metadata = getVisibleAttachmentsMetadata();
            var selectedAttachmentIds = metadata.filter(function(attachment) {
                return attachment.isSelected;
            });

            var deletableSelectedAttachments = metadata.filter(function(attachment) {
                return attachment.isSelected && attachment.isDeletable;
            });

            var undeletableSelectedAttachments = metadata.filter(function(attachment) {
                return attachment.isSelected && !attachment.isDeletable;
            });

            var deletableSelectedAttachmentIds = deletableSelectedAttachments.map(function(
                attachment
            ) {
                return attachment.id;
            });

            var undeletableSelectedAttachmentsFilenames = undeletableSelectedAttachments.map(
                function(attachment) {
                    return attachment.fileName;
                }
            );

            var numberOfTextLines = 0;

            // This is for the copy text that exists in the beginning of the dialog
            numberOfTextLines += 2;
            // This is for the copy text which exists if there are undeletable attachments
            numberOfTextLines += undeletableSelectedAttachments.length > 0 ? 2 : 0;
            numberOfTextLines += undeletableSelectedAttachments.length;

            if (selectedAttachmentIds.length > 0) {
                confirmBatchRemovalDialog(
                    deletableSelectedAttachmentIds,
                    templates.removalSelectedConfirmationTitle({
                        fileCount: deletableSelectedAttachments.length
                    }),
                    templates.selectedAttachmentsRemovalConfirmationBody({
                        undeletable: undeletableSelectedAttachmentsFilenames
                    }),
                    numberOfTextLines
                );
            }
            return false;
        });

        function updateBulkActionsVisibility() {
            var count = retrieveSelectedAttachmentCount();
            if (count > 0) {
                document.getElementById('menu-attachment-bulk-action-left-children').style.display =
                    'flex';
            } else {
                document.getElementById('menu-attachment-bulk-action-left-children').style.display =
                    'none';
            }
        }

        function updateSelectedAttachmentCount() {
            var count = retrieveSelectedAttachmentCount();
            var displayText = AJS.I18n.getText('selected.attachments.count', count);
            $('#bulk-action-selected-count').text(displayText);
        }

        function updateSelectAllCheckbox() {
            var selectedAttachmentCount = retrieveSelectedAttachmentCount();

            var isAllSelected = selectedAttachmentCount === retrieveAllDisplayedAttachmentCount();

            var selectAllCheckbox = $('#select-all-attachments-checkbox');
            var selectAllCheckboxHTMLElement = selectAllCheckbox[0];
            if (selectedAttachmentCount === 0) {
                selectAllCheckboxHTMLElement.checked = false;
                selectAllCheckboxHTMLElement.ariaChecked = false;
                selectAllCheckbox.prop('indeterminate', false);
            } else if (isAllSelected) {
                selectAllCheckboxHTMLElement.checked = true;
                selectAllCheckboxHTMLElement.ariaChecked = true;
                selectAllCheckbox.prop('indeterminate', false);
            } else {
                selectAllCheckboxHTMLElement.checked = false;
                selectAllCheckboxHTMLElement.ariaChecked = 'mixed';
                selectAllCheckbox.prop('indeterminate', true);
            }
        }

        function onCheckboxChange() {
            updateBulkActionsVisibility();
            updateSelectedAttachmentCount();
            updateSelectAllCheckbox();
        }

        $('#select-all-attachments-checkbox').click(function() {
            var selectedAttachmentCount = retrieveSelectedAttachmentCount();
            if (selectedAttachmentCount === 0) {
                $('[id^=selected-attachment]')
                    .toArray()
                    .forEach(function(checkbox) {
                        var checkboxHTMLElement = checkbox;
                        checkboxHTMLElement.checked = true;
                        checkboxHTMLElement.ariaChecked = true;
                    });
            } else {
                $('[id^=selected-attachment]')
                    .toArray()
                    .forEach(function(checkbox) {
                        var checkboxHTMLElement = checkbox;
                        checkboxHTMLElement.checked = false;
                        checkboxHTMLElement.ariaChecked = false;
                    });
            }

            onCheckboxChange();
        });

        if (DarkFeatures.isEnabled(BULK_DELETE_ATTACHMENT_FEATURE_FLAG_KEY)) {
            $('tr[id^=attachment-]').change(onCheckboxChange);
            $('[id^=select-all-attachments-checkbox]').change(onCheckboxChange);

            // Update the page state once the listeners have been added in case some are selected before the listeners
            // are added.
            onCheckboxChange();
        }
    }

    /**
     * @exports confluence/attachments
     */
    return {
        component: AttachmentsComponent,
        initialiser: initialiser,
        submitHandler: submitHandler
    };
});

require('confluence/module-exporter').safeRequire('confluence/attachments', function(Attachments) {
    'use strict';

    // Make this binding available to the attachments macro
    // which dynamically loads the table.
    var AJS = require('ajs');
    AJS.Attachments = Attachments.component;
    AJS.toInit(Attachments.initialiser);
});
