/**
 * @module confluence/user-dialog
 */
define('confluence/user-dialog', [
    'jquery',
    'confluence/dark-features',
    'ajs',
    'confluence/meta'
], function($,
    DarkFeatures,
    AJS,
    Meta) {
    'use strict';

    var defaults = {
        fadeTime: 100,
        hideDelay: 500,
        showDelay: 700,
        loadDelay: 50,
        width: 300,
        mouseMoveThreshold: 10,
        container: 'body'
    };

    if (typeof AJS.followCallbacks === 'undefined') { AJS.followCallbacks = []; }
    if (typeof AJS.favouriteCallbacks === 'undefined') { AJS.favouriteCallbacks = []; }

    // if you want to customize the behaviour of the follow button you can add callbacks to the above array
    // by adding this code to your javascript page:
    //
    // if (typeof AJS.followCallbacks == "undefined") AJS.followCallbacks = [];
    // AJS.followCallbacks.push(function(user) {
    //    alert('favourite added'+user');
    // });
    //
    // these callbacks are called after the post to the server has completed.
    //
    // You can add to the followCallbacks or the favouriteCallbacks if you want callbacks on the follow functions
    // or the favourite functions respectively.

    AJS.toInit(function($) {
        $(self).bind('hover-user.follow', function(e, data) {
            for (var i = 0, ii = AJS.followCallbacks.length; i < ii; i++) {
                AJS.followCallbacks[i](data.username);
            }
        });
    });

    /**
     * Creates a new hover popup
     *
     * @param $items jQuery object - the items that trigger the display of this popup when the user mouses over.
     * @param identifier A unique identifier for this popup. This should be unique across all popups on the page and a
     *                   valid CSS class.
     * @param url The URL to retrieve popup contents.
     * @param postProcess A function called after the popup contents are loaded.
     *                    `this` will be the popup jQuery object, and the first argument is the popup identifier.
     * @param options Custom options to change default behaviour. See confluence/content-hover defaults for default values and valid options.
     *
     * @return jQuery object - the popup that was created
     */
    const ContentHover = function($items, identifier, url, postProcess, options) {
        var opts = $.extend(false, defaults, options);
        var contents;
        var mousePosition;

        var getPopup = function() {
            var popup = $('#content-hover-' + identifier);
            if (!popup.length) {
                // This is the first contentHover call for this identifier - create and set up the content container div.
                $(opts.container).append($('<div id="content-hover-' + identifier + '" class="ajs-content-hover dialog-content-container aui-box-shadow">'
                + '<div class="contents"></div></div>'));
                popup = $('#content-hover-' + identifier);
                contents = popup.find('.contents');
                contents.css('width', opts.width + 'px')
                    .mouseover(function() {
                        var p = getPopup()[0];
                        clearTimeout(p.hideDelayTimer);
                        popup.unbind('mouseover');
                    }).mouseout(function() {
                        hidePopup();
                    });
                $(document).keydown(function(e) {
                    if (popup.length) {
                        if (e.key === 'Escape') {
                            hidePopup(true);
                        }
                    }
                });
                contents.click(function(e) {
                    e.stopPropagation();
                });
            } else {
                // A popup has already been created for this identifier (ie. this contentHover call is from a newly-created
                // element with an existing identifier).
                contents = popup.find('.contents');
            }
            return popup;
        };

        var showPopup = function() {
            var popup = getPopup();
            var p = popup[0];

            if (popup.is(':visible')) {
                return;
            }

            p.showTimer = setTimeout(function() {
                if (!p.contentLoaded || !p.shouldShow) {
                    return;
                }
                p.beingShown = true;
                var $window = $(window);
                var posx = mousePosition.x - 3;
                var posy = mousePosition.y + 15;

                if (posx + opts.width + 30 > $window.width()) {
                    popup.css({
                        right: '20px',
                        left: 'auto'
                    });
                } else {
                    popup.css({
                        left: posx + 'px',
                        right: 'auto'
                    });
                }

                var bottomOfViewablePage = (window.pageYOffset || document.documentElement.scrollTop) + $window.height();
                if ((posy + popup.height()) > bottomOfViewablePage) {
                    posy = bottomOfViewablePage - popup.height() - 5;
                    popup.mouseover(function() {
                        var p = getPopup()[0];
                        clearTimeout(p.hideDelayTimer);
                    }).mouseout(function() {
                        hidePopup();
                    });
                }
                popup.css({
                    top: posy + 'px'
                });

                // reset position of popup box
                popup.fadeIn(opts.fadeTime, function() {
                    // once the animation is complete, set the tracker variables
                    p.beingShown = false;
                });
            }, opts.showDelay);
        };

        var hidePopup = function(noDelay) {
            var popup = getPopup();
            var p = popup[0];

            p.beingShown = false;
            p.shouldShow = false;
            clearTimeout(p.hideDelayTimer);
            clearTimeout(p.showTimer);
            clearTimeout(p.loadTimer);
            p.contentLoading = false;
            p.shouldLoadContent = false;
            // store the timer so that it can be cleared in the mouseover if required
            p.hideDelayTimer = setTimeout(function() {
                popup.fadeOut(opts.fadeTime);
            }, noDelay ? 0 : opts.hideDelay);
        };

        $items.mousemove(function(e) {
            mousePosition = { x: e.pageX, y: e.pageY };

            var popup = getPopup();
            var p = popup[0];

            if (!p.beingShown) {
                clearTimeout(p.showTimer);
            }
            p.shouldShow = true;
            // lazy load popup contents
            if (!p.contentLoaded) {
                if (p.contentLoading) {
                    // If the mouse has moved more than the threshold don't load the contents
                    if (p.shouldLoadContent) {
                        var distance = (mousePosition.x - p.initialMousePosition.x) * (mousePosition.x - p.initialMousePosition.x)
                                + (mousePosition.y - p.initialMousePosition.y) * (mousePosition.y - p.initialMousePosition.y);
                        if (distance > (opts.mouseMoveThreshold * opts.mouseMoveThreshold)) {
                            p.contentLoading = false;
                            p.shouldLoadContent = false;
                            clearTimeout(p.loadTimer);
                            return;
                        }
                    }
                } else {
                    // Save the position the mouse started from
                    p.initialMousePosition = mousePosition;
                    p.shouldLoadContent = true;
                    p.contentLoading = true;
                    p.loadTimer = setTimeout(function() {
                        if (!p.shouldLoadContent) {
                            return;
                        }

                        contents.load(url, function() {
                            p.contentLoaded = true;
                            p.contentLoading = false;
                            postProcess.call(popup, identifier);
                            showPopup();
                        });
                    }, opts.loadDelay);
                }
            }
            // stops the hide event if we move from the trigger to the popup element
            clearTimeout(p.hideDelayTimer);
            // don't trigger the animation again if we're being shown
            if (!p.beingShown) {
                showPopup();
            }
        }).mouseout(function() {
            hidePopup();
        });

        $('body').click(function() {
            var popup = getPopup();
            var p = popup[0];

            p.beingShown = false;
            clearTimeout(p.hideDelayTimer);
            clearTimeout(p.showTimer);
            popup.hide();
        });
    };

    var UserDialog = function($trigger, identifier, url, callback) {
        if ($trigger.attr('data-aui-trigger')) {
            return;
        }
        $trigger.attr('aria-controls', `dialog-${identifier}`);
        $trigger.attr('data-aui-trigger', true);
        let $dialog;
        $trigger.on('click keydown', function(e) {
            if (e.type === 'click' || (e.type === 'keydown' && (e.key === 'Enter' || e.key === ' '))) {
                if (!$dialog) {
                    e.preventDefault();
                    e.stopImmediatePropagation();
                    $dialog = $(`
                        <aui-inline-dialog id="dialog-${identifier}"
                            class="dialog-content-container"
                            aui-focus-trap="true"
                            alignment="bottom left"
                        >
                            <div class="contents"></div>
                        </aui-inline-dialog>`);
                    $dialog.insertAfter($trigger);
                    $dialog
                        .find('.contents')
                        .load(url, function() {
                            callback.call($dialog, identifier);
                            $dialog.attr('open', true);
                        });
                }
            }
        });

        return $dialog;
    };

    // global list of IDs to ensure user-hovers don't get reloaded unnecessarily
    var users = [];
    var contextPath = AJS.contextPath();

    // Do not show user hovers for authenticated unlicensed users.
    if (Meta.get('remote-user') && !Meta.get('remote-user-has-licensed-access')) {
        return $.noop;
    }

    var contentHoverPostProcess = function(id) {
        var username = users[id] || id;
        var data = { username: username, target: this };
        $(self).trigger('hover-user.open', data);
        $('.ajs-menu-bar', this).ajsMenu();
        $('.follow, .unfollow', this).each(function() {
            var $this = $(this).click(function(e) {
                if ($this.hasClass('waiting')) {
                    return;
                }
                var url = $this.hasClass('unfollow') ? '/unfollowuser.action' : '/followuser.action';
                $this.addClass('waiting');
                AJS.safe.post(contextPath + url + '?username=' + encodeURIComponent(username) + '&mode=blank', {}, function() {
                    $this.removeClass('waiting');
                    $this.parent().toggleClass('follow-item').toggleClass('unfollow-item');
                    $(self).trigger('hover-user.follow', data);
                });
                e.stopPropagation();
                return false;
            });
        });
    };

    var selectors = [
        'span.user-hover-trigger',
        'a.confluence-userlink',
        'img.confluence-userlink',
        'a.userLogoLink'
    ].join(', ');

    const triggers = [
        'button[data-user-popup-trigger]',
        'button[data-user-dialog-trigger]'
    ].join(', ');

    var showBusiness = DarkFeatures.isEnabled('show.business.group.in.user.hover');

    /**
     * User Hover Binder component.
     * Expected markup:
     * &lt;a class="confluence-userlink" data-username="test"&gt;Test User&lt;/a&gt;
     * <br>
     * Events Thrown:
     * hover-user.open, hover-user.follow
     *
     * @alias module:confluence/hover-user
     */
    const exports = function() {
        $(triggers).each(function(index, trigger) {
            const $trigger = $(trigger);
            const user = $trigger.attr('data-username');

            if (!user) {
                console.warn('Username not found for trigger:', trigger);
                return;
            }

            let url = `${contextPath}/users/userinfopopup.action?username=${encodeURIComponent(user)}`;
            if (showBusiness) {
                url += '&profileGroups=business';
            }

            try {
                UserDialog(
                    $trigger,
                    user + '-' + index,
                    url,
                    contentHoverPostProcess
                );
            } catch (error) {
                console.error(`Failed to create UserPopup for ${user}:`, error);
            }
        });
        $(selectors).not('[data-disable-user-hover]').filter('[data-user-hover-bound!=\'true\']').each(function() {
            var userlink = $(this);
            var username = userlink.attr('data-username');
            var imageTitle = $('img', userlink).attr('title');

            // Ensure no "popup" title will clash with the user hover.
            userlink.removeAttr('title').attr('data-user-hover-bound', 'true');
            if (imageTitle) {
                $('img', userlink).attr('alt', imageTitle);
            }
            $('img', userlink).removeAttr('title');

            var arrayIndex = $.inArray(username, users);
            if (arrayIndex === -1) {
                users.push(username);
                arrayIndex = $.inArray(username, users);
            }
            userlink.addClass('userlink-' + arrayIndex);
        });

        $.each(users, function(i) {
            var url = contextPath + '/users/userinfopopup.action?username=' + encodeURIComponent(users[i]);
            if (showBusiness) {
                // Show Position, Department and Location
                url += '&profileGroups=business';
            }
            ContentHover($('.userlink-' + i),
                i,
                url,
                contentHoverPostProcess);
        });
    };

    AJS.toInit(exports);

    return exports;
});
