/* Based on AUI 9.13.x */
/* The confluence-aui-dropdown class can be used to easily identify dropdowns created via this decoupled aui dropdown */
/* Changes compared to the original are:
 *     * Removal of unused ajax support
 *     * Removal of the dependency on jquery aop
 *     * Removal of the dependency on jquery throbber
 */

/**
 * @deprecated
 * Please use the modern AUI dropdown components instead.
 * Migration guide: https://aui.atlassian.com/aui/latest/docs/upgrades/dropdown-menu-component.html
 */
define('confluence/dropdown', ['jquery'], function ($) {
    /* eslint complexity:off, eqeqeq:off, max-depth:off, no-unused-vars:off */
    const ZINDEX_DROPDOWN_1 = 2700; // see also dropdown.less

    /**
     * Creates and displays a dropdown menu, typically used for navigation or action menus.
     * 
     * @param {jQuery|String|Array} obj - The object to populate the dropdown from. Can be:
     *   - A jQuery object
     *   - A CSS selector string
     *   - An array of menu items with their properties
     * @param {Object} [usroptions] - Optional dropdown configuration
     * @param {string} [usroptions.alignment='right'] - Alignment of the dropdown ('left' or 'right')
     * @param {Function} [usroptions.escapeHandler] - Function to handle escape key presses
     * @param {string} [usroptions.activeClass='active'] - Class name added to dropdown items when active
     * @param {Function} [usroptions.selectionHandler] - Function to handle when dropdown items are selected
     * @param {Function} [usroptions.hideHandler] - Function to handle when the dropdown is hidden
     * @param {boolean} [usroptions.isHiddenByDefault=false] - Whether to hide the dropdown on initialization
     * @param {Function} [usroptions.displayHandler] - Function to display text in the dropdown
     * @param {boolean} [usroptions.useDisabled=false] - Whether to respect disabled state of parent
     * @param {Function} [usroptions.moveHandler] - Function to handle movement between items
     * 
     * @returns {Array} An array of jQuery objects, referring to the dropdown container elements
     * @throws {Error} If called with an illegal parameter type
     */
    function dropDown(obj, usroptions) {
        var dd = null;
        var result = [];
        var moving = false;
        var $doc = $(document);
        var options = {
            item: 'li:has(a)',
            activeClass: 'active',
            alignment: 'right',
            displayHandler: function (obj) {
                return obj.name;
            },
            escapeHandler: function () {
                this.hide('escape');
                return false;
            },
            hideHandler: function () {},
            moveHandler: function () {},
            useDisabled: false,
        };

        $.extend(options, usroptions);
        options.alignment = { left: 'left', right: 'right' }[options.alignment.toLowerCase()] || 'left';

        if (obj && obj.jquery) {
            // if $
            dd = obj;
        } else if (typeof obj === 'string') {
            // if $ selector
            dd = $(obj);
        } else if (obj && obj.constructor === Array) {
            // if JSON
            dd = $('<div></div>')
                .addClass('aui-dropdown confluence-aui-dropdown')
                .toggleClass('hidden', !!options.isHiddenByDefault);
            for (var i = 0, ii = obj.length; i < ii; i++) {
                var ol = $('<ol></ol>');
                for (var j = 0, jj = obj[i].length; j < jj; j++) {
                    var li = $('<li></li>');
                    var properties = obj[i][j];
                    if (properties.href) {
                        li.append(
                            $('<a></a>')
                                .html('<span>' + options.displayHandler(properties) + '</span>')
                                .attr({ href: properties.href })
                                .addClass(properties.className)
                        );

                        // deprecated - use the properties on the li, not the span
                        $.data($('a > span', li)[0], 'properties', properties);
                    } else {
                        li.html(properties.html).addClass(properties.className);
                    }
                    if (properties.icon) {
                        li.prepend($('<img />').attr('src', properties.icon));
                    }
                    if (properties.insideSpanIcon) {
                        li.children('a').prepend($('<span></span>').attr('class', 'icon'));
                    }
                    if (properties.iconFontClass) {
                        li.children('a').prepend(
                            $('<span></span>').addClass(
                                'aui-icon aui-icon-small aui-iconfont-' + properties.iconFontClass
                            )
                        );
                    }

                    $.data(li[0], 'properties', properties);
                    ol.append(li);
                }
                if (i === ii - 1) {
                    ol.addClass('last');
                }
                dd.append(ol);
            }
            $('body').append(dd);
        } else {
            throw new Error(
                'dropDown function was called with illegal parameter. Should be $ object, $ selector or array.'
            );
        }

        var moveDown = function () {
            move(+1);
        };

        var moveUp = function () {
            move(-1);
        };

        var move = function (dir) {
            var trigger = !moving;
            var cdd = dropDown.current.$[0];
            var links = dropDown.current.links;
            var oldFocus = cdd.focused;

            moving = true;

            if (links.length === 0) {
                // Nothing to move focus to. Abort.
                return;
            }

            cdd.focused = typeof oldFocus === 'number' ? oldFocus : -1;

            if (!dropDown.current) {
                console.log('move - not current, aborting');
                return true;
            }

            cdd.focused += dir;

            // Resolve out of bounds values:
            if (cdd.focused < 0) {
                cdd.focused = links.length - 1;
            } else if (cdd.focused >= links.length) {
                cdd.focused = 0;
            }

            options.moveHandler($(links[cdd.focused]), dir < 0 ? 'up' : 'down');
            if (trigger && links.length) {
                $(links[cdd.focused]).addClass(options.activeClass);
                moving = false;
            } else if (!links.length) {
                moving = false;
            }
        };

        var moveFocus = function (e) {
            if (!dropDown.current) {
                return true;
            }
            var c = e.which;
            var cdd = dropDown.current.$[0];
            var links = dropDown.current.links;

            dropDown.current.cleanActive();
            switch (c) {
                case 40: {
                    moveDown();
                    break;
                }
                case 38: {
                    moveUp();
                    break;
                }
                case 27: {
                    return options.escapeHandler.call(dropDown.current, e);
                }
                case 13: {
                    if (cdd.focused >= 0) {
                        if (!options.selectionHandler) {
                            if ($(links[cdd.focused]).prop('nodeName') != 'a') {
                                return $('a', links[cdd.focused]).trigger('focus'); //focus on the "a" within the parent item elements
                            } else {
                                return $(links[cdd.focused]).trigger('focus'); //focus on the "a"
                            }
                        } else {
                            return options.selectionHandler.call(dropDown.current, e, $(links[cdd.focused])); //call the selection handler
                        }
                    }
                    return true;
                }
                default: {
                    if (links.length) {
                        $(links[cdd.focused]).addClass(options.activeClass);
                    }
                    return true;
                }
            }

            e.stopPropagation();
            e.preventDefault();
            return false;
        };

        var hider = function (e) {
            if (!((e && e.which && e.which == 3) || (e && e.button && e.button == 2) || false)) {
                // right click check
                if (dropDown.current) {
                    dropDown.current.hide('click');
                }
            }
        };
        var active = function (i) {
            return function () {
                if (!dropDown.current) {
                    return;
                }
                dropDown.current.cleanFocus();
                this.originalClass = this.className;
                $(this).addClass(options.activeClass);
                dropDown.current.$[0].focused = i;
            };
        };

        var handleClickSelection = function (e) {
            if (e.button || e.metaKey || e.ctrlKey || e.shiftKey) {
                return true;
            }
            if (dropDown.current && options.selectionHandler) {
                options.selectionHandler.call(dropDown.current, e, $(this));
            }
        };

        var isEventsBound = function (el) {
            var bound = false;
            if (el.data('events')) {
                $.each(el.data('events'), function (i, handler) {
                    $.each(handler, function (type, handler) {
                        if (handleClickSelection === handler) {
                            bound = true;
                            return false;
                        }
                    });
                });
            }
            return bound;
        };

        dd.each(function () {
            var cdd = this;
            var $cdd = $(this);
            var res = {};
            var methods = {
                reset: function () {
                    res = $.extend(res, {
                        $: $cdd,
                        links: $(options.item || 'li:has(a)', cdd),
                        cleanActive: function () {
                            if (cdd.focused + 1 && res.links.length) {
                                $(res.links[cdd.focused]).removeClass(options.activeClass);
                            }
                        },
                        cleanFocus: function () {
                            res.cleanActive();
                            cdd.focused = -1;
                        },
                        moveDown: moveDown,
                        moveUp: moveUp,
                        moveFocus: moveFocus,
                        getFocusIndex: function () {
                            return typeof cdd.focused === 'number' ? cdd.focused : -1;
                        },
                    });
                    res.links.each(function (i) {
                        var $this = $(this);
                        if (!isEventsBound($this)) {
                            $this.hover(active(i), res.cleanFocus);
                            $this.click(handleClickSelection);
                        }
                    });
                },
                appear: function (dir) {
                    if (dir) {
                        $cdd.removeClass('hidden');
                        //handle left or right alignment
                        $cdd.addClass('aui-dropdown-' + options.alignment);
                    } else {
                        $cdd.addClass('hidden');
                    }
                },
                fade: function (dir) {
                    if (dir) {
                        $cdd.fadeIn('fast');
                    } else {
                        $cdd.fadeOut('fast');
                    }
                },
                scroll: function (dir) {
                    if (dir) {
                        $cdd.slideDown('fast');
                    } else {
                        $cdd.slideUp('fast');
                    }
                },
            };

            res.reset = methods.reset;
            res.reset();

            res.addControlProcess = function (method, process) {
                throw new Error(
                    'The addControlProcess method is intentionally removed and not supported in this dropdown.'
                );
            };

            /**
             * Inserts a callback function to be executed <em>after</em> the
             * specified method has returned. This is a direct method replacement approach
             * that avoids the dependency on Aspect Oriented Programming (AOP).
             * 
             * @method addCallback
             * @param {string} method - Name of the public method to add a callback to
             * @param {Function} callback - Function to be executed after the method returns
             * @returns {Array} weaved aspect
             * @throws {Error} If the specified method doesn't exist
             */
            res.addCallback = function (method, callback) {
                const originalMethod = this[method];
                if (typeof originalMethod !== 'function') {
                    throw new Error(`Method ${method} does not exist`);
                }

                this[method] = function (...args) {
                    const result = originalMethod.apply(this, args);
                    callback.apply(this, args);
                    return result;
                };
            };

            res.show = function (method) {
                if (options.useDisabled && this.$.closest('.aui-dd-parent').hasClass('disabled')) {
                    return;
                }

                this.alignment = options.alignment;
                hider();
                dropDown.current = this;
                this.method = method || this.method || 'appear';

                this.timer = setTimeout(function () {
                    $doc.click(hider);
                }, 0);

                $doc.keydown(moveFocus);

                if (options.firstSelected && this.links[0]) {
                    active(0).call(this.links[0]);
                }

                $(cdd.offsetParent).css({ zIndex: ZINDEX_DROPDOWN_1 });
                methods[this.method](true);

                $(document).trigger('showLayer', ['dropdown', dropDown.current]);
            };

            res.hide = function (causer) {
                this.method = this.method || 'appear';
                $($cdd.get(0).offsetParent).css({ zIndex: '' });
                this.cleanFocus();
                methods[this.method](false);
                $doc.unbind('click', hider).unbind('keydown', moveFocus);
                $(document).trigger('hideLayer', ['dropdown', dropDown.current]);
                dropDown.current = null;
                return causer;
            };

            res.addCallback('reset', function () {
                if (options.firstSelected && this.links[0]) {
                    active(0).call(this.links[0]);
                }
            });

            if (!dropDown.iframes) {
                dropDown.iframes = [];
            }

            dropDown.createShims = (function createShims() {
                $('iframe').each(function (idx) {
                    var iframe = this;
                    if (!iframe.shim) {
                        iframe.shim = $('<div />')
                            .addClass('shim hidden')
                            .css({ position: 'absolute' })
                            .appendTo('body');
                        dropDown.iframes.push(iframe);
                    }
                });
                return createShims;
            })();

            res.addCallback('show', function () {
                $(dropDown.iframes).each(function () {
                    var $this = $(this);

                    if ($this.is(':visible')) {
                        var offset = $this.offset();
                        offset.height = $this.height();
                        offset.width = $this.width();
                        this.shim
                            .css({
                                left: offset.left + 'px',
                                top: offset.top + 'px',
                                height: offset.height + 'px',
                                width: offset.width + 'px',
                            })
                            .removeClass('hidden');
                    }
                });
            });

            res.addCallback('hide', function () {
                $(dropDown.iframes).each(function () {
                    this.shim.addClass('hidden');
                });
                options.hideHandler();
            });
            result.push(res);
        });
        return result;
    }

    /**
     * Retrieves the value of a named property from a dropdown item.
     * 
     * @method getAdditionalPropertyValue
     * @namespace dropDown
     * @param {jQuery} item - jQuery object of the dropdown item (LI element)
     * @param {string} name - Name of the property to retrieve
     * @returns {*} The value of the property, or null if the property doesn't exist
     * @throws {Error} Logs a warning if the item is not an LI element
     */
    dropDown.getAdditionalPropertyValue = function (item, name) {
        var el = item[0];
        if (!el || typeof el.tagName !== 'string' || el.tagName.toLowerCase() !== 'li') {
            // we are moving the location of the properties and want to deprecate the attachment to the span
            // but are unsure where and how its being called so for now we just log
            console.log(
                'dropDown.getAdditionalPropertyValue : item passed in should be an LI element wrapped by jQuery'
            );
        }
        var properties = $.data(el, 'properties');
        return properties ? properties[name] : null;
    };

    /**
     * Creates a standard dropdown control that identifies triggers which display dropdowns when clicked.
     * 
     * @class Standard
     * @constructor
     * @namespace dropDown
     * @param {Object} [usroptions] - Configuration options for the dropdown
     * @param {string} [usroptions.selector='.aui-dd-parent'] - Selector for dropdown parent elements
     * @param {string} [usroptions.dropDown='.aui-dropdown'] - Selector for dropdown elements
     * @param {string} [usroptions.trigger='.aui-dd-trigger'] - Selector for dropdown trigger elements
     * @param {boolean} [usroptions.useLiveEvents=false] - Whether to use live event binding
     * @param {boolean} [usroptions.isHiddenByDefault=true] - Whether dropdowns are hidden by default
     * @returns {Array} Array of dropdown control objects
     */
    dropDown.Standard = function (usroptions) {
        var res = [];
        var options = {
            selector: '.aui-dd-parent',
            dropDown: '.aui-dropdown',
            trigger: '.aui-dd-trigger',
        };
        var dropdownParents;

        // extend defaults with user options
        $.extend(options, usroptions);

        var hookUpDropDown = function ($trigger, $parent, $dropdown, ddcontrol) {
            // extend to control to have any additional properties/methods
            $.extend(ddcontrol, { trigger: $trigger });

            // flag it to prevent additional dd controls being applied
            $parent.addClass('dd-allocated');

            //hide dropdown if not already hidden
            $dropdown.addClass('hidden');

            //show the dropdown if isHiddenByDefault is set to false
            if (options.isHiddenByDefault == false) {
                ddcontrol.show();
            }

            ddcontrol.addCallback('show', function () {
                $parent.addClass('active');
            });

            ddcontrol.addCallback('hide', function () {
                $parent.removeClass('active');
            });
        };

        var handleEvent = function (event, $trigger, $dropdown, ddcontrol) {
            if (ddcontrol != dropDown.current) {
                $dropdown.css({ top: $trigger.outerHeight() });
                ddcontrol.show();
                event.stopImmediatePropagation();
            }
            event.preventDefault();
        };

        if (options.useLiveEvents) {
            // cache arrays so that we don't have to recalculate the dropdowns. Since we can't store objects as keys in a map,
            // we have two arrays: keysCache stores keys of dropdown triggers; valuesCache stores a map of internally used objects
            var keysCache = [];
            var valuesCache = [];

            $(options.trigger).live('click', function (event) {
                var $trigger = $(this);
                var $parent;
                var $dropdown;
                var ddcontrol;

                // if we're cached, don't recalculate the dropdown and do all that funny shite.
                var index;
                if ((index = $.inArray(this, keysCache)) >= 0) {
                    var val = valuesCache[index];
                    $parent = val.parent;
                    $dropdown = val.dropdown;
                    ddcontrol = val.ddcontrol;
                } else {
                    $parent = $trigger.closest(options.selector);
                    $dropdown = $parent.find(options.dropDown);
                    // Sanity checking
                    if ($dropdown.length === 0) {
                        return;
                    }

                    ddcontrol = dropDown($dropdown, options)[0];
                    // Sanity checking
                    if (!ddcontrol) {
                        return;
                    }

                    // cache
                    keysCache.push(this);
                    val = {
                        parent: $parent,
                        dropdown: $dropdown,
                        ddcontrol: ddcontrol,
                    };

                    hookUpDropDown($trigger, $parent, $dropdown, ddcontrol);

                    valuesCache.push(val);
                }

                handleEvent(event, $trigger, $dropdown, ddcontrol);
            });
        } else {
            // handling for jQuery collections
            if (this instanceof $) {
                dropdownParents = this;
                // handling for selectors
            } else {
                dropdownParents = $(options.selector);
            }

            // a series of checks to ensure we are dealing with valid dropdowns
            dropdownParents = dropdownParents
                .not('.dd-allocated')
                .filter(':has(' + options.dropDown + ')')
                .filter(':has(' + options.trigger + ')');

            dropdownParents.each(function () {
                var $parent = $(this);
                var $dropdown = $(options.dropDown, this);
                var $trigger = $(options.trigger, this);
                var ddcontrol = dropDown($dropdown, options)[0];

                // extend to control to have any additional properties/methods
                $.extend(ddcontrol, { trigger: $trigger });

                hookUpDropDown($trigger, $parent, $dropdown, ddcontrol);

                $trigger.on('click', function (e) {
                    handleEvent(e, $trigger, $dropdown, ddcontrol);
                });

                // add control to the response
                res.push(ddcontrol);
            });
        }
        return res;
    };

    /**
     * Placeholder for the deprecated Ajax method.
     * This method has been intentionally removed and is not supported in this dropdown implementation.
     * 
     * @method Ajax
     * @namespace dropDown
     * @param {*} _ - Any parameter (ignored)
     * @throws {Error} Always throws an error indicating the method is not supported
     * @deprecated This method has been intentionally removed
     */
    dropDown.Ajax = function (_) {
        throw new Error('The Ajax method is intentionally removed and not supported in this dropdown.');
    };

    return dropDown;
});
